/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.LabelInterface;
import org.opengion.fukurou.util.XHTMLTag;

import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;

/**
 * サブミットボタンを表示するHTML拡張タグです(forward.jsp の commonForward タグと同時に使用します)。
 *
 * @og.formSample
 * ●形式：&lt;og:submit value="…" msg="…" /&gt;
 * ●body：なし
 *
 * ●使用例
 *     &lt;og:writeCheck&gt;
 *         &lt;og:submit value="copy"     msg="MSG0035" accesskey="C"  /&gt;
 *         &lt;og:submit value="modify"   msg="MSG0036" accesskey="M"  /&gt;
 *         &lt;og:submit value="delete"   msg="MSG0037" accesskey="D"  /&gt;
 *         &lt;og:submit value="dbset"    msg="XXMSG01" accesskey="X" command="NEW" gamenId="../TEST01" /&gt;
 *         &lt;jsp:directive.include file="/jsp/common/Excel_direct.jsp" /&gt;&lt;br /&gt;
 *     &lt;/og:writeCheck&gt;
 *
 * @og.rev 3.1.1.0 (2003/03/28) 新規作成
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SubmitTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String TYPE = "submit" ;
	private static final String NAME = "command" ;
	private static final String MARGIN = "<span style=\"width: 3px;\" ></span>" ;

	private String  command		= null;
	private String  value		= null;
	private String  gamenId		= null;
	private String  target		= null;		// 3.5.5.2 (2004/04/02)
	private String  dbkeys		= null;		// 4.0.0 (2007/05/23)

	private String[] keys		= null;		// 3.5.5.5 (2004/04/23)
	private String[] vals		= null;		// 3.5.5.5 (2004/04/23)

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.5.2 (2004/04/02) target 属性の追加
	 * @og.rev 3.5.5.5 (2004/04/23) keys,vals 属性の追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= null;
		value		= null;
		gamenId		= null;
		target		= null;		// 3.5.5.2 (2004/04/02)
		keys		= null;		// 3.5.5.5 (2004/04/23)
		vals		= null;		// 3.5.5.5 (2004/04/23)
		dbkeys		= null;		// 4.0.0 (2007/05/23)
	}

	/**
	 * サブミットボタンを作成します。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
	 * @og.rev 3.5.5.2 (2004/04/02) target 属性の追加
	 * @og.rev 3.5.5.9 (2004/06/07) target 属性を、set ではなく add で追加。
	 * @og.rev 3.5.5.9 (2004/06/07) target 属性を、set ではなく add で追加。
	 * @og.rev 4.0.0 (2005/11/30) title 属性が未設定時の処理追加
	 *
	 * @return  サブミットボタンタグ
	 */
	protected String makeTag() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		if( value == null ) {
			String errMsg = "value に null がセットされています。";
			throw new HybsSystemException( errMsg );
		}

		String msg = nval( getMsglbl(), value.toUpperCase( Locale.JAPAN ) );
		String accesskey = get( "accesskey" );
		if( accesskey != null ) {
			msg = msg + "(" + accesskey + ")" ;
		}

		// キャッシュエントリ
		command = nval( command,value.toUpperCase(Locale.JAPAN) );
		String valueLink = getValueLink( gamenId , value );
		if( valueLink == null ) { return ""; }	// 	アクセス不可時は null

		// 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
		rtn.append( getHiddenTag( command, msg, valueLink ) );
		rtn.append( HybsSystem.CR );

		set( "type",TYPE );
		set( "name",NAME );
		set( "value",msg );

		// 3.5.5.2 (2004/04/02) target 属性の追加
		if( target != null ) {
			add( "onClick","this.form.target='" + target + "'",";" );	// 3.5.5.9 (2004/06/07)
		}

		// 4.0.0 (2005/11/30) title 属性が未設定時の処理追加
		if( get( "title" ) == null ) {
			LabelInterface msglbl = getLabelInterface();
			if( msglbl != null ) {
				String desc = msglbl.getDescription();
				if( desc != null && desc.length() > 0 ) {
					set( "title",desc );
				}
			}
		}

		rtn.append( XHTMLTag.input( getAttributes() ) );
		rtn.append( MARGIN );
		rtn.append( HybsSystem.CR );

		return rtn.toString() ;
	}

	/**
	 * 画面ＩＤとvalue から、指定のＵＲＬを作成します。
	 * 画面へのアクセス許可が与えられていない場合は、null を返します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) URLを求めるのに、GUIInfo#getRealAddress() を使用する。
	 * @og.rev 4.0.0 (2005/01/31) GUIInfoの実アドレスのパラメータを考慮する。
	 *
	 * @param   gamenId String
	 * @param   value String
	 * @return  link String アクセス不可時は null
	 */
	private String getValueLink( final String gamenId,final String value ) {
		String link = value + ".jsp";

		if( gamenId != null && gamenId.length() > 0 ) {
			GUIInfo guiInfo = getGUIInfo( gamenId );		// 4.0.0 (2005/01/31)
			if( guiInfo == null ) { return null; }	// 見つからない場合は、アクセス不可

			String address = guiInfo.getRealAddress( link );
			link = getRequestParameter( address );
		}

		return link ;
	}

	/**
	 * 【TAG】forward したいＪＳＰファイル名を記述します(例:insert,copy,modify,delete など)。
	 *
	 * @og.tag
	 * ＪＳＰファイル名は、標準で、insert,copy,modify,delete などと指定します。
	 * 実際には、各JSP画面（insert.jsp,copy.jsp,modify.jsp,delete.jsp )に
	 * リクエストが転送されます。
	 * このＪＳＰファイル名は、同一画面ＩＤ内のフォルダに属している必要があります。
	 * ここのＩＤは、ＪＳＰ画面そのものですので、大文字小文字は区別されます。
	 *
	 * @param   val String
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
		if( value == null ) {
			String errMsg = "value に null がセットされています。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】gamenId 属性を登録します。
	 *
	 * @og.tag
	 * gamenId 属性は、別の画面にforward する場合に使用します。
	 * 実際は、forward ではなく、sendRedirect されます。
	 *
	 * @param   id gamenId 属性
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 隠し属性タグを取得します。
	 * 各ボタンに設定された値を、隠しフィールドに設定して受け渡しします。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
	 * @og.rev 3.5.5.5 (2004/04/23) 余計なボタン関連情報を転送しない為に、キーを変更します。
	 * @og.rev 3.5.5.5 (2004/04/23) keys,vals 属性で指定した値を、出力します。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.8.0.8 (2005/10/03) gamenId が指定されている場合は、BACK_GAMENID を出力する。
	 *
	 * @return  String
	 */
	private String getHiddenTag( final String command, final String msg, final String valueLink ) {

		StringBuilder strRet = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		String prefix = HybsSystem.NO_XFER_KEY + msg ;
		strRet.append( XHTMLTag.hidden( prefix        ,valueLink ) );	// 3.5.5.5 (2004/04/23)
		strRet.append( XHTMLTag.hidden( prefix + "CMD",command   ) );	// 3.5.5.5 (2004/04/23)

		// 3.5.5.5 (2004/04/23) keys,vals 属性で指定した値を、出力します。
		if( keys != null ) {
			if( keys.length != vals.length ) {
				String errMsg = "SubmitTag の keys と vals の引数の個数が異なります。" + HybsSystem.CR
							+ "keys=[" + StringUtil.array2csv( keys ) + "], vals=["
							+ StringUtil.array2csv( vals ) + "]" ;
				throw new HybsSystemException( errMsg );
			}
			for( int i=0; i<keys.length; i++ ) {
				strRet.append( XHTMLTag.hidden( prefix + "KEY_" + keys[i],vals[i] ) );	// 3.5.5.5 (2004/04/23)
			}
		}

		// 4.0.0 (2007/05/23) dbkeys が指定されている場合
		if( dbkeys != null && dbkeys.length() > 0 ) {
			strRet.append( XHTMLTag.hidden( prefix + "KEY_dbkeys",dbkeys ) );	// 4.0.0 (2007/05/23)
		}

		// 3.8.0.8 (2005/10/03) gamenId が指定されている場合は、BACK_GAMENID を出力する。
		if( gamenId != null && gamenId.length() > 0 ) {
			String backGamenId = getGUIInfoAttri( "KEY" );
			strRet.append( XHTMLTag.hidden( prefix + "KEY_BACK_GAMENID",backGamenId ) );
		}

		return strRet.toString();
	}

	/**
	 * 【TAG】処理コマンドを登録します(初期値:大文字の value 属性値[INSERT,COPY,MODIFY,DELETE など])。
	 *
	 * @og.tag
	 * command 属性を指定しない場合は、このvalue 属性値が、コマンドになります。
	 * value 属性に、insert,copy,modify,delete などと指定されていた場合は、
	 * それぞれ、INSERT,COPY,MODIFY,DELETE というコマンドになります。
	 * コマンドは、大文字です。
	 *
	 * @param	cmd コマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
		if( command != null ) { command = command.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】サブミット先の文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @og.tag サブミット先のフレーム名（ターゲット属性）を設定します。
	 *
	 * @og.rev 3.5.5.2 (2004/04/02) 新規追加
	 *
	 * @param	flag サブミット先の文書のフレーム名（ターゲット属性）
	 */
	public void setTarget( final String flag ) {
		target = nval( getRequestParameter( flag ),target );
	}

	/**
	 * 【TAG】ボタン専用のリクエストキーをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * このサブミットボタンが押された場合のみ、転送されるリクエスト情報の
	 * キーを設定できます。カンマ区切りで複数指定できます。
	 * vals 属性には、キーに対応する値を、設定してください。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	key ボタンが押された時に転送するキー
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】ボタン専用のリクエスト値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * キーに対応した値を、カンマ区切り文字で複数指定出来ます。
	 * 指定順序は、キーと同じにしておいて下さい。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	val keys属性に対応する値
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】commonForward の dbkeys にカラム指定を行います。
	 *
	 * @og.tag
	 * カラム指定は、CSV形式(カンマ区切り)で指定してください。
	 * なお、引数は、自動的に受けるのではなく、commonForward タグに、
	 * dbkeys="{&#064;dbkeys}" の記述は必要です。
	 *
	 * @og.rev 4.0.0 (2007/05/23) 新規追加
	 *
	 * @param	keys String dbkeys のにカラム指定
	 */
	public void setDbkeys( final String keys ) {
		dbkeys = nval( getRequestParameter( keys ),dbkeys );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "command"		,command	)
				.println( "value"		,value		)
				.println( "gamenId"		,gamenId	)
				.println( "target"		,target		)
				.println( "keys"		,keys		)
				.println( "vals"		,vals		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
