/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelUtil;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * XMLパーサによる、OpenOffice.org Calcの表計算ドキュメントファイルを読み取る実装クラスです。
 * 
 * ①カラム名が指定されている場合
 *  #NAMEで始まる行を検索し、その行のそれぞれの値をカラム名として処理します。
 *  #NAMEで始まる行より以前の行については、全て無視されます。
 *  また、#NAMEより前のカラム及び、#NAMEの行の値がNULL(カラム名が設定されていない)カラムも
 *  無視します。
 *  読み飛ばされたカラム列に入力された値は取り込まれません。
 *  また、#NAME行以降の#で始まる行は、コメント行とみなされ処理されません。
 * 
 * ②カラム名が指定されている場合
 *  指定されたカラム名に基づき、値を取り込みます。
 *  カラム名の順番と、シートに記述されている値の順番は一致している必要があります。
 *  指定されたカラム数を超える列の値については全て無視されます。
 *  #で始まる行は、コメント行とみなされ処理されません。
 *  
 * また、いずれの場合も全くデータが存在していない行は読み飛ばされます。
 * 
 * @og.group ファイル入力
 * 
 * @version 4.0
 * @author Hiroki Nakamura
 * @since JDK5.0,
 */
public class TableReader_Calc extends TableReader_Default {
	// * このプログラムのVERSION文字列を設定します。 {@value} */
	private static final String VERSION = "4.0.0 (2009/01/05)";

	private String		sheetName		= null;
	private String		filename		= null;
	private int			numberOfRows	= 0;
	private int			firstClmIdx		= 0;
	private int[]		valueClmIdx		= null;

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、項目名が必要です。 
	 * (但し、カラム名を指定することで、項目名を省略することができます)
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、Calc 読み込み時に使用します。
	 * 
	 * @og.rev 4.3.5.0 (2009/02/01) toArray するときに、サイズの初期値指定を追加
	 * 
	 * @see #isExcel()
	 */
	public void readDBTable() {

		ZipFile zipFile = null;
		boolean errFlag = false; 	// 5.0.0.1 (2009/08/15) finally ブロックの throw を避ける。
		try {
			// OpenOffice.org odsファイルを開く
			zipFile = new ZipFile( filename );

			ZipEntry entry = zipFile.getEntry( "content.xml" );
			if ( null == entry ) {
				String errMsg = "ODSファイル中にファイルcontent.xmlが存在しません。";
				throw new HybsSystemException( errMsg );
			}

			// content.xmlをパースし、行、列単位のオブジェクトに分解します。
			DomOdsParser odsParser = new DomOdsParser();
			odsParser.doParse( zipFile.getInputStream( entry ), sheetName );
			List<RowInfo> rowInfoList = odsParser.getRowInfoList();

	 		// 4.3.5.0 (2009/02/01) toArray するときに、サイズの初期値指定を追加
//			makeDBTableModel( rowInfoList.toArray( new RowInfo[0] ) );
			makeDBTableModel( rowInfoList.toArray( new RowInfo[rowInfoList.size()] ) );
		}
		catch ( IOException ex ) {
			String errMsg = "ファイル読込みエラー[" + filename + "]";
			throw new HybsSystemException( errMsg, ex );
		}
		finally {
			if ( null != zipFile ) {
				try {
					zipFile.close();
				}
				catch ( IOException ex ) {
					errFlag = true;
				}
			}
		}

		if( errFlag ) {
			String errMsg = "ODSファイルのクローズ中にエラーが発生しました[" + filename + "]";
			throw new HybsSystemException ( errMsg );
		}

//		finally {
//			if ( null != zipFile )
//				try {
//					zipFile.close();
//				}
//				catch ( IOException ex ) {
//					String errMsg = "ODSファイルのクローズ中にエラーが発生しました[" + filename + "]";
//					throw new HybsSystemException ( errMsg, ex );
//				}
//		}
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * このメソッドは、この実装クラスでは使用できません。
	 * 
	 * @param reader BufferedReader (使用していません)
	 */
	public void readDBTable( final BufferedReader reader ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、第一シートです。
	 *
	 * @param sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		this.sheetName = sheetName;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename) これは、OpenOffice.org
	 * Calc追加機能として実装されています。
	 * 
	 * @param filename 読み取り元ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = filename;
		if ( filename == null ) {
			String errMsg = "ファイル名が指定されていません。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 * 
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の 関係があり、問い合わせによる条件分岐で対応します。
	 * 
	 * @return boolean
	 */
	public boolean isExcel() {
		return true;
	}

	/**
	 * ODSファイルをパースした結果からDBTableModelを生成します。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCountの追加
	 * 
	 * @param rowInfoList 行オブジェクトの配列
	 */
	private void makeDBTableModel( final RowInfo[] rowInfoList ) {
		// カラム名が指定されている場合は、優先する。
		if( columns != null && columns.length() > 0 ) {
			makeHeaderFromClms();
		}

//		for( RowInfo rowInfo : rowInfoList ) {
		int skip = getSkipRowCount();						// 5.1.6.0 (2010/05/01)
		for( int row=skip; row<rowInfoList.length; row++ ) {
			RowInfo rowInfo = rowInfoList[row];				// 5.1.6.0 (2010/05/01)
			if( valueClmIdx == null ) {
				makeHeader( rowInfo );
			}
			else {
				makeBody( rowInfo );
			}
		}

		// 最後まで、#NAME が見つから無かった場合
		if ( valueClmIdx == null ) {
			String errMsg = "最後まで、#NAME が見つかりませんでした。" + HybsSystem.CR + "ファイルが空か、もしくは損傷している可能性があります。" + HybsSystem.CR;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 指定されたカラム一覧からヘッダー情報を生成します。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber の追加
	 */
	private void makeHeaderFromClms() {
		table = DBTableModelUtil.newDBTable();
		String[] names = StringUtil.csv2Array( columns );
		table.init( names.length );
		setTableDBColumn( names ) ;
		valueClmIdx = new int[names.length];
		int adrs = (isUseNumber()) ? 1:0 ;	// useNumber =true の場合は、１件目(No)は読み飛ばす。
		for( int i=0; i<names.length; i++ ) {
//			valueClmIdx[i] = i;
			valueClmIdx[i] = adrs++;
		}
	}
	
	/**
	 * ヘッダー情報を読み取り、DBTableModelのオブジェクトを新規に作成します。
	 * ※ 他のTableReaderと異なり、#NAME が見つかるまで、読み飛ばす。
	 * 
	 * @og.rev 4.3.5.0 (2009/02/01) toArray するときに、サイズの初期値指定を追加
	 * 
	 * @param rowInfo 行オブジェクト
	 */
	private void makeHeader( final RowInfo rowInfo ) {
//		int rowRepeat = rowInfo.rowRepeat;
		CellInfo[] cellInfos = rowInfo.cellInfos;

		int cellLen = cellInfos.length;
		int runPos = 0;
		ArrayList<String> nameList = null;
		ArrayList<Integer> posList = null;
		for ( int idx = 0; idx < cellLen; idx++ ) {
			// テーブルのヘッダ(#NAME)が見つかる前の行、列は全て無視される
			CellInfo cellInfo = cellInfos[idx];
			String text = cellInfo.text.trim();

			for ( int cellRep = 0; cellRep < cellInfo.colRepeat; cellRep++ ) {
				// 空白のヘッダは無視(その列にデータが入っていても読まない)
				if ( text.length() != 0 ) {
					if ( firstClmIdx == 0 && text.equalsIgnoreCase( "#NAME" ) ) {
						nameList = new ArrayList<String>();
						posList = new ArrayList<Integer>();
						table = DBTableModelUtil.newDBTable();
						firstClmIdx = idx;
					}
					else if ( nameList != null ) {
						nameList.add( text );
						posList.add( runPos );
					}
				}
				runPos++;
			}
		}

//		if ( posList != null && posList.size() > 0 ) {
		if ( posList != null && ! posList.isEmpty() ) {
			table = DBTableModelUtil.newDBTable();
//			String[] names = nameList.toArray( new String[0] );
//			table.init( names.length );
			// 4.3.5.0 (2009/02/01) サイズの初期値指定
			int size = nameList.size();
			String[] names = nameList.toArray( new String[size] );
			table.init( size );
			setTableDBColumn( names );

			valueClmIdx = new int[posList.size()];
			for( int i = 0; i<posList.size(); i++ ) {
				valueClmIdx[i] = posList.get( i ).intValue();
			}
		}
	}

	/**
	 * 行、列(セル)単位の情報を読み取り、DBTableModelに値をセットします
	 * 
	 * @param rowInfo 行オブジェクト
	 */
	private void makeBody( final RowInfo rowInfo ) {
		int rowRepeat = rowInfo.rowRepeat;
		CellInfo[] cellInfos = rowInfo.cellInfos;
		int cellLen = cellInfos.length;
		boolean isExistData = false;

		List<String> colData = new ArrayList<String>();
		for ( int cellIdx = 0; cellIdx < cellLen; cellIdx++ ) {
			CellInfo cellInfo = cellInfos[cellIdx];
			for ( int cellRep = 0; cellRep < cellInfo.colRepeat; cellRep++ ) {
				colData.add( cellInfo.text );
				if( cellInfo.text.length() > 0 ) {
					isExistData = true;
				}
			}
		}

		if( isExistData ) {
			// 初めの列(#NAMEが記述されていた列)の値が#で始まっている場合は、コメント行とみなす。
			String firstVal = colData.get( firstClmIdx );
			if( firstVal.length() > 0 && firstVal.startsWith( "#" ) ) {
				return;
			}
			else {
				String[] vals = new String[valueClmIdx.length];
				for( int col = 0; col < valueClmIdx.length; col++ ) {
					vals[col] = colData.get( valueClmIdx[col] );
				}

				// 重複行の繰り返し処理
				for ( int rowIdx = 0; rowIdx < rowRepeat; rowIdx++ ) {
					// テーブルモデルにデータをセット
					if ( numberOfRows < getMaxRowCount() ) {
						table.addColumnValues( vals );
						numberOfRows++;
					}
					else {
						table.setOverflow( true );
					}
				}
			}
		}
		// 全くデータが存在しない行は読み飛ばし
		else {
			return;
		}
	}

	/**
	 * ODSファイルに含まれるcontent.xmlをDOMパーサーでパースし、行、列単位に
	 * オブジェクトに変換します。
	 *
	 */
	private static class DomOdsParser{

		// OpenOffice.org Calc tag Names
//		private static final String OFFICE_DOCUMENT_CONTEBT_ELEM = "office:document-content";
//		private static final String OFFICE_SPREADSHEET_ELEM = "office:spreadsheet";
		private static final String TABLE_TABLE_ELEM = "table:table";
//		private static final String TABLE_TABLE_COLUMN_ELEM = "table:table-column";
		private static final String TABLE_TABLE_ROW_ELEM = "table:table-row";
		private static final String TABLE_TABLE_CELL_ELEM = "table:table-cell";
		private static final String TEXT_P_ELEM = "text:p";

		// Sheet tag attributes
		private static final String TABLE_NAME_ATTR = "table:name";
//		private static final String OFFICE_VALUE_YPE_ATTR = "office:value-type";
		private static final String TABLE_NUMBER_ROWS_REPEATED_ATTR = "table:number-rows-repeated";
		private static final String TABLE_NUMBER_COLUMNS_REPEATED_ATTR = "table:number-columns-repeated";
//		private static final String TABLE_NUMBER_ROWS_SPANNED_ATTR = "table:number-rows-spanned";
//		private static final String TABLE_NUMBER_COLUMNS_SPANNED_ATTR = "table:number-columns-spanned";

//		ArrayList<RowInfo> rowInfoList = new ArrayList<RowInfo>();
		List<RowInfo> rowInfoList = new ArrayList<RowInfo>();
		/**
		 * DomパーサでXMLをパースする
		 * 
		 * @param inputStream
		 * @param sheetName
		 */
		public void doParse( final InputStream inputStream, final String sheetName ) {
			try {
				// ドキュメントビルダーファクトリを生成
				DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
				dbFactory.setNamespaceAware( true );

				// ドキュメントビルダーを生成
				DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
				// パースを実行してDocumentオブジェクトを取得
				Document doc = dBuilder.parse( inputStream );
				processBook( doc, sheetName );
			}
			catch ( ParserConfigurationException ex ) {
				throw new HybsSystemException( ex );
			}
			catch ( SAXException ex ) {
				String errMsg = "ODSファイル中に含まれるcontent.xmlがXML形式ではありません。";
				throw new HybsSystemException( errMsg, ex );
			}
			catch ( IOException ex ) {
				throw new HybsSystemException( ex );
			}
		}
		
		/**
		 * 行オブジェクトのリストを返します。
		 * 
		 * @return List<RowInfo>
		 */
		public List<RowInfo> getRowInfoList() {
			return rowInfoList;
		}

		/**
		 * ODSファイル全体のパースを行い、処理対象となるシートを検索します。
		 * 
		 * @param doc
		 * @param sheetName
		 */
		private void processBook( final Document doc, final String sheetName ) {
			// table:tableを探す
			NodeList sheetList = doc.getElementsByTagName( TABLE_TABLE_ELEM );
			int listLen = sheetList.getLength();

			Element sheet = null;
			// シート探し：シート名があれば、そのまま使用、なければ、最初のシートを対象とします。
			for ( int idx = 0; idx < listLen; idx++ ) {
				Element st = (Element)sheetList.item( idx );
				if ( ( sheetName == null || sheetName.length() == 0 ) || sheetName.equals( st.getAttribute( TABLE_NAME_ATTR ) ) ) {
					sheet = st;
					break;
				}
			}
			// 指定のシートがなければ、エラー
			if ( sheet == null ) {
				String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]";
				throw new HybsSystemException( errMsg );
			}
			else {
				processSheet( sheet );
			}
		}

		/**
		 * ODSファイルのシート単位のパースを行い、行単位のオブジェクトを生成します。
		 * 
		 * @param sheet
		 */
		private void processSheet( final Element sheet ) {
			NodeList rows = sheet.getElementsByTagName( TABLE_TABLE_ROW_ELEM );
			int listLen = rows.getLength();
			int rowRepeat;
			for ( int idx = 0; idx < listLen; idx++ ) {
				Element row = (Element)rows.item( idx );
				// 行の内容が全く同じ場合、table:number-rows-repeatedタグにより省略される。
				String repeatStr = row.getAttribute( TABLE_NUMBER_ROWS_REPEATED_ATTR );
				if ( repeatStr == null || repeatStr.length() == 0 ) {
					rowRepeat = 1;
				}
				else {
					rowRepeat = Integer.parseInt( repeatStr, 10 );
				}
				
				processRow( row, rowRepeat );
			}
		}

		/**
		 * ODSファイルの行単位のパースを行い、カラム単位のオブジェクトを生成します。
		 * 
		 * @og.rev 4.3.5.0 (2009/02/01) toArray するときに、サイズの初期値指定を追加
		 * 
		 * @param row
		 * @param rowRepeat
		 */
		private void processRow( final Element row, final int rowRepeat ) {
			NodeList cells = row.getElementsByTagName( TABLE_TABLE_CELL_ELEM );
			int listLen = cells.getLength();
			int colRepeat;
			String cellText;
			ArrayList<CellInfo> cellInfoList = new ArrayList<CellInfo>();
			for ( int idx = 0; idx < listLen; idx++ ) {
				Element cell = (Element)cells.item( idx );
				// カラムの内容が全く同じ場合、table:number-columns-repeatedタグにより省略される。
				String repeatStr = cell.getAttribute( TABLE_NUMBER_COLUMNS_REPEATED_ATTR );
				if ( repeatStr == null || repeatStr.length() == 0 ) {
					colRepeat = 1;
				}
				else {
					colRepeat = Integer.parseInt( repeatStr, 10 );
				}

				// text:p
				NodeList texts = cell.getElementsByTagName( TEXT_P_ELEM );
				if ( texts.getLength() == 0 ) {
					cellText = "";
				}
				else {
					StringBuilder sb = new StringBuilder();
					NodeList textElems = texts.item( 0 ).getChildNodes();
					int textElemLen = textElems.getLength();
					for ( int idxt = 0; idxt < textElemLen; idxt++ ) {
						Node textElem = textElems.item( idxt );
						if ( textElem.getNodeType() == Node.TEXT_NODE ) {
							sb.append( textElem.getNodeValue() );
						}
					}
					cellText = sb.toString();
				}
				cellInfoList.add( new CellInfo( colRepeat, cellText ) );
			}

//			if ( cellInfoList.size() > 0 ) {
			if ( ! cellInfoList.isEmpty() ) {
//				rowInfoList.add( new RowInfo( rowRepeat, cellInfoList.toArray( new CellInfo[0] ) ) );
		 		// 4.3.5.0 (2009/02/01) toArray するときに、サイズの初期値指定を追加
				rowInfoList.add( new RowInfo( rowRepeat, cellInfoList.toArray( new CellInfo[cellInfoList.size()] ) ) );
			}
		}
	}

	/**
	 * ODSファイルの行情報を表す構造体
	 */
	private static final class RowInfo {
		public final int rowRepeat;
		public final CellInfo[] cellInfos;

		RowInfo( final int rep, final CellInfo[] cell ) {
			rowRepeat = rep;
			cellInfos = cell;
		}
	}

	/**
	 * ODSファイルのカラム情報を表す構造体
	 */
	private static final class CellInfo {
		public final int colRepeat;
		public final String text;

		CellInfo( final int rep, final String tx ) {
			colRepeat = rep;
			text = tx;
		}
	}
}
