/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Locale ;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;

// import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;

/**
 * ネイティブEXCELファイルの書き出しクラスです。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，MIcrosoft Excelファイルの出力機能を実現しています。
 * 
 * 出力形式は、openXML形式にも対応しています。
 * 出力ファイルの拡張子が、.xlsならExcel2003のバイナリ形式、.xlsxならExcel2007の
 * openXML形式で出力されます。
 *
 * @og.group ファイル出力
 *
 * @og.rev 4.3.4.3 (2008/12/22) 一部protected化
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * 
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_Excel extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private Workbook wb			= null;
	private Sheet	sheet		= null;
	protected OutputStream out		= null;
	protected	int		nRowIndex	= 0;
	private String  sheetName		= "Sheet1";			// 3.5.4.3 (2004/01/05)
	private String  refSheetName	= null;				// 3.5.4.3 (2004/01/05)
	private String  filename		= null;		// 3.5.4.3 (2004/01/05)
	private String  refFilename		= null;		// 3.5.4.3 (2004/01/05)
	private String	fontName		= null;		// 3.8.5.3 (2006/08/07)
	private short	fontPoint		= -1;		// 3.8.5.3 (2006/08/07)
	private CreationHelper createHelper	= null;	// poi.xssf対応
	// 5.1.4.0 (2010/03/01) 行番号情報を、出力する(true)/しない(false)を指定
	private boolean		useNumber	= true;

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、EXCEL 書き出し時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応 、useNumber属性対応
	 *
	 * @see #isExcel()
	 */
	public void writeDBTable() {
		if( ! createDBColumn() ) { return ; }

		useNumber = isUseNumber();

//		numberOfColumns = getDBTableModel().getColumnCount();

//		if( numberOfColumns <= 0 ) { return; }

		// 3.5.6.0 (2004/06/18) 移動
		if( filename == null ) {
			String errMsg = "ファイルが指定されていません。";
			throw new HybsSystemException(errMsg );
		}

		// メモリにEXCELデータを作る
		boolean isRefFileExisted  = false;
		boolean isRefFile = false;
		boolean isWorkFileExisted = checkAvailabity(filename);
		boolean hasFile   = isWorkFileExisted;
		String  nameUse   = filename;

		// 同じワークブーク中に雛型シートが存在してある場合
		boolean hasRefSheet = ((null != refSheetName) && (0 <= refSheetName.length()));

		if( hasRefSheet && (null != refFilename) && (0 < refFilename.length())) {
			if(isWorkFileExisted ) {
				if( 0 == refFilename.compareToIgnoreCase(filename) ) {
					nameUse = filename;
					hasFile = true;
				}
				else {
					String errMsg = "追加の時、雛型ファイル名と出力ファイル名が同じしか対応していなかった[" + refFilename + "]"  ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				nameUse = refFilename;
				hasFile = true;
				isRefFile = true;
			}
		}

		if( hasFile ) {
			wb = createWorkbook(nameUse);
		}
		else {
			// 新規の場合、ファイル名に.xlsxで終了した場合⇒.xlsx形式ファイル作成、その他⇒.xls形式ファイル作成
			if(filename.toLowerCase(Locale.JAPAN).endsWith( ".xlsx" ) ) {
				wb = new org.apache.poi.xssf.usermodel.XSSFWorkbook();
			}
			else {
				wb = new org.apache.poi.hssf.usermodel.HSSFWorkbook();
			}

			// 3.8.6.0 (2006/08/07) フォント名やフォントサイズの指定
			Font font = wb.getFontAt((short)0);
			if( fontName != null ) {
				font.setFontName( fontName );	// "ＭＳ Ｐゴシック" など
			}
			if( fontPoint > 0 ) {
				font.setFontHeightInPoints( fontPoint );
			}
		}

		int nSheetIndex = wb.getSheetIndex(sheetName);
		int nSheetPattern = -1;

		if( isRefFileExisted ) {
			sheet = wb.createSheet();
		}
		else {
			if( hasRefSheet ) { nSheetPattern = wb.getSheetIndex(refSheetName); }

			if( isRefFile ) {
				if(-1 >= nSheetPattern ) {
					String errMsg = "雛型の中に参照としてのシートが存在しません[" + refFilename + "]"  ;
					throw new HybsSystemException( errMsg );
				}
				while(true) {
					int nTotalSheets = wb.getNumberOfSheets();

					if( 1 == nTotalSheets ) { break; }

					for( int nIndex = ( nTotalSheets - 1 ); nIndex >= 0; nIndex--) {
						if( nIndex != nSheetPattern ) { wb.removeSheetAt(nIndex); }
					}

					if( hasRefSheet ) { nSheetPattern = wb.getSheetIndex(refSheetName); }
				}
			}
			else {
				// 新規の場合シートが存在すると、そのシートを削除
				if( -1 < nSheetIndex && !isAppend() && ( nSheetIndex != nSheetPattern ) && hasFile ) {
					wb.removeSheetAt(nSheetIndex);
				}
			}
			// シートを削除して、もう一回雛型シートの位置を求める
			if( hasRefSheet ) { nSheetPattern = wb.getSheetIndex(refSheetName); }

			sheet = (-1 >= nSheetPattern) ? wb.createSheet() : wb.cloneSheet(nSheetPattern);

			// 雛型ファイルを使っていた場合、その雛形シートを削除する
			if(isRefFile) { wb.removeSheetAt(nSheetPattern); }
		}

		wb.setSheetName(wb.getNumberOfSheets() -1, getNewSheetNameByName(wb, sheetName) );

		// poi.xssf対応(2009/04/08)
		createHelper = wb.getCreationHelper();

		nRowIndex = 0;

		super.writeDBTable( null );

		// 余計な行を削除
		removeSheetRow( sheet, nRowIndex );

		// メモリ中のデータをファイルに書き込む
		// 3.5.6.0 (2004/06/18) close を finally で処理するように変更。
		try {
			FileOutputStream fileOut = null ;
			try {
				fileOut = new FileOutputStream(filename);
				wb.write(fileOut);
			}
			finally {
				Closer.ioClose( fileOut );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				if( null != sheet ) { sheet = null; }
				if( null != wb    ) { wb    = null; }
			}
		}
		catch( IOException e) {
			String errMsg = "ファイルへ書込み中にエラーが発生しました。";
			throw new HybsSystemException( errMsg,e );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		// メモリ中のデータをファイルに書き込む
	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 * @og.rev 3.8.5.3 (2006/08/07) フォント名やフォントサイズの指定
	 * @og.rev 4.0.0 (2006/09/31) UnsupportedOperationException を発行します。
	 *
	 * @param	writer PrintWriter
	 */
	public void writeDBTable( final PrintWriter writer )  {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * PrintWriter に DBTableModelのラベル情報を書き込みます。
	 * 第一カラム目は、ラベル情報を示す "#Label" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeLabel( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex;
		Row  oRow;

		nColIndex = 0;
		oRow = setFirstCellValue( nRowIndex++, nColIndex++, "#Label" );
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = dbColumn[clm].getLabel();
			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻して、値に # を付ける。
			if( i==0 && !useNumber ) {
				nColIndex-- ;
				val = "#" + val;
			}
//			setRowCellValue( oRow, nColIndex++, dbColumn[clm].getLabel(),Cell.CELL_TYPE_STRING );
			setRowCellValue( oRow, nColIndex++, val,Cell.CELL_TYPE_STRING );
		}

		// 余計なセルを削除
		removeRowCell( oRow, nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelの項目名情報を書き込みます。
	 * 第一カラム目は、項目名情報を示す "#Name" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeName( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex;
		Row  oRow;

		nColIndex = 0;
		oRow = setFirstCellValue( nRowIndex++, nColIndex++, "#Name" );
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = table.getColumnName(clm);
			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻して、値に # を付ける。
			if( i==0 && !useNumber ) {
				nColIndex-- ;
				val = "#" + val;
			}
//			setRowCellValue( oRow, nColIndex++, table.getColumnName(clm),HSSFCell.CELL_TYPE_STRING );
			setRowCellValue( oRow, nColIndex++, val,Cell.CELL_TYPE_STRING );
		}

		// 余計なセルを削除
		removeRowCell( oRow, nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのサイズ情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Size" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBColumn の size と maxlength の 意味を変更
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeSize( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex;
		Row  oRow;

		nColIndex = 0;
		oRow = setFirstCellValue( nRowIndex++, nColIndex++, "#Size" );
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			// 4.0.0 (2005/01/31) メソッド名変更
			String val = String.valueOf(dbColumn[clm].getTotalSize());
			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻して、値に # を付ける。
			if( i==0 && !useNumber ) {
				nColIndex-- ;
				val = "#" + val;
			}
			setRowCellValue( oRow, nColIndex++, val, Cell.CELL_TYPE_NUMERIC );
		}

		// 余計なセルを削除
		removeRowCell( oRow, nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのクラス名情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Class" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeClass( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex;
		Row  oRow;

		nColIndex = 0;
		oRow = setFirstCellValue( nRowIndex++, nColIndex++, "#Class" );
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = dbColumn[clm].getClassName();
			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻して、値に # を付ける。
			if( i==0 && !useNumber ) {
				nColIndex-- ;
				val = "#" + val;
			}
//			setRowCellValue( oRow, nColIndex++, dbColumn[clm].getClassName(),Cell.CELL_TYPE_STRING );
			setRowCellValue( oRow, nColIndex++, val,Cell.CELL_TYPE_STRING );
		}

		// 余計なセルを削除
		removeRowCell( oRow, nColIndex );
	}

	/**
	 * PrintWriter に セパレーターを書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#----" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeSeparator( final DBTableModel table,final PrintWriter writer ) {
		String sep = "----" ;
		short nColIndex;
		Row  oRow;

		nColIndex = 0;
		oRow = setFirstCellValue( nRowIndex++, nColIndex++, "#----" );
		for( int i=0; i<numberOfColumns; i++ ) {
			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、単になにもしないだけ
			if( i==0 && !useNumber ) {
				continue;
			}
			setRowCellValue( oRow, nColIndex++, sep,Cell.CELL_TYPE_STRING );
		}

		// 余計なセルを削除
		removeRowCell( oRow, nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション（"）で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 3.8.5.3 (2006/08/07) DBType の nativeType に対応した、CELL_TYPE をセットします。
	 * @og.rev 4.1.1.2 (2008/02/28) NativeタイプをEnum型(fukurou.model.NativeType)に変更
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows =	table.getRowCount();

		short nColIndex;
		Row  oRow;

		// 5.1.4.0 columns 対応。forループは、引数i で廻す。
		boolean[] nvar = new boolean[numberOfColumns];
		int[] cellType = new int[numberOfColumns];
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			NativeType nativeType = dbColumn[clm].getNativeType();
			switch( nativeType ) {
				case INT    :
				case LONG   :
				case DOUBLE :
					cellType[i] = Cell.CELL_TYPE_NUMERIC ;
						break;
				case STRING :
				case CALENDAR :
				default :
						cellType[i] = Cell.CELL_TYPE_STRING ;
						break;
			}
			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType()) ;
		}

		for( int row=0; row<numberOfRows; row++ ) {
			nColIndex = 0;
			oRow = setFirstCellValue( nRowIndex++, nColIndex++, String.valueOf( row+1 ) );

			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻す。
			if( !useNumber ) {
				nColIndex-- ;
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue(row,clm);
				if( nvar[i] ) {
					val = StringUtil.getReplaceEscape( val );
				}

				setRowCellValue( oRow, nColIndex++, val,cellType[i] );
			}

			// 余計なセルを削除
			removeRowCell( oRow, nColIndex );
		}
	}

	/**
	 * Excelの指定行の一番目セルにデータを設定する。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * @og.rev 4.3.4.3 (2008/12/22) protected化
	 * 
	 * @param	 indexOfRow 行の番号
	 * @param	 indexOfCell セルの番号
	 * @param	 dataVal    String文字列
	 * @return	 Rowのobject型
	 */
	protected Row setFirstCellValue( final int indexOfRow, final int indexOfCell, final String dataVal ) {
		Row  oRow = sheet.getRow( indexOfRow );
		if( oRow == null ) { oRow = sheet.createRow( indexOfRow ); }
		Cell oCell = oRow.getCell( indexOfCell );
		if( null == oCell ) { oCell = oRow.createCell( indexOfCell ); }

		RichTextString richText = createHelper.createRichTextString( dataVal );
		oCell.setCellValue( richText );

		return oRow;
	}

	/**
	 * Excelの指定セルにデータを設定する。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * @og.rev 4.3.4.3 (2008/12/22) protected化
	 *
	 * @param	 oThisRow   Row型のオブジェクト
	 * @param	 indexOfCell セルの番号
	 * @param	 dataVal    String文字列
	 * @param	 cellType   int(Cell.CELL_TYPE_STRING/Cell.CELL_TYPE_NUMERIC)
	 */
	protected void setRowCellValue( final Row oThisRow, final int indexOfCell, final String dataVal,final int cellType ) {
		Cell oCell = oThisRow.getCell( indexOfCell );
		if( null == oCell ) { oCell = oThisRow.createCell( indexOfCell ); }

		if( cellType == Cell.CELL_TYPE_NUMERIC ) {
			oCell.setCellValue( StringUtil.parseDouble( dataVal ) );
		}
		else {
			RichTextString richText = createHelper.createRichTextString( dataVal );
			oCell.setCellValue( richText );
		}
	}

	/**
	 * Excelの指定セルをシートから削除する。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * @og.rev 4.3.4.3 (2008/12/22) protected化
	 *
	 * @param	 oThisRow   Row型のオブジェクト
	 * @param	 nBegin     セルの開始番号
	 */
	protected void removeRowCell( final Row oThisRow, final int nBegin ) {
		int nEnd = oThisRow.getLastCellNum();
		for( int nIndex = nBegin; nIndex <= nEnd; nIndex++) {
			Cell oCell = oThisRow.getCell( nIndex );
			if( null != oCell ) { oThisRow.removeCell(oCell); }
		}
	}

	/**
	 * Excelの指定行をシートから削除する。
	 *
	 * @param	 oThisSheet Sheet型のオブジェクト
	 * @param	 nBegin     行の開始番号
	 * @param	 nEnd       行の終了番号
	 */
	private void removeSheetRow( final Sheet oThisSheet, final int nBegin ) {
		int nEnd = oThisSheet.getLastRowNum();
		for( int nIndex = nBegin; nIndex <= nEnd; nIndex++) {
			Row oRow = oThisSheet.getRow( nIndex );
			if( null != oRow ) { oThisSheet.removeRow(oRow); }
		}
	}

	/**
	 * DBTableModelのデータとして書き込むときのシート名をセットします。
	 * デフォルトは、"Sheet1" です。同じ名称のシートが存在する場合、そのシート
	 * 名の後に(2)、(3)のような文字列をくっ付けます。
	 *
	 * @param	 workbook   Workbook
	 * @param	 nameSet    String文字列,指定のシート名
	 * @return String シート名
	 */
	protected String getNewSheetNameByName( final Workbook workbook, final String nameSet) {
		String nameSheet = nameSet;
		String strAppendix;
		// POIのソースからみると、シートの名前は30桁文字(31個文字)だと思われる。
		int nMaxLen = 30;
		int nCount = 1;
		int nIndex = 0;
		while( nIndex > -1) {
			if( nCount >= 2 ) {
				strAppendix = "(" + Integer.toString(nCount) + ")";
				if(nameSet.length() < ( nMaxLen - strAppendix.length()) ) {
					nameSheet = nameSet + strAppendix;
				}else {
					nameSheet = nameSet.substring(0, nMaxLen - strAppendix.length()) + strAppendix;
				}
			}
			nIndex = workbook.getSheetIndex(nameSheet);
			nCount++;
		}

		return nameSheet;
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、"Sheet1" です。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		if( sheetName != null ) { this.sheetName = sheetName; }
	}

	/**
	 * EXCEL雛型参考ファイルのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * デフォルトは、null（第一シート） です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheetName String
	 */
	public void setRefSheetName( final String sheetName )  {
		if( sheetName != null ) { refSheetName = sheetName; }
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return boolean
	 */
	public boolean isExcel() {
		return true;
	}

	/**
	 * 出力先ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename EXCEL雛型参考ファイル名
	 * @throws UnsupportedOperationException
	 */
	public void setFilename( final String filename ) {
		this.filename = filename;
	}

	/**
	 * EXCEL雛型参考ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename EXCEL雛型参考ファイル名
	 */
	public void setRefFilename( final String filename ) {
		refFilename = filename;
	}

	/**
	 * EXCEL出力時のデフォルトフォント名を設定します。<br />
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。<br />
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   fontName String
	 */
	public void setFontName( final String fontName ) {
		this.fontName = fontName ;
	}

	/**
	 * EXCEL出力時のデフォルトフォントポイント数を設定します。<br />
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point short
	 */
	public void setFontPoint( final short point ) {
		fontPoint = point;
	}

	/**
	 * EXCELファイルのWookbookというStream(MicrosoftのOLE用語)を作ります
	 * 条件によって、新規かとファイルから読み込み書き込みかが分かれます。
	 *
	 * @param   fname EXCEL雛型参考ファイル名
	 * @return  Workbook
	 */
	protected Workbook createWorkbook( final String fname ) {
		final Workbook myWookbook ;
		FileInputStream fileIn  = null;
		try {
			fileIn = new FileInputStream(fname);
			myWookbook = WorkbookFactory.create(fileIn);
		}
		catch ( Exception e ) {
			String errMsg = "ファイル読込みエラー[" + fname + "]"  ;
			throw new HybsSystemException( errMsg,e );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.ioClose( fileIn );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}

		return myWookbook;
	}

	/**
	 * 指定の名前のファイルを使うかどうか確認します。
	 *
	 * @param   fname EXCEL雛型参考ファイル名
	 * @return  boolean
	 */
	private boolean checkAvailabity( final String fname ) {
		boolean bRet = false;
		// 4.0.0.0 (2007/11/29) 入れ子if の統合
		if( isAppend() && null != fname ) {
			File oFile = new File(fname);
			if(oFile.exists() &&  oFile.isFile() && (oFile.length() > 0)) { bRet = true; }
		}
		return bRet;
	}
}
