/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

/**
 * <p>TableFilter_LABEL は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。<br />
 *
 * ここでは、キーのCLM,LBL と、LANG より、ラベルリソースに変換した値を設定します。
 *  CLM :ラベルリソースのキーとなる値が設定されているカラム名を指定します。
 *  LBL :CLMで指定されたカラム名から値を取り出し、ラベルリソースに変換した結果をセットします。
 *  LANG:ラベルリソースの言語を指定します。無指定時は、日本語になります。
 *
 * CLM と LBL に同じカラムを指定すると、このフィルターを通過するたびに、変換後のラベルが
 * 再びキーとして変換しようと試みることになります。不測の事態に陥る可能性があるため、
 * その様な設定は出来なくなっています。
 *
 * また、CLM または LBL で指定したカラムが DBTableModel に存在しない場合は、処理そのものを
 * 無視します。その場合は、警告も出力されませんので、ご注意ください。
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_LABEL extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * DBTableModel処理を実行します。<br />
	 *
	 * @return DBTableModel
	 */
	public DBTableModel execute() {
		String lang = getValue( "LANG" );
		ResourceManager resource = ResourceFactory.newInstance( lang );

		String clm = getValue( "CLM" );
		int clmNo  = table.getColumnNo( clm,false );	// 存在しない場合は、-1 を返す。
		String lbl = getValue( "LBL" );
		int lblNo  = table.getColumnNo( lbl,false );

		if( clmNo == lblNo && clmNo >= 0 ) {
			String errMsg = "CLM と LBL に同じカラムを指定することは出来ません。" ;
			ErrorMessage errMessage = makeErrorMessage( "TableFilter_LABEL Error",ErrorMessage.NG );
			errMessage.addMessage( 0,ErrorMessage.NG,errMsg );
			errMessage.addMessage( 0,ErrorMessage.NG,"CLM=[" + clm + "],LBL=[" + lbl + "]" );
		}
		else if( clmNo >= 0 && lblNo >= 0 ) {
			String[] data  = null;
			int rowCnt = table.getRowCount();
			for( int row=0; row<rowCnt; row++ ) {
				String clmVal = null;
				try {
					data   = table.getValues( row );
					clmVal = data[clmNo];		// 変換する元のカラム名
					data[lblNo] = resource.getLabel( clmVal ) ;		// 副作用を及ぼしています。注意
				}
				catch( RuntimeException ex ) {
					ErrorMessage errMessage = makeErrorMessage( "TableFilter_LABEL Error",ErrorMessage.NG );
					errMessage.addMessage( row+1,ErrorMessage.NG,ex.getMessage() );
					errMessage.addMessage( row+1,ErrorMessage.NG,StringUtil.array2csv( data ) );
					errMessage.addMessage( row+1,ErrorMessage.NG,"CLM=[" + clm + "],VAL=[" + clmVal + "]" );
				}
			}
		}

		return table;
	}
}
