/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.hayabusa.html.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

/**
 * INDBMENU エディターは、カラムの表示パラメーターのＳＱＬ文を実行結果より、
 * 作成したプルダウンメニューと、テキストフィールドによる入力の両方をサポートする、
 * 編集に使用するクラスです。
 *
 * JavaScript によりテキストフィールドとメニュー（コンボボックス）を重ね合わせて
 * 表示しておき、メニューで選択した値を、テキストフィールドに設定しています。
 * このエディタを使用するには、jsp/common/inputMenu.js を予め使用できるように
 * 設定しておく必要があります。
 * 
 * このエディタはeventColumnに対応していません。
 * 
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_INDBMENU extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final String SEL1 = "<script type=\"text/javascript\">makeInputMenu('" ;
	private static final String SEL2 = "');</script>" ;

	private final String query ;
	private final String dbid ;
	private final String lang ;				// 4.0.0 (2006/11/15)
	private final boolean addNoValue ;		// 3.5.5.7 (2004/05/10)
	private final boolean seqFlag ;			// 3.6.0.6 (2004/10/22)
	private final TagBuffer selTagBuffer = new TagBuffer() ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 */
	public Editor_INDBMENU() {
		// 4.3.4.4 (2009/01/01)
//		super();
		query   = null;
		dbid	= null;
		lang	= null;			// 4.0.0 (2006/11/15)
		addNoValue = false;		// 3.5.5.7 (2004/05/10)
		seqFlag    = false;		// 3.6.0.6 (2004/10/22)
		name	= null;			// 4.3.4.0 (2008/12/01)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0 (2006/11/24) TextField分の属性設定
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Editor_INDBMENU( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		addNoValue	= clm.isAddNoValue() ;		// 3.5.5.7 (2004/05/10)
		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();			// 4.0.0 (2006/11/15)
		seqFlag		= false;		// 3.6.0.6 (2004/10/22)
		name		= clm.getName(); //4.3.4.0 (2008/12/01)

		if( query == null || query.length() == 0 ) {
			String errMsg = "DBMENU Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}

		String	disabled = clm.isWritable() ? null : "disabled" ;

		Attributes selAttri = new Attributes();
		selAttri.set( "disabled"	,disabled );
		selAttri.set( "onChange" ,"selChanged(this);" );	// INDBMENU 特有のJavaScript

		selAttri.addAttributes( clm.getEditorAttributes() );
		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_INDBMENU( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 *
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 *
	 * @param	value String
	 * @return	データの編集用文字列
	 */
	public String getValue( final String value ) {

		// input タグの作成
		TagBuffer intag = new TagBuffer( "input" );
		intag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			intag.add( "id"      , name );		// INDBMENU 特有のJavaScript用のキー
		}
		intag.add( "value"   , value );
		intag.add( "size"    , size1 );
		intag.add( tagBuffer.makeTag() );
		intag.add( optAttr );

		// select タグの作成
		TagBuffer seltag = new TagBuffer( "select" );
		seltag.add( "id"      , name + ".sel" );	// INDBMENU 特有のJavaScript用のキー
		seltag.add( "style"   , "position:absolute;" );	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
		seltag.add( selTagBuffer.makeTag() );
		seltag.add( optAttr );

		seltag = getOption( seltag,value );

		return intag.makeTag() + HybsSystem.CR +
				seltag.makeTag() + HybsSystem.CR +
				SEL1 + name + SEL2;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 *
	 * @og.rev 3.8.5.1 (2006/04/28) makeInputMenu 呼び出し時の引数記述ミスを修正
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 *
	 * @param	row   int 行番号
	 * @param	value String
	 * @return	データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {

		String name2 = name + HybsSystem.JOINT_STRING + row ;

		// input タグの作成
		TagBuffer intag = new TagBuffer( "input" );
		intag.add( "name"    , name2 );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			intag.add( "id"      , name2 );		// INDBMENU 特有のJavaScript用のキー
		}
		intag.add( "value"   , value );
		intag.add( "size"    , size2 );
		intag.add( tagBuffer.makeTag() );
		intag.add( optAttr );

		// select タグの作成
		TagBuffer seltag = new TagBuffer( "select" );
		seltag.add( "id"      , name2 + ".sel" );		// INDBMENU 特有のJavaScript用のキー
		seltag.add( "style"   , "position:absolute;" );	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
		seltag.add( selTagBuffer.makeTag() );
		seltag.add( optAttr );

		seltag = getOption( seltag,value );

		return intag.makeTag( row,value ) + HybsSystem.CR +
				seltag.makeTag( row,value ) + HybsSystem.CR +
				SEL1 + name2 + SEL2;
	}

	/**
	 * 初期値が選択済みの 選択肢（オプション）をTagBuffer に反映します。
	 * このオプションは、引数の値を初期値とするオプションタグ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 * 又、$Cには自分自身のカラム名を割り当てます。<br />
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) getOption( String value )の廃止を受けて、新規作成
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $C対応
	 *
	 * @param	buf  TagBuffer 反映させるTagBufferオブジェクト
	 * @param	value String 選択されている値
	 * @return	オプションタグ
	 */
	private TagBuffer getOption( final TagBuffer buf,final String value ) {

		// StringFormat format = new StringFormat( query,value );
		StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01) 
		String newQuery = format.format();
		String newValue = format.getValue();

		Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang );

		if( addNoValue ) {
			buf.setBody( Selection.NO_VALUE_OPTION + selection.getOption( newValue, seqFlag ) );
		}
		else {
			buf.setBody( selection.getOption( newValue, seqFlag ) );
		}

		return buf;
	}
}
