/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.odftoolkit.odfdom.OdfFileDom;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.doc.office.OdfOfficeSpreadsheet;
import org.odftoolkit.odfdom.doc.table.OdfTable;
import org.odftoolkit.odfdom.doc.table.OdfTableCell;
import org.odftoolkit.odfdom.doc.table.OdfTableRow;
import org.odftoolkit.odfdom.doc.text.OdfTextParagraph;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeValueTypeAttribute;
import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.w3c.dom.Node;

/**
 * Calcファイルの書き出しクラスです。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，OpenOfficeのCalcファイルの出力機能を実現しています。
 *
 * @og.group ファイル出力
 * 
 * @version  5.0
 * @author	 Hiroki Nakamura
 * @since    JDK6.0,
 */
public class TableWriter_Calc extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	protected OdfSpreadsheetDocument	wb					= null;
	protected OdfTable					sheet				= null;
	protected OdfFileDom				contentDom			= null;
	protected OdfOfficeSpreadsheet		officeSpreadsheet	= null;
	protected OdfOfficeAutomaticStyles	contentAutoStyles	= null;

	protected boolean					useNumber			= true;	

	private String					sheetName			= "Sheet1";
	private String					refSheetName		= null;
	private String					filename			= null;
	private String					refFilename			= null;
	private String					fontName			= null;			// 現時点では未使用
	private short					fontPoint			= -1;			// 現時点では未使用

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、Calc 書き出し時に使用します。
	 *
	 * @see #isExcel()
	 */
	@Override
	public void writeDBTable() {
		if( !createDBColumn() ) { return; }

		useNumber = isUseNumber();

		if( filename == null ) {
			String errMsg = "ファイルが指定されていません。";
			throw new HybsSystemException( errMsg );
		}

		/* 以下は未実装 ***********************************************/
		if( isAppend() ) {
			String errMsg = "Calcの場合はAppend利用できません。";
			throw new HybsSystemException( errMsg );
		}

		if( ( refFilename != null && refFilename.length() > 0 ) || ( refSheetName != null && refSheetName.length() >= 0 ) ) {
			String errMsg = "Calcの場合は,refFilenamerefSheetName利用できません。";
			throw new HybsSystemException( errMsg );
		}

		fontName = null;
	//	if( fontName != null && fontName.length() > 0 ) {
	//		String errMsg = "Calcの場合はfontName利用できません。";
	//		throw new HybsSystemException( errMsg );
	//	}

		fontPoint = 0;
	//	if( fontPoint > 0 ) {
	//		String errMsg = "Calcの場合はfontPoint利用できません。";
	//		throw new HybsSystemException( errMsg );
	//	}
		/* *************************************************************/

		// 新規の場合
		try {
			wb = OdfSpreadsheetDocument.newSpreadsheetDocument();
			// コンテンツのルートとDOMを取得
			contentDom = wb.getContentDom();
			officeSpreadsheet = wb.getContentRoot();
			contentAutoStyles = contentDom.getOrCreateAutomaticStyles();

			// デフォルトで用意されているノードを削除
			Node childNode = officeSpreadsheet.getFirstChild();
			while( childNode != null ) {
				officeSpreadsheet.removeChild( childNode );
				childNode = officeSpreadsheet.getFirstChild();
			}
		}
		catch( Exception ex ) {
			String errMsg = "Calcの文書宣言ができません。";
			throw new HybsSystemException( errMsg, ex );
		}

		//デフォルトで用意されているStyles調整
		resetAutoStylesAndMasterStyles();

		sheet = new OdfTable( contentDom );
		sheet.setTableNameAttribute( sheetName );

		super.writeDBTable( null );

		officeSpreadsheet.appendChild( sheet );

		try {
			wb.save( filename );
			wb.close();
		}
		catch( Exception ex ) {
			String errMsg = "Calcの文書saveができません。";
			throw new HybsSystemException( errMsg, ex );
		}
		finally {
			if( null != sheet )	{ sheet = null; }
			if( null != wb )	{ wb = null; }
		}
	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @param	writer PrintWriter
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * PrintWriter に DBTableModelのラベル情報を書き込みます。
	 * 第一カラム目は、ラベル情報を示す "#Label" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeLabel( final DBTableModel table, final PrintWriter writer ) {
		OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Label", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = dbColumn[clm].getLabel();
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelの項目名情報を書き込みます。
	 * 第一カラム目は、項目名情報を示す "#Name" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeName( final DBTableModel table, final PrintWriter writer ) {
		OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Name", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = table.getColumnName( clm );
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}
	
	/**
	 * PrintWriter に DBTableModelのサイズ情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Size" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeSize( final DBTableModel table, final PrintWriter writer ) {
		OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Size", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = String.valueOf( dbColumn[clm].getTotalSize() );
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, true, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelのクラス名情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Class" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeClass( final DBTableModel table, final PrintWriter writer ) {
		OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Class", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			int clm = clmNo[i];
			String val = dbColumn[clm].getClassName();
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に セパレーターを書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#----" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeSeparator( final DBTableModel table, final PrintWriter writer ) {
		String sep = "----";

		OdfTableRow row = new OdfTableRow( contentDom );
		row.appendCell( createTextCell( contentDom, "#----", false, false ) );
		for( int i = 0; i < numberOfColumns; i++ ) {
			if( i == 0 && !useNumber ) {
				continue;
			}
			row.appendCell( createTextCell( contentDom, sep, false, false ) );
		}
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション（"）で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows =	table.getRowCount();

		boolean[] nvar = new boolean[numberOfColumns];
		boolean[] cellType = new boolean[numberOfColumns];
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			NativeType nativeType = dbColumn[clm].getNativeType();
			switch( nativeType ) {
				case INT    :
				case LONG   :
				case DOUBLE :
					cellType[i] = true ;
					break;
				case STRING :
				case CALENDAR :
				default :
					cellType[i] = false ;
					break;
			}
			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType() );
		}

		for( int r = 0; r < numberOfRows; r++ ) {
			OdfTableRow row = new OdfTableRow( contentDom );

			if( useNumber ) {
				row.appendCell( createTextCell( contentDom, String.valueOf( r + 1 ), true, true ) );
			}

			for( int i = 0; i < numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue( r, clm );
				if( nvar[i] ) {
					val = StringUtil.getReplaceEscape( val );
				}
				row.appendCell( createTextCell( contentDom, val, cellType[i], false ) );
			}
			row.setStyleName( "ro1" );
			sheet.appendRow( row );
		}
	}

	/**
	 * テキストコンテンツ用のセルを生成する
	 *
	 * @param contentDom OdfFileDom
	 * @param content String
	 * @param isCellTypeNumber Boolean
	 * @param isNumberList Boolean
	 * @return OdfTableCell
	 */
	protected OdfTableCell createTextCell( final OdfFileDom contentDom, final String content, final Boolean isCellTypeNumber, final Boolean isNumberList ) {
		OdfTableCell cell = new OdfTableCell( contentDom );
		if( isCellTypeNumber ) {
			cell.setOfficeValueAttribute( Double.parseDouble( content ) );
			cell.setOfficeValueTypeAttribute( OfficeValueTypeAttribute.Value.FLOAT.toString() );
		}
		else {
			cell.setOfficeValueTypeAttribute( OfficeValueTypeAttribute.Value.STRING.toString() );
		}
		OdfTextParagraph paragraph = new OdfTextParagraph( contentDom, null, content );
		cell.appendChild( paragraph );
		return cell;
	}

  	/**
	 * デフォルトで用意されているStylesを調整します。
	 * ※ここでは何もしません。
	 *
	 */
	protected void resetAutoStylesAndMasterStyles(){
		// Document empty method 対策
	}
	
	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、"Sheet1" です。
	 * これは、Calc追加機能として実装されています。
	 *
	 * @param   sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		if( sheetName != null ) {
			this.sheetName = sheetName;
		}
	}

	/**
	 * Calc雛型参考ファイルのシート名を設定します。
	 * これは、Calc追加機能として実装されています。
	 *
	 * Calcファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にCalcを作成する場合にフォームを設定する事が可能になります。
	 * デフォルトは、null（第一シート） です。
	 *
	 * @param   sheetName String
	 */
	public void setRefSheetName( final String sheetName ) {
		if( sheetName != null ) {
			refSheetName = sheetName;
		}
	}

	/**
	 * このクラスが、Calc対応機能(=Excel対応機能)を持っているかどうかを返します。
	 *
	 * Calc対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @return boolean
	 */
	public boolean isExcel() {
		return true;
	}

	/**
	 * 出力先ファイル名をセットします。(DIR + Filename)
	 * これは、Calc追加機能として実装されています。
	 *
	 * @param   filename Calc雛型参考ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = filename;
	}

	/**
	 * Calc雛型参考ファイル名をセットします。(DIR + Filename)
	 * これは、Calc追加機能として実装されています。
	 *
	 * @param   filename Calc雛型参考ファイル名
	 */
	public void setRefFilename( final String filename ) {
		refFilename = filename;
	}

	/**
	 * Calc出力時のデフォルトフォント名を設定します。<br />
	 * これは、Calc追加機能として実装されています。<br />
	 *
	 * Calcファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、Calcのフォント名をそのまま使用してください。<br />
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
	 *
	 * @param   fontName String
	 */
	public void setFontName( final String fontName ) {
		this.fontName = fontName ;
	}

	/**
	 * Calc出力時のデフォルトフォントポイント数を設定します。<br />
	 * これは、Calc追加機能として実装されています。<br />
	 *
	 * Calcファイルを書き出す時に、デフォルトポイント数を指定します。
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
	 *
	 * @param   point short
	 */
	public void setFontPoint( final short point ) {
		fontPoint = point;
	}
}
