/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * Query画面の拡張検索領域を指定用のタグです。
 * 
 * 通常は属性を何もセットせずにQuery画面の隠しメニューに利用します。
 * queryButtonタグが存在しない場合はボタンを出力したい場所にIDをセットしたspanタグを
 * 配置し、そのIDをこのタグのbuttonId属性で指定して下さい。
 * タグの動きとしてはにはbody内に書いた内容をbuttonIdで指定した領域に動的に作成した
 * ボタン（初期値はシステムパラメータの画像)でON/OFF制御可能とします。
 * なお、body部分にはog:firldsetタグも利用可能です。
 *
 * @og.formSample
 * <pre>
 * ●形式：
 *       ・&lt;og:hideMenu
 *                   [ id           = "uniqHideId"   ] タグで囲われた部分に付けるID
 *                   [ buttonId     = "targetId"     ] ボタンを出力する場所のIDを指定
 *                   [ defaultNone  = "[true|false]" ] 初期状態の開閉指定
 *                   [ openButton    = "open.png"    ] 開く（＋）ボタンの画像
 *                   [ closeButton   = "close.png"   ] 閉じる（－）ボタンの画像
 *         &gt;
 * ●body：あり
 *
 * ●使用例
 * 例１:通常のQUERY画面での利用
 *         &lt;og:hideMenu&gt;
 *           &lt;table&gt;&lt;tr&gt;
 *             &lt;og:column name="CLM"/&gt;
 *           &lt;/tr&gt;&lt;/table&gt;
 *         &lt;/og:hideMenu&gt;
 * 
 * 例２:queryButtonタグが存在しない状況での利用(buttonIdの明示的な指定)
 *         &lt;span id="hideButtonSet"&gt; &lt;/span&gt;
 *         &lt;og:hideMenu&gt;
 *             ......
 *         &lt;/og:hideMenu&gt;
 * </pre>
 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class HideMenuTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/11/30)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/11/30)

	private static volatile int uniqID = 1 ;								// ユニークID作成用
	
	private String id		= null;											// この領域に付けるid
	private String buttonId	= "query-button-append";						// ボタンを出力する場所のid
	private boolean defaultNone = true;										// 開閉初期状態
	
	private String openButton = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_OPEN_BUTTON" );		// 開くボタン
	private String closeButton = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		String IMG = null; // imgタグ
//		String hideClass = "hideMenu" + String.valueOf( uniqID++ ); // ユニークだが、JS側がクラスで制御するため。
		String hideClass = "hideMenu" + ( uniqID++ );				// ユニークだが、JS側がクラスで制御するため。

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		buf.append( "<div class=\"" ).append( hideClass ).append( "\" " );
		if( id != null){ buf.append( "id\"=" + id + "\" " ); }
		if( defaultNone ){
			buf.append( "style=\"display: none;\"" );
			IMG = "  <img id=\"queryHide\" src=\"" + openButton + "\" alt=\"Expand\" border=\"0px\" "+
			" onClick=\"hideClass( event, this, '" + hideClass + "', '" + openButton + "', '" + closeButton + "' )\"/> ";
		}
		else{
			IMG = "  <img id=\"queryHide\" src=\"" + closeButton + "\" alt=\"Expand\" border=\"0px\" "+
			" onClick=\"hideClass( event, this, '" + hideClass + "', '" + openButton + "', '" + closeButton + "' )\"/> ";
		}
		buf.append( ">" );
		
		// 指定したIDにボタン画像を放り込むスクリプトを出力
		if( buttonId != null && buttonId.length() > 0){
			buf.append( "<script language=\"javascript\">" ).append( HybsSystem.CR );
			buf.append( "$(function(){$('#" + buttonId + "').append(\"" );
			buf.append( IMG.replace( "\"", "\\\"" ) );
			buf.append( "\");})" ).append( HybsSystem.CR );
			buf.append( "</script>").append( HybsSystem.CR );
		}

		jspPrint( buf.toString() );
		return( EVAL_BODY_INCLUDE );	// Body インクルード（ extends TagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		jspPrint( HybsSystem.CR + "</div>" );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		id = null;
		buttonId = "query-button-append";
		defaultNone = true;
		openButton = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_OPEN_BUTTON" );		// 開くボタン
		closeButton = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン
	}

	/**
	 * 【TAG】このタグのBODY部分にIDを指定します。
	 * 
	 * @og.tag
	 * 開閉される部分を囲っているDIVにIDを指定します。
	 * 特定のhideMenuにスタイルを指定したい場合等に利用して下さい。
	 *
	 * @param input String BODY部分のID
	 */
	public void setId( final String input) {
		id = nval( getRequestParameter( input ), id );
	}

	/**
	 * 【TAG】画像ボタン出力先のIDを指定します。
	 * 
	 * @og.tag
	 * 初期値は"query-button-append"です。
	 * queryButtonタグが存在する画面では通常指定する必要はありません。
	 * queryButtonタグが存在しない場合は、別途用意したspanタグの
	 * IDを指定する事でボタンが表示されます。
	 * 
	 * @param input String 出力先のID
	 *
	 */
	public void setButtonId( final String input) {
		buttonId = nval( getRequestParameter( input ), buttonId );
	}

	/**
	 * 【TAG】初期状態の開閉指定（初期値：true）
	 * 
	 * @og.tag
	 * 初期状態が開いているか、閉じているかを指定します。
	 * 初期値は閉じる（true)です。
	 * 
	 * @param flag String 開閉指定(true/false)
	 *
	 */
	public void setDefaultNone( final String flag) {
		defaultNone = nval( getRequestParameter( flag ), defaultNone );
	}
	
	/**
	 * 【TAG】開くボタンの画像を指定します。
	 * 
	 * @og.tag
	 * ボタンの画像を指定します。
	 * 相対パスの場合、基底は各画面のフォルダとなります。
	 * デフォルトはjsp/imageフォルダ内のシステムパラメータHIDE_OPEN_BUTTONのファイルです。
	 * 
	 * @param name String 画像ファイル名
	 *
	 */
	public void setOpenButton( final String name) {
		openButton = nval( getRequestParameter( name ), openButton );
	}

	/**
	 * 【TAG】閉じるボタンの画像を指定します。
	 * 
	 * @og.tag
	 * ボタンの画像を指定します。
	 * 相対パスの場合、基底は各画面のフォルダとなります。
	 * デフォルトはjsp/imageフォルダ内のシステムパラメータHIDE_CLOSE_BUTTONのファイルです。
	 * 
	 * @param name String 画像ファイル名
	 *
	 */
	public void setCloseButton( final String name) {
		closeButton = nval( getRequestParameter( name ), closeButton );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION						)
				.println( "id"				,id								)
				.println( "buttonId"		,buttonId						)
				.println( "defaultNone"		,defaultNone					)
				.println( "openButton"		,openButton						)
				.println( "closeButton"		,closeButton					)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
