/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
// import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * データベース関連の便利なメソッドを集めた簡易ユーティリティークラスです。
 * 全てのメソッドは、static メソッドになっています。
 *
 * @og.rev 2.1.1.1 (2002/11/15) Serializable インターフェースを削除する。
 * @og.rev 4.0.0.0 (2007/10/16) 名称変更(DBUtil > DBTableModelUtil) DBアクセス関係のメソッドはfukurou/db/DBUtilに移動
 * @og.group ＤＢ/Shell制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTableModelUtil {
	/**
	 * インスタンスを作らないので、コンストラクタは、private に設定します。
	 */
	private DBTableModelUtil() {}

	/**
	 * 初期データベースに接続して、Queryを実行します。
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,DBTableModel として返されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   resource ResourceManager
	 * @param   appInfo ApplicationInfo
	 * @return  検索結果の配列
	 */
	public static DBTableModel makeDBTable( final String stmt ,final String[] args, final ResourceManager resource, final ApplicationInfo appInfo ) {
//		return makeDBTable( stmt ,args,resource,appInfo,"DEFAULT" );		// 4.0.0 (2005/01/31)
		return makeDBTable( stmt ,args,resource,appInfo,null );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します。
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,DBTableModel として返されます。
	 * 検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 新規追加
	 * @og.rev 3.0.0.1 (2003/02/14) ヘッダー、フッター情報が null のときの処理追加。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   resource ResourceManager
	 * @param   appInfo ApplicationInfo
	 * @param   dbid 接続先ID
	 * @return  検索結果の配列
	 */
	public static DBTableModel makeDBTable( final String stmt ,final String[] args ,
											final ResourceManager resource, final ApplicationInfo appInfo, final String dbid ) {
		if( stmt == null || stmt.length() == 0 ) { return null; }

		DBTableModel table = null;

		Query query = QueryFactory.newInstance( "JDBCPrepared" );

		ErrorMessage errMessage = null;
		try {
			Transaction tran = new TransactionReal( dbid,appInfo );		// 5.1.9.0 (2010/08/01) Transaction 対応
			query.setTransaction( dbid,tran );							// 5.1.9.0 (2010/08/01) Transaction 対応
//			query.setConnectionID( dbid );
			query.setResourceManager( resource );	// 4.0.0 (2005/01/31)
			query.setStatement( stmt );
			query.execute( args );
			int errCode = query.getErrorCode();
			int executeCount = query.getExecuteCount();
			if( errCode < ErrorMessage.NG && executeCount >= 0 ) {		// 異常以外の場合
				table = query.getDBTableModel();
//				if( query.getUpdateFlag() ) { query.commit(); }
				if( query.isUpdate() ) { query.commit(); }
			}
			else {
				errMessage = query.getErrorMessage();
			}
		}
		catch( HybsSystemException ex ) {
			if( query != null ) { query.rollback(); }
			throw ex;
		}
		finally {
			if( query != null ) { query.close(); }
			QueryFactory.close( query );
		}

		if( errMessage != null ) {
			throw new HybsSystemException( errMessage.toString() );
		}

		return table;
	}

	/**
	 * 空の DBTableModel オブジェクトを作成します。
	 * これは、本来、ファクトリクラスで作成すべきですが、簡易作成メソッドとして
	 * DBUtil の static メソッドとして実装します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  DBTableModel
	 */
	public static DBTableModel newDBTable() {
		return new DBTableModelImpl();
	}

	/**
	 * カラム名の配列及びデータの2次元配列からDBテーブルモデルを作成します。
	 * カラム名がセットされていない若しくはデータがセットされていない場合は、nullを返します。
	 *
	 * @og.rev 4.2.1.0 (2008/04/26) 新規追加
	 *
	 * @param   clms カラム名の配列
	 * @param   vals 値の配列
	 * @param   resource ResourceManager
	 * @return  DBテーブルモデル
	 */
	public static DBTableModel makeDBTable( final String[] clms, final String[][] vals, final ResourceManager resource ) {
		if( clms == null || clms.length == 0
				|| vals == null || vals.length == 0 || vals[0] == null || vals[0].length == 0 ) {
			return null;
		}

		if( clms.length != vals[0].length ) {
//			throw new HybsSystemException( "キーのカラム数とデータのカラム数が一致していません" );
			String errMsg = "キーのカラム数とデータのカラム数が一致していません。"
						+ HybsSystem.CR
						+ " clms.length=[" + clms.length + "]  vals.length=[" + vals[0].length + "]"
						+ " clms=" + StringUtil.array2csv( clms ) + HybsSystem.CR
						+ " vals=" + StringUtil.array2csv( vals[0] )  ;	// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		int numberOfColumns = clms.length;
		DBTableModel table = newDBTable() ;
		table.init( numberOfColumns );

		DBColumn[] dbColumn = new DBColumn[numberOfColumns];
		for( int column=0; column<numberOfColumns; column++ ) {
			dbColumn[column] = resource.makeDBColumn( clms[column] );
			table.setDBColumn( column,dbColumn[column] );
		}
		
		int numberOfRows = vals.length;
		for( int row=0; row<numberOfRows; row++ ) {
			table.addColumnValues( vals[row] );
		}

		return table;
	}
}
