/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.html.ViewForm;
import org.opengion.hayabusa.html.ViewFormFactory;
import org.opengion.hayabusa.servlet.MultipartRequest;

/**
 * クライアントのファイルをサーバーにアップロードするタグです。
 *
 * アップロード後の属性は、DBTableModel に格納することにより、通常のデータと
 * 同様の取り出し方が可能です。
 * また、通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、{&#064;XXXX} 変数で取り出すことが可能です。
 *
 * この upload タグでは、アップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * 新ファイル名に拡張子が設定されていない場合は、オリジナルファイル名の拡張子をセット
 * します。
 *
 * @og.formSample
 * ●形式：&lt;og:upload fileURL="…" maxPostSize="…" /&gt;
 * ●body：なし
 *
 * ●使用例 :
 *
 * 【query.jsp】
 *       &lt;form method="POST" action="result.jsp" enctype="multipart/form-data" target="RESULT"&gt;
 *       &lt;table summary="layout" &gt;
 *       &lt;tr&gt;&lt;og:input type="text" name="submitter" value="{&#064;USER.JNAME}" size="20" msg="MSG0014" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;
 *           &lt;og:input type="file" name="file1"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file1_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file2"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file2_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file3"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file3_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:column name="writable"     value="false"  /&gt;
 *       &lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * 【result.jsp】
 *       &lt;og:upload
 *           fileURL     = "{&#064;USER.ID}"
 *       /&gt;
 *       &lt;br /&gt;
 *       &lt;og:message msg="MSG0003" comment="ファイルの登録が完了しました。" /&gt;
 *
 *       &lt;og:view
 *           command      = "NEW"
 *           viewFormType = "HTMLTable"
 *           writable     = "{&#064;writable}"
 *       /&gt;
 *
 *       &lt;table&gt;
 *       &lt;tr&gt;&lt;og:input name="submitter" value="{&#064;submitter}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="writable"  value="{&#064;writable}"  /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="directory" value="{&#064;directory}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1"     value="{&#064;file1}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1_NEW" value="{&#064;file1_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1_ORG" value="{&#064;file1_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2"     value="{&#064;file2}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2_NEW" value="{&#064;file2_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2_ORG" value="{&#064;file2_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3"     value="{&#064;file3}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3_NEW" value="{&#064;file3_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3_ORG" value="{&#064;file3_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileManagerTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

//	 3.5.2.0 (2003/10/20) カラム名に、ISFILEを追加。
	private static final String   ENCODE = "UTF-8";		// 3.5.2.0 (2003/10/20) 廃止
	private String  filename    = null;				// 3.5.4.2 (2003/12/15)

	private int 	maxPostSize = 10*1024*1024;				// 最大ファイル容量  10MB
	private String	tableId 	= HybsSystem.TBL_MDL_KEY ;

	private static final String	FILE_BASE_URL 	= HybsSystem.sys( "FILE_URL" ) + "FileMdTemp/";

	private static enum TYPE { VIEW, UPLOAD };

	private static final String GE38_NAMES =
		" SYFILE,SYNAME,DESCRIPTION,HINAGATA,SAVE_FOLDER"
		+ ",FILE_URL,ICON_URL,ROLES,SU_LMT,DY_LMT,FGANGO,FGZIP"
		+ ",SIZE_LMT,SUFFIX_LMT";

	private static final String[] GE38_NAME_ARR = StringUtil.csv2Array( GE38_NAMES );

	private static final String GE38_SELECT =
		"select " + GE38_NAMES + " from GE38 where SYSTEM_ID=? and FGJ='1'";

	private static final String GE38_ORDER_BY = " order by SYFILE";

	private static final String GE39_NAMES =
		" UNIQ_ID,SEQNO,SUFFIX"
		+ ",SYFILE,SYNAME,SAVE_FOLDER"
		+ ",FILE_URL,ICON_URL,ROLES,SU_LMT,DY_LMT,FGANGO,FGZIP"
		+ ",FILE_NAME,CONTENTS,SIZE_FILE,DYFILESET,USRTOROKU,SETGUIID,FGKOSHIN,FGKOKAI"
		+ ",SRCHKEY1,SRCHKEY2,SRCHKEY3,SRCHKEY4,SRCHKEY5,SRCHKEY6,SRCHKEY7,SRCHKEY8,SRCHKEY9,SRCHKEY10";

	private static final String[] GE39_NAME_ARR = StringUtil.csv2Array( GE39_NAMES );

	private static final String GE39_SELECT =
		"select " + GE39_NAMES + " from GE39 where SYSTEM_ID=? and REF_KEY=? and FG_NEW ='1'and FGJ='1'";

	private static final String GE39_INSERT;
	static {
		StringBuilder tmp = new StringBuilder();
		tmp.append( "insert into GE39 (" ).append( GE39_NAMES ).append( ",FGJ,DYSET,PGSET,USRSET,DYUPD,PGUPD,USRUPD)" );
		tmp.append( "values (" );
		for( int i=0; i<GE39_NAME_ARR.length; i++ ) {
			if( i>0 ) { tmp.append( "," ); }
			tmp.append( "?" );
		}
		tmp.append( ",?,?,?,?,?,?,?)" );
		GE39_INSERT = tmp.toString();
	}

	private static final String GE39_ORDER_BY		= " order by SYFILE, UNIQ_ID";
	
	private static final String DEFAULT_SYFILE	= "**";

	private static final String[] names =
		new String[] { "SYNAME","FILELINK","UPDFILE","FILENAME","DISCRIPTION","HINAGATA","FILE_URL","SRCHKEY" };

	private static final String writableNames = "UPDFILE";

	private static final String SESSION_KEY = "h_fdMap";
	
	private static String SYSTEM_ID	= HybsSystem.sys( "SYSTEM_ID" );

	private TYPE	type				= TYPE.VIEW;
	private String	refKey				= null;
	private String	fileKeys			= null;
	private boolean	readOnly			= false;
	private int		freeCount			= 3;

	private Map<String,FileData>	fileDataMap = null;
	private DBTableModel			fileTable	= null;

	private boolean	quotCheck		= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.6.0.8 (2004/11/19) エラー発生時に確実にリリースされるように try finally 追加
	 * @og.rev 3.8.5.3 (2006/08/07) USER.LASTSQL へのSQL文の保存は、実行前に行っておきます。
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 4.0.0 (2005/08/31) useQuotCheck() によるＳＱＬインジェクション対策
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグへの対応
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 * @og.rev 5.1.9.0 (2010/08/01) TransactionTag 対応。上位に TransactionTag があれば、そこからConnection をもらう。
	 *
	 * @return	int
	 */
	@Override
	public int doAfterBody() {
		useQuotCheck( quotCheck );
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちのエンコード情報を取得する。
	 * @og.rev 3.5.2.0 (2003/10/20) scope 属性を追加。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 *
	 * @return	int
	 */
	@Override
	@SuppressWarnings("unchecked")
	public int doEndTag() {
		debugPrint();

		fileKeys = "AA,AAB,AABC";

		if( type == TYPE.VIEW ) {
			fileDataMap = makeFileDataMap();

			setSessionAttribute( HybsSystem.FILE_MANAGER_KEY, fileDataMap );

			fileTable = makeFileTable( fileDataMap );
			display( fileTable );
		}
		else if( type == TYPE.UPLOAD ) {
			String tmpFileURL = FILE_BASE_URL + getUser().getUserID() + "/" + System.currentTimeMillis();

			fileDataMap = (Map<String,FileData>)getSessionAttribute( HybsSystem.FILE_MANAGER_KEY );

			MultipartRequest multi = upload( fileDataMap, tmpFileURL );

			String[] syFileKeys = fileDataMap.keySet().toArray( new String[0] );

			String[] formNames = multi.getFilenames();
			for( String formName : formNames ) {
				String fileName = multi.getOriginalFileName( formName );
				if( fileName != null && fileName.length() > 0 ) {
					int row =  Integer.valueOf( formName.split( HybsSystem.JOINT_STRING )[1] );
					FileData fData = fileDataMap.get( syFileKeys[row] );
					System.out.println( "FILE=" + fileName + ":SYFILE=" + fData.getVal( "SYFILE" ) );
				}
				System.out.println(  );
			}

		}

//		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
//
//		try {
//			File dir = new File(directory);
//			if( ! dir.exists() && ! dir.mkdirs() ) {
//				String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
//				throw new HybsSystemException( errMsg );
//			}
//
//			// 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
//			MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename );
//			DBTableModel table = makeDBTable( multi );
//
//			// 3.5.2.0 (2003/10/20) scope 属性を追加。
//			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
//			if( ! commitTableObject( tableId, table ) ) {
//				jspPrint( "FileUploadTag Query処理が割り込まれました。DBTableModel は登録しません。" );
//				return (SKIP_PAGE);
//			}
//
//		} catch(IOException ex) {
//			String errMsg = "ファイルの取り扱い中にエラーが発生しました。"
//					+ toString() + HybsSystem.CR
//					+ "FileURL=" + fileURL + HybsSystem.CR
//					+ ex.getMessage();				// 5.1.8.0 (2010/07/01) errMsg 修正
//			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
//		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.1.1 (2003/03/06) columns を廃止
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
//		fileURL 		= HybsSystem.sys( "FILE_URL" );
//		maxPostSize		= 10*1024*1024;				// 最大ファイル容量  10MB
//		tableId 		= HybsSystem.TBL_MDL_KEY ;
//		filename		= null;				// 3.5.4.2 (2003/12/15)

		type			= TYPE.VIEW;
		fileKeys		= null;
		refKey			= null;
		readOnly		= false;
		freeCount		= 3;
		fileDataMap		= null;
		quotCheck		= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
	}

	private Map<String,FileData> makeFileDataMap() {
		Map<String,FileData> fMap = new LinkedHashMap<String,FileData>();
		
		String[] fileKeyArr = StringUtil.csv2Array( fileKeys );
		String syfileWhere = " and SYFILE in (";
		for( int i=0; i<fileKeyArr.length; i++ ) {
			if( i > 0 ) { syfileWhere += ','; }
			syfileWhere += "'" + fileKeyArr[i] + "'";
		}
		syfileWhere += ")";

		String ge38Stmt = GE38_SELECT + syfileWhere + GE38_ORDER_BY;
		String[] ge38Args = { SYSTEM_ID };
		String[][] ge38Vals = DBUtil.dbExecute( ge38Stmt, ge38Args, getApplicationInfo() );
		for( String[] ge38Val : ge38Vals ) {
			FileData fData = new FileData();
			fData.setFromGE38( ge38Val );
			fMap.put( ge38Val[0], fData );
		}

		String ge39Stmt = GE39_SELECT + syfileWhere + GE39_ORDER_BY;
		String[] ge39Args = { SYSTEM_ID, refKey };
		String[][] ge39Vals = DBUtil.dbExecute( ge39Stmt, ge39Args, getApplicationInfo() );
		for( String[] ge39Val : ge39Vals ) {
			FileData fData = fMap.get( ge39Val[0] );
			if( fData == null ) {
				fData = new FileData();
				fData.setFromGE39( ge39Val );
				fMap.put( ge39Val[0], fData );
			}
			else {
				fData.setFromGE39( ge39Val );
			}
		}

		for( int i=0; i<freeCount; i++ ) {
			FileData fData = new FileData();
			fMap.put( DEFAULT_SYFILE + i, fData );
		}

		return fMap;
	}

	private DBTableModel makeFileTable( Map<String,FileData> fMap ) {
		DBTableModel table = DBTableModelUtil.newDBTable();

		table.init( names.length );
		for( int i=0; i<names.length; i++ ) {
			DBColumn dbColumn = getDBColumn( names[i] );

			if( "UPDFILE".equals( names[i] ) ) {
				DBColumnConfig config = dbColumn.getConfig();
				config.setEditor( "UPLOAD" );
				dbColumn = new DBColumn( config );
			}

			table.setDBColumn( i,dbColumn );
		}

		for( Map.Entry<String,FileData> entry : fMap.entrySet() ) {
			FileData fData = entry.getValue();
			String[] vals = new String[names.length];
			for( int i=0; i<names.length; i++ ) {
				vals[i] = fData.getVal( names[i] );
				if( "FILE_LINK".equals( names[i] ) ) {
					
				}
				else if( "HINAGATA".equals( names[i] ) ) {
					TagBuffer tag = new TagBuffer( "a" );
					tag.add( "href", vals[i] );
					tag.setBody( vals[i] );
					vals[i] = tag.makeTag();
				}



			}
			table.addColumnValues( vals );
		}

		table.setDefaultRowWritable( true );
		return table;
	}

	private void display( DBTableModel table ) {
		ViewForm form = ViewFormFactory.newInstance( "HTMLTable" );
		form.init( table );
		form.setColumnWritable( writableNames );
		jspPrint( form.create() );
	}

	private MultipartRequest upload( final Map<String, FileData> fMap, final String fileURL ) {
		HttpServletRequest request = (HttpServletRequest) getRequest();
		String directory = HybsSystem.url2dir( fileURL );

		MultipartRequest multi = null;
		try {
			File dir = new File( directory );
			if( !dir.exists() && !dir.mkdirs() ) {
				String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
				throw new HybsSystemException( errMsg );
			}

			multi = new MultipartRequest( request, directory, maxPostSize, ENCODE, null );
		}
		catch( IOException ex ) {
			String errMsg = "ファイルの取り扱い中にエラーが発生しました。" + toString()
							+ HybsSystem.CR + "FileURL=" + fileURL
							+ HybsSystem.CR + ex.getMessage();
			throw new HybsSystemException( errMsg, ex );
		}
		return multi;
	}

	/**
	 * ファイルアップロードの実行結果を DBTableModel に記述します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.0.1.1 (2003/03/06) request 情報から{&#064;XXXX} で値が取り出せる様に修正。
	 * @og.rev 3.5.2.0 (2003/10/20) カラム名（KEY,VALUE）に ISFILE を追加
	 * @og.rev 3.5.6.5 (2004/08/09) MultipartRequest 変更に伴なう修正(Enum変更、元ファイル名取得)
	 * @og.rev 3.5.6.6 (2004/08/23) 上記変更時のバグ修正。
	 * @og.rev 3.5.6.6 (2004/08/23) 元ファイルのキーを、XXXX_ORG にします。
	 * @og.rev 4.0.0.0 (2007/10/12) テーブルモデルの登録方法を変更
	 *
	 * @param	 multi	   MultipartRequest オブジェクト
	 * @return	 DBTableModel
	 */
	private DBTableModel makeDBTable( final MultipartRequest multi ) {

		DBTableModel table = DBTableModelUtil.newDBTable();

//		table.init( names.length );
//
//		for( int i=0; i<names.length; i++ ) {
//			DBColumn dbColumn = getDBColumn( names[i] );
//			table.setDBColumn( i,dbColumn );
//		}
//
//		String[] values ;		// 4.0.0.0 (2007/10/12)
//
////		int numberOfRows = 0;
//		String[] files = multi.getFilenames();		// 3.6.0.0 (2004/09/17)
//		for( int i=0; i<files.length; i++ ) {
//			String name = files[i];
//			File fin = multi.getFile(name);
//			if( fin != null ) {
//				String val = multi.getFilesystemName(name);
////				String isFile = "1" ;
//
//				// "KEY","VALUE","ISFILE" の順にデータを作成します。
//				values = new String[] { name, val, "1" };
//				table.addColumnValues( values );
//				setRequestAttribute( name,val );
//
////				table.setValue( numberOfRows, "KEY", name );
////				table.setValue( numberOfRows, "VALUE", val );
////				table.setValue( numberOfRows, "ISFILE", isFile );
////				numberOfRows++ ;
////				setRequestAttribute( name,val );
//
//				String orgName = name + "_ORG" ;
//				val  = multi.getOriginalFileName(name);	// 注意：取得は、送信名
////				isFile = "2" ;
//
//				// "KEY","VALUE","ISFILE" の順にデータを作成します。
//				values = new String[] { orgName, val, "2" };
//				table.addColumnValues( values );
//				setRequestAttribute( orgName,val );
//
////				table.setValue( numberOfRows, "KEY", orgName );
////				table.setValue( numberOfRows, "VALUE", val );
////				table.setValue( numberOfRows, "ISFILE", isFile );
////				numberOfRows++ ;
////				setRequestAttribute( orgName,val );
//			}
//		}
//
//		// "KEY","VALUE","ISFILE" の順にデータを作成します。
//		values = new String[] { "directory", fileURL, "0" };
//		table.addColumnValues( values );
//		setRequestAttribute( "directory",fileURL );
//
////		table.setValue( numberOfRows, "KEY", "directory" );
////		table.setValue( numberOfRows, "VALUE", fileURL );
////		table.setValue( numberOfRows, "ISFILE", "0" );
////		numberOfRows++ ;
////		setRequestAttribute( "directory",fileURL );
//
//		String[] params = multi.getParameterNames();
//		for( int i=0; i<params.length; i++ ) {
//			String name = params[i];
//			String val = multi.getParameter(name);
//
//			// "KEY","VALUE","ISFILE" の順にデータを作成します。
//			values = new String[] { name, val, "0" };
//			table.addColumnValues( values );
//			setRequestAttribute( name,val );
//
////			table.setValue( numberOfRows, "KEY", name );
////			table.setValue( numberOfRows, "VALUE", value );
////			table.setValue( numberOfRows, "ISFILE", "0" );
////			numberOfRows ++ ;
////			setRequestAttribute( name,value );
//		}
////		table.resetModify();

		return table ;
	}

//	/**
//	 * 【TAG】ファイルをアップロードするディレクトリを指定します(初期値:FILE_URL)。
//	 *
//	 * @og.tag
//	 * この属性で指定されるディレクトリに、アップロードされたファイルをセーブします。
//	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
//	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
//	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
//	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
//	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
//	 *
//	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
//	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
//	 *
//	 * @param	url ファイルURL
//	 */
//	public void setFileURL( final String url ) {
//		String furl = nval( getRequestParameter( url ),null );
//		if( furl != null ) {
//			char ch = furl.charAt( furl.length()-1 );
//			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
//			fileURL = StringUtil.urlAppend( fileURL,furl );
//		}
//	}

	/**
	 * 【TAG】最大転送サイズ(Byte)を指定します(初期値:10485760)。
	 *
	 * @og.tag
	 * 最大転送サイズを指定します。初期値は、10*1024*1024 = 10MB です。
	 * 指定は、Byte 単位で指定します。
	 *
	 * @og.rev 3.0.1.1 (2003/03/06) maxPostSize の設定バグ修正。
	 *
	 * @param	maxPS 最大転送サイズ
	 */
	public void setMaxPostSize( final String maxPS ) {
		maxPostSize = nval( getRequestParameter( maxPS ),maxPostSize );
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) nvalメソッドを適用するように変更。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】(通常使いません)ファイルを作成するときのファイル名をセットします。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * これは、複数同時にアップロードファイル名を変更する時に使用できません。
	 * 通常、アップロードされたファイル名を指定する場合、アップロードするinput タグの
	 * name 属性に指定する名称 ＋ "_NEW" というリクエスト値を同時に送信すれば、
	 * 内部的に関連付けて、ファイル名を更新します。
	 * その場合、クライアントより指定したファイル名は、name属性＋"_ORG" という
	 * リクエスト値として取得することが可能になります。
	 * name属性 には、最終的に設定されたファイル名がセットされています。
	 * いずれの値も、{&#064;name属性＋"_ORG"} や、{&#064;name属性＋"_NEW"}として、
	 * アップロードのオリジナルと変更後のファイル名を取得することが出来ます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 *
	 * @param   filename ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】タブの一覧をどこから取得するかを指定します(初期値:AUTO)
	 *
	 * @og.tag
	 * タブの一覧をどこから取得するかを指定します。
	 * 現状の実装では、クエリを発行して一覧を生成する「DB」と、子タグである
	 * tabListタグを列挙してタブを定義する「TAG」が実装されています。
	 * 
	 * また、「AUTO」と指定した場合は、Body部分の内容に応じて自動的に判定されます。
	 * 初期値は、｢AUTO」です。
	 *
	 * @param	tp タブ一覧取得方法(「AUTO」)
	 */
	public void setType( final String tp ) {
		String typeStr = nval( getRequestParameter( tp ), null );
		try {
			type = TYPE.valueOf( typeStr );
		}
		catch ( IllegalArgumentException ex ) {
			StringBuilder errBuf = new StringBuilder( 100 );
			errBuf.append( "listType は" );
			for ( TYPE obj : TYPE.values() ) {
				errBuf.append( ',' );
				errBuf.append( obj.name() );
			}
			errBuf.append( "から選んでください。" );
			throw new HybsSystemException( errBuf.toString(), ex );
		}
	}

	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_SQL_INJECTION_CHECK)。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_SQL_INJECTION_CHECK です。
	 *
	 * @og.rev 4.0.0 (2005/08/31) 新規追加
	 *
	 * @param   flag クォーティションチェックする ("true")／しない (それ以外)
	 */
	public void setRefKey( final String key ) {
		refKey = nval( getRequestParameter( key ),refKey );
	}
	
	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_SQL_INJECTION_CHECK)。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_SQL_INJECTION_CHECK です。
	 *
	 * @og.rev 4.0.0 (2005/08/31) 新規追加
	 *
	 * @param   flag クォーティションチェックする ("true")／しない (それ以外)
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "upload" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "names"		,names		)
				.println( "ENCODE"		,ENCODE		)
//				.println( "fileURL" 	,fileURL	)
				.println( "filename"    ,filename	)
				.println( "maxPostSize" ,maxPostSize)
				.println( "tableId" 	,tableId	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}

	private static class FileData {

		private Map<String,String> data = new HashMap<String,String>();

		public void setFromGE38( final String[] vals ) {
			for( int i=0; i<GE38_NAME_ARR.length; i++ ) {
				data.put( GE38_NAME_ARR[i], vals[i] );
			}
		}

		public void setFromGE39( final String[] vals ) {
			for( int i=0; i<GE39_NAME_ARR.length; i++ ) {
				data.put( GE39_NAME_ARR[i], vals[i] );
			}
		}

		/**
		 * @return the syFile
		 */
		public String getVal( final String key ) {
			return data.get( key );
		}
	}
}
