/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;

/**
 * テキストフィールドの入力を補助するポップアップを作成します。
 *
 * 通常のテキストフィールドに組み込むケースと、一覧表に組み込むケースに対応しています。
 * 基本的には、columnMarker タグと同じ使い方ですが、フィールドが書き込み許可時に表示させる必要が
 * あるため、実際には、viewMarker の isRendere="false" で使用するケースが多いと思います。
 * なお、ポップアップ専用なので href="index.jsp" target="CONTENTS" がデフォルトで設定されています。
 *
 * @og.formSample
 * ●形式：&lt;og:popup gamenId="…" href="…" keys="…,…" vals="…,…" rtnKeys="…,…" /&gt;
 * ●body：なし
 *
 * ●使用例
 *    ・普通のポップアップ
 *        &lt;og:popup gamenId="GE0002" href="index.jsp" target="CONTENTS"
 *                    keys="AAA,BBB,CCC" vals="{&#064;AAA},{&#064;B},ABC"
 *                    rtnKeys="DDD,EEE" /&gt;
 *
 *    ・QUERYの入力フィールドに組み込む
 *      ①keys、vals、rtnKeys がセットされる場合
 *      　　　　セットされる keys に対して vals の値をポップアップ画面に渡し、ポップアップ画面の戻り値を rtnKeys にセットされます。 
 *        &lt;og:column name="AAA" value="{&#064;AAA}" &gt;
 *            &lt;og:popup gamenId="GE0002"
 *                        keys="AAA,BBB,CCC" vals="{&#064;AAA},{&#064;B},ABC"
 *                        rtnKeys="DDD,EEE" /&gt;
 *        &lt;/og:column&gt;
 *      ②keys、vals、rtnKeys がセットされない場合  
 *      　　　　 ポップアップ画面にリクエスト変数 XNAME に外側のカラム名、XVALUE にカラムの入力値、rtnKeys にカラム名を自動的にセットされます。
 *        &lt;og:column name="AAA" value="{&#064;AAA}" &gt;
 *            &lt;og:popup gamenId="GE0002" /&gt;
 *        &lt;/og:column&gt;
 *
 *    ・RESULTの入力フィールドに組み込む(viewMarker 経由)
 *      gamenId を指定しなければこの画面へのURLを作成します。
 *      このURLは、その画面の index.jsp に直接ジャンプするためのURLです。
 *        &lt;og:viewMarker command="{&#064;command}" isRenderer="false"&gt;
 *            &lt;og:popup gamenId="GE0002" column="ABC" onMark="[ABC]" markList="1|2|3"
 *                        keys="AAA,BBB,CCC" vals="[AAA],[BBB].value,{&#064;CCC}"
 *                        rtnKeys="DDD,EEE" /&gt;
 *        &lt;/og:viewMarker&gt;
 *
 *    ・ポップアップ側のJSPファイル(view 部分)
 *
 * @og.group 画面部品
 * @og.rev 3.8.6.1 (2006/10/20) 新規作成
 *
 * @version  0.9.0	2000/10/17
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class PopupTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2006/10/20)

	private static final String DEF_BODY  = "▼" ;
	private static final String DEF_STYLE = "width:20px;text-align:center;" ;

	private String		column		= null;
	private String		gamenId		= null;
	private String		href		= "index.jsp";
	private String		target		= "CONTENTS";
	private String		onMark		= "true";	// true または 1
	private String		markList	= "true|TRUE|1";	// true または 1
	private String		height		= null;
	private String		width		= null;
	private String		imgsrc		= null;
	private String[]	keys		= null;
	private String[]	vals		= null;
	private String[]	rtnKeys		= null;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	private static final String CMD_NEW	 = "NEW" ;		// 5.1.7.0 (2010/06/01)
	private String		command		= CMD_NEW;			// 5.1.7.0 (2010/06/01)

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		makeGamenLink();

		// GAMENID属性のＵＲＬへの自動セットは行わない。
		ViewMarkerTag viewMark = (ViewMarkerTag)findAncestorWithClass( this,ViewMarkerTag.class );
		if( viewMark != null ) {
			set( "onMark"  ,onMark );
			set( "markList",markList );
			set( "body"    ,makePopup( true ) );

			if( column != null && column.length() > 0 ) {
				set( "column"  ,column );	// 4.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
				viewMark.addAttribute( getAttributes() );
			}
		}
		else {
			if( markList.indexOf( onMark ) >= 0 ) {
				jspPrint( makePopup( false ) );
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * 使用しません。
	 *
	 * @return null
	 */
	protected String makeTag() { return null; }

	/**
	 * タグリブオブジェクトをリリースします。<br />
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。<br />
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		column		= null;
		gamenId		= null;
		href		= "index.jsp";
		target		= "CONTENTS";
		onMark		= "true";			// true または 1
		markList	= "true|TRUE|1";	// true または 1
		keys		= null;
		vals		= null;
		rtnKeys		= null;
		height		= null;
		width		= null;
		imgsrc		= null;
		command		= CMD_NEW;			// 5.1.7.0 (2010/06/01)
	}

	/**
	 * リンクを作成します。
	 *
     * @og.rev 4.2.4.0 (2008/07/01) keys,vals,rtnKeysが設定されていない場合、外側のog:columnタグのname値を設定するように変更
	 * @og.rev 5.1.7.0 (2010/06/01) 引数にcommandを追加     *
	 *
	 * @param	useKey boolean (true:KEY付き/false:KEY無し)
	 * @return	リンクタグ文字列
	 */
	private String makePopup( final boolean useKey ) {
		StringBuilder rtn = new StringBuilder();
		rtn.append( "ogPopup( '" ).append( get( "href" ) ).append( "'," );
		rtn.append( height ).append( "," ).append( width ).append( "," );

		String columnName = null; //4.2.4.0 2008/07/01
		if( keys != null && keys.length > 0 ) {
			rtn.append( "new Array(" );
			addQuote( rtn, keys[0], false );
			rtn.append( "," );
			addQuote( rtn, vals[0], false );
			for( int i = 1; i < keys.length; i++ ) {
				rtn.append( "," );
				addQuote( rtn, keys[i], false );
				rtn.append( "," );
				addQuote( rtn, vals[i], false );
			}
			rtn.append( ")," );
		}
		else {
			ColumnTag outerColumn = (ColumnTag) findAncestorWithClass( this, ColumnTag.class );
			columnName = ( outerColumn == null ) ? null : outerColumn.getName();
			if( columnName != null ) {
				rtn.append( "new Array('XNAME','" );
				rtn.append( columnName );
				rtn.append( "','XVALUE'," );
				rtn.append( columnName );
				rtn.append( ".value" );
				rtn.append( ")," );
			}
			else {
				rtn.append( "null," );
			}
		}
		rtn.append( "" );

		if( rtnKeys != null && rtnKeys.length > 0 ) {
			rtn.append( "this,new Array(" );
			addQuote( rtn, rtnKeys[0], useKey );
			for( int i = 1; i < rtnKeys.length; i++ ) {
				rtn.append( "," );
				addQuote( rtn, rtnKeys[i], useKey );
			}
			rtn.append( ")" );
		}
		else if( columnName != null ) {
			rtn.append( "this,new Array('" );
			rtn.append( columnName );
			rtn.append( "')" );
		}
		else {
			rtn.append( "null,null" );
		}

		// 5.1.7.0 (2010/06/01) 引数にcommandを追加
//		rtn.append( ");" );
		rtn.append( ",'" ).append( command ).append( "');" );

		if( imgsrc == null ) {
			set( "type"   , "button" );
			set( "body"   , nval( getMsglbl(),DEF_BODY) );
			set( "onClick", rtn.toString() );
			set( "style"  , nval( get( "style" ),DEF_STYLE) );
			return XHTMLTag.button( getAttributes() ) ;
		}
		else {
			set( "src"    , imgsrc );
			set( "title"  , getMsglbl() );
			set( "onClick", rtn.toString() );
			set( "style"  , get( "style" ) );
			return XHTMLTag.img( getAttributes() ) ;
		}
	}

	/**
	 * 引数 val の設定値を設定する簡易メソッド
	 *
	 * @param	buf StringBuilder
	 * @param	val String
	 * @param	useKey boolean (true:KEY付き/false:KEY無し)
	 * @return	StringBuilder
	 */
	private StringBuilder addQuote( final StringBuilder buf, final String val, final boolean useKey ) {
		if( useKey ) {
			if( val != null && val.endsWith( ".value" ) ) {
				buf.append( val.substring( 0,val.length()-6 ) ).append( "__{I}.value" );
			}
			else {
				buf.append( "'" ).append( val ).append( "__{I}'" );
			}
		}
		else {
			if( val != null && val.endsWith( ".value" ) ) {
				if( val.charAt(0) == '[' ) {
					buf.append( val.substring( 1,val.length()-7 ) ).append( "__{I}.value" );
				}
				else {
					buf.append( val );
				}
			}
			else {
				buf.append( "'" ).append( val ).append( "'" );
			}
		}
		return buf ;
	}

	/**
	 * 画面ＩＤとhref から、指定のＵＲＬを作成します。
	 */
	private void makeGamenLink() {
		if( gamenId == null || gamenId.length() == 0 ) { return ; }

		GUIInfo guiInfo = getGUIInfo( gamenId );		// 4.0.0 (2005/01/31)
		if( guiInfo == null ) { return ; }	// 見つからない場合は、アクセス不可

		String address = guiInfo.getRealAddress( get( "href" ) );
		String url = getRequestParameter( address );
		set( "href",url );

		if( get( "title" ) == null ) {
			set( "title","To:" + guiInfo.getLabel() );
		}
	}

	/**
	 * 【TAG】この要素をリンクの対象とするための名前を指定します。
	 *
	 * @og.tag <p>この要素をリンクの対象とするための名前を指定します。</p>
	 *
	 * @param	name String
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【TAG】リンクをつけたいカラムを指定します。
	 *
	 * @og.tag
	 * <p>このカラム名のTableModelに対して、 ViewLink タグが存在する場合は,
	 * リンクを張ります。</p>
	 *
	 * @param	clm String
	 */
	public void setColumn( final String clm ) {
		column = nval( getRequestParameter( clm ),column );
	}

	/**
	 * 【TAG】gamenId 属性を登録します。
	 *
	 * @og.tag
	 * <p>gamenId 属性は、画面IDをキーに、実アドレスを求めるのに使用します。<br />
	 * 画面IDが指定された場合は、実アドレスに変換する。指定されない場合は、
	 * href 属性をそのままリンク情報を作成します。<br />
	 * 画面IDを指定する場合の href は、JSPファイル名（例えば、index.jsp 等）です。</p>
	 *
	 * @param   id gamenId 属性
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 【TAG】ポップアップ先のURLを指定します(初期値:index.jsp)。
	 *
	 * @og.tag
	 * 通常、gamenId を指定している場合は、それ以降(index.jsp , result.jsp など)を指定します。
	 * 通常のリンクの場合、GAMENID/href というアドレスに合成されます。( GE0001/result.jsp など ）
	 * (正確には、画面IDではなく、対応する画面ADDRESSを使用します。)<br />
	 * ポップアップ専用のリンクなので、初期値は、index.jsp です。
	 *
	 * @param	href String
	 */
	public void setHref( final String href ) {
		set( "href",nval( getRequestParameter( href ),this.href ) );
	}

	/**
	 * 【TAG】ポップアップ先の文字コードセットを指定します。
	 *
	 * @og.tag <p>ポップアップ先の文字コードセットを指定します。</p>
	 *
	 * @param	charset String
	 */
	public void setCharset( final String charset ) {
		set( "charset",getRequestParameter( charset ) );
	}

	/**
	 * 【TAG】ポップアップ先の表示ターゲットを指定します(初期値:CONTENTS)。
	 *
	 * @og.tag <p>ポップアップ先の表示ターゲットを指定します。</p>
	 *
	 * @param	target String
	 */
	public void setTarget( final String target ) {
		set( "target",nval( getRequestParameter( target ),this.target ) );
	}

	/**
	 * 【TAG】ポップアップ先に渡す入力キー配列をCSV形式で複数指定します。
	 *
	 * @og.tag <p>ポップアップ先に渡す入力キー配列を指定します。</p>
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key String
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】ポップアップ先に渡す値配列をCSV形式で複数指定します。
	 *
	 * @og.tag <p>ポップアップ先に渡す値配列を指定します。</p>
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val String
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】ポップアップ先より受け取るキー配列をCSV形式で複数指定します。
	 *
	 * @og.tag <p>ポップアップ先より受け取るキー配列を指定します。</p>
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key String
	 */
	public void setRtnKeys( final String key ) {
		rtnKeys = getCSVParameter( key );
	}

	/**
	 * 【TAG】リンクを張る(true or 1)/張らない(false or 0)の設定を指定します(初期値:true)。
	 *
	 * @og.tag
	 * <p>リンクを張る場合は、"true"(または "1") リンクを張らない
	 * 場合は, "true以外"(または "0")をセットします。<br />
	 * デフォルトは、 "true"(リンクを張る)です。</p>
	 * <p>さらに、[カラム名] で、動的にカラムの値で、リンクを張る、張らないを
	 * 選ぶ事が可能になります。値は、"true"(または "1") で、マークします。
	 * ただし、単独 linkタグ時は使用できません。</p>
	 * <p>追記 3.5.2.0 (2003/10/20)：markList属性に、処理対象文字列郡を指定できます。
	 * これにより、リンクを処理するかどうかの判断が、true,1 以外にも使用できるようになりました。</p>
	 *
	 * @param	flag リンクの設定 "true"(リンクを張る)/その他(リンクを張らない)
	 */
	public void setOnMark( final String flag ) {
		onMark = nval( getRequestParameter( flag ),onMark );
	}

	/**
	 * 【TAG】処理するマークを、"|"区切りの文字列で指定します。
	 *
	 * @og.tag
	 * <p>markListで与えられた文字列に、onMark文字列（大文字/小文字の区別あり）が
	 * 含まれていれば、処理します。 <br />
	 * 例えば、"A","B","C" という文字列が、onMark で指定された
	 * 場合に処理するようにしたい場合は、"A|B|C" をセットします。<br />
	 * markList には、[カラム名]指定が可能です。(ただし、単独 linkタグ時は除く)<br />
	 * デフォルトは、 "true|TRUE|1"です。</p>
	 *
	 * @param	list 処理するマーク(indexOf による含む/含まない判定)
	 */
	public void setMarkList( final String list ) {
		markList = nval( getRequestParameter( list ),markList );
	}

	/**
	 * 【TAG】ダイアログの縦幅をピクセル数で指定します。
	 *
	 * @og.tag
	 * <p>縦幅を指定します。単位も含めて入力してください。</p>
	 * 設定しない場合は、ogPopup JavaScript の内部初期値が採用されます。
	 *
	 * @param   ht String
	 */
	public void setHeight( final String ht ) {
		height = nval( getRequestParameter( ht ),height );
	}

	/**
	 * 【TAG】ダイアログの横幅をピクセル数で指定します。
	 *
	 * @og.tag
	 * <p>横幅を指定します。単位も含めて入力してください。</p>
	 * 設定しない場合は、ogPopup JavaScript の内部初期値が採用されます。
	 *
	 * @param   wh String
	 */
	public void setWidth( final String wh ) {
		width = nval( getRequestParameter( wh ),width );
	}

	/**
	 * 【TAG】表示させる画像のURLを指定します。
	 *
	 * @og.tag
	 * この属性を設定すると、自動的に イメージリンクでの onClick属性で
	 * ポップアップの起動画面を作成します。
	 * この属性が設定されていない場合は、button での起動になります。
	 *
	 * @param   src String
	 */
	public void setSrc( final String src ) {
		imgsrc = nval( getRequestParameter( src ),imgsrc );
	}

	/**
	 * 【TAG】(通常使いません)処理の実行を指定する command を設定できます(初期値:NEW)。
	 *
	 * @og.tag
	 * ここで指定されたcommandは、ポップアップ画面の引数として渡されます。
	 * 初期値は、NEW です。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @param	cmd コマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "column"		,column		)
				.println( "gamenId"		,gamenId	)
				.println( "href"		,href		)
				.println( "target"		,target		)
				.println( "onMark"		,onMark		)
				.println( "markList"	,markList	)
				.println( "height"		,height		)
				.println( "width"		,width		)
				.println( "keys"		,keys		)
				.println( "vals"		,vals		)
				.println( "rtnKeys"		,rtnKeys	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString()
			+ HybsSystem.CR
			+ super.toString() ;
	}
}
