/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

/**
 * テーブルレイアウトで column タグ等の存在しない空欄箇所に挿入して、
 * テーブルの行列の個数を合わせるタグです。
 *
 * 通常の空欄は、ラベルと値という２つの項目を別々のカラムに登録するように
 * &lt;td&gt;&lt;/td&gt;タグが、２つ続いた形式をしていますが、一般に、
 * 個数は表示上の都合で任意の個数の&lt;td&gt;タグを使用したいケースが
 * あります。
 * その場合、このSpaceTag を使っておけば,大量の作成済み画面を修正しなくても
 * このクラスを修正するだけで,すべての空欄のデザインがキープできます。
 *
 * @og.formSample
 * ●形式：&lt;og:space /&gt;
 * ●body：なし
 *
 * ●使用例
 *      通常の、ラベル部、フィールド部が空の td タグを生成します。
 *      &lt;og:space /&gt;
 *
 *      td タグの大きさを指定することができます。
 *      &lt;og:space
 *         labelWidth = &quot;ラベル部分（前半部分）の大きさ(width 属性)を指定します。&quot;
 *         valueWidth = &quot;値部分（後半部分）の大きさ(width 属性)を指定します。&quot;
 *      /&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SpaceTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private String labelWidth = null ;
	private String valueWidth = null ;
	private String width      = null ;		// 5.1.9.0 (2010/08/01) 追加
	private String colspan    = null ;		// 5.1.9.0 (2010/08/01) 追加

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		jspPrint( makeTag() );
		return( SKIP_BODY );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.9.0 (2010/08/01) width , colspan 属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		labelWidth = null ;
		valueWidth = null ;
		width      = null ;		// 5.1.9.0 (2010/08/01) 追加
		colspan    = null ;		// 5.1.9.0 (2010/08/01) 追加
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) width , colspan 属性追加
	 * @return  入力用のテキストフィールドタグ
	 */
	private String makeTag() {
		StringBuilder rtn = new StringBuilder();

		// width 指定
		if( width != null && width.length() > 0  ) {
			// width が指定されると、強制的に colspan を設定する。
			if( colspan == null || colspan.length() == 0  ) {
				colspan = "2";
			}
			rtn.append( "<td width=\"" );
			rtn.append( width );
			rtn.append( "\" colspan=\"" );
			rtn.append( colspan );
			rtn.append( "\"></td>" );
		}
		// すでに、width は未指定なので、colspan のみ出力
		else if( colspan != null && colspan.length() > 0  ) {
			rtn.append( "<td colspan=\"" );
			rtn.append( colspan );
			rtn.append( "\"></td>" );
		}
		// width も colspan も未指定の場合のみ、labelWidth、valueWidth を出力
		else {
			if( labelWidth == null || labelWidth.length() == 0  ) {
				rtn.append( "<td></td>" );
			}
			else {
				rtn.append( "<td width=\"" );
				rtn.append( labelWidth );
				rtn.append( "\"></td>" );
			}

			if( valueWidth == null || valueWidth.length() == 0  ) {
				rtn.append( "<td></td>" );
			}
			else {
				rtn.append( "<td width=\"" );
				rtn.append( valueWidth );
				rtn.append( "\"></td>" );
			}
		}
		return rtn.toString() ;
	}

	/**
	 * 【TAG】全体の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 通常の space タグは、td ２個分のタグを出力しますが、width 属性を
	 * 指定した場合は、colspan="2" をセット下上で、width 属性に設定されます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 *
	 * @param   width 属性
	 */
	public void setWidth( final String width ) {
		this.width = getRequestParameter( width );
	}

	/**
	 * 【TAG】全体の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 属性は td タグの colspan 属性に設定されます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 *
	 * @param   colspan 属性
	 */
	public void setColspan( final String colspan ) {
		this.colspan = getRequestParameter( colspan );
	}

	/**
	 * 【TAG】ラベル部分（前半部分）の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 属性は td タグの width 属性に設定されます。
	 *
	 * @param   width 属性
	 */
	public void setLabelWidth( final String width ) {
		labelWidth = getRequestParameter( width );
	}

	/**
	 * 【TAG】値部分（後半部分）の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 属性は td タグの width 属性に設定されます。
	 *
	 * @param   width 属性
	 */
	public void setValueWidth( final String width ) {
		valueWidth = getRequestParameter( width );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "width"			,width		)
				.println( "colspan"			,colspan	)
				.println( "labelWidth"		,labelWidth	)
				.println( "valueWidth"		,valueWidth	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
