/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.SystemInstaller;

/**
 * システムのインストールを行うためのタグです。
 *
 * システムインストール用のアーカイブ(ZIP)形式を指定することで、以下の設定を自動で行います。
 * ①webapps以下へのコンテキストファイルの展開T
 * ②Tomcatのコンテキスト設定
 * ③各種DB環境のインストール
 * 
 * このアーカイブは、ルートディレクトリにコンテキストパスのドキュメントベースが配置されている必要があります。
 * 
 * [アーカイブの構成]
 * xx.zip - gf - db - ...               (DB環境インストール用のスクリプト)
 *             - filetemp - ...         (一時ファイルの保存場所 ※中身は空)
 *             - jsp - ...              (画面JSP)
 *             - log - ...              (ログファイルの保存場所 ※中身は空)
 *             - WEB-INF -  ...         (接続先情報、openGionのjarファイル)
 *                       - [CONTEXT].xml(コンテキスト設定ファイル)
 *　
 * ※WEB-INF直下の[CONTEXT].xmlは、Tomcatのconfディレクトリ以下に、コンテキスト設定ファイルとして、コピーされます。
 *   なお、この[CONTEXT].xmlがアーカイブに含まれていない場合、インストールは行われません。
 *
 * @og.formSample
 * ●形式：&lt;og:sysInstall fileURL="…" insFile="…" /&gt;
 * ●body：なし
 *
 * ●使用例
 *    ・Calc(ods)ファイルをPDFに変換
 *        &lt;og:sysInstall insFile="context.zip" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class SysInstallTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.9.0 (2010/08/01)";

	private static final long serialVersionUID = 5190 ;

	private String	insFile		= null;
	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		String directory = HybsSystem.url2dir( fileURL );
		File buildFile = new File( StringUtil.urlAppend( directory,insFile ) );

		Connection conn = ConnectionFactory.connection( null, null );
		PrintWriter out = new PrintWriter( pageContext.getOut() );
		SystemInstaller installer = new SystemInstaller( conn, out );

		try {
			installer.install( buildFile );
		}
		catch( SQLException ex ) {
			String msg = "インストール中に、DB処理でエラーが発生しました。";
			throw new HybsSystemException( msg, ex );
		}

		return( EVAL_PAGE );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		insFile		= null;
		fileURL 	= HybsSystem.sys( "FILE_URL" );
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します(初期値:システムパラメータのFILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下を、使用します。
	 *
	 * @param	url ファイルURL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】インストールファイルを指定します。
	 *
	 * @og.tag
	 * インストールファイルを指定します。
	 * 各コンテキストのインストール用アーカイブを指定する必要があります。
	 *
	 * @param	fname 出力ファイル名
	 */
	public void setInsFile( final String fname ) {
		insFile = nval( getRequestParameter( fname ),insFile );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "insFile"		,insFile		)
				.println( "fileURL"		,fileURL	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
