/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.io.TableWriter;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.ServletResponse ;
import javax.servlet.http.HttpServletResponse ;
import javax.servlet.jsp.JspWriter ;

import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;
import java.util.List;
import java.util.ArrayList;
import java.util.Locale;
import java.io.File;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * DBTableModel オブジェクトを指定のファイルに出力するタグです。
 *
 * データ（DBTableModel)と、コントローラ（WriteTableTagForm)を与えて、
 * 外部からコントロールすることで、各種形式で データ（DBTableModel)を表示できます。
 * writeTableタグ に対して、コマンドを与えることにより、内部のコントローラの
 * 実装に対応した、 形式でデータを作成します。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:writeTable
 *         fileURL   = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename  = "{&#064;filename}"   保存ファイル名
 *         encode    = "UnicodeLittle"      保存ファイルエンコード名
 *     /&gt;
 *
 * ●body：あり(writeTableParam)
 *
 * ●使用例
 *     &lt;og:writeTable ･････ &gt;
 *         &lt;og:writeTableParam
 *             key  = "Tablename"  value="GE12"
 *         /&gt;
 *         &lt;og:writeTableParam
 *             key  = "First"             First:最初に登録
 *         &gt;
 *              &lt;jsp:text&gt;
 *                      insert into GE12bk
 *                         select * from GE12
 *                         where SYSTEM_ID='**'
 *              &lt;/jsp:text&gt;
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "First"             First:の２番目に登録
 *         &gt;
 *              &lt;jsp:text&gt;delete from GE12 where SYSTEM_ID='**' and KBSAKU='0'&lt;/jsp:text&gt;
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "Last"              Last:最後に登録
 *         &gt;
 *              &lt;jsp:text&gt;update GE12 set XXXX='YYYY' where SYSTEM_ID='**' and KBSAKU='0'&lt;/jsp:text&gt;
 *         &lt;/og:writeTableParam
 *     &lt;/og:writeTableParam
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteTableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String	separator		= TableWriter.TAB_SEPARATOR;	 // 項目区切り文字
	private String	headerSequence	= null;			// ヘッダー項目の並び順
	private String	fileURL 		= BASE_URL;
	private String	filename		= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
	private String	refFileURL		= BASE_URL;		// 3.5.4.3 (2004/01/05)
	private String	refFilename 	= null;			// 3.5.4.3 (2004/01/05)
	private String	fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
	private String	fontPoint	 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
	private String	encode			= HybsSystem.sys( "FILE_ENCODE" );	// ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private String	writerClass 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_CLASS" );		// 3.8.5.3 (2006/08/07)
	private boolean	fileAppend		= false;		// ファイルをAPPENDモードで出力するか
	private boolean	direct			= false;
	private boolean	zip 			= false;
	private String  disposition     = "attachment";   // 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment

	private transient DBTableModel table	  = null;
	private String	tableId 		= HybsSystem.TBL_MDL_KEY ;
	private String	sheetName		= "Sheet1" ;	// 3.5.4.2 (2003/12/15)
	private String	refSheetName	= null ;		// 3.5.4.3 (2004/01/05)
	private String	directory		= null ;		// 3.8.6.0 (2006/08/23)

	private transient List<HybsEntry>	param	= null ;		// 4.0.0 (2005/02/28)

	private boolean	useNumber		= true;			// 3.7.0.2 (2005/02/14)
	// 4.0.0 (2005/12/31) 外部よりカラム列（カンマ区切り）を指定できるようにする。
	private String		columns		= null;
	// 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか
	private boolean	useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.1.3.1 (2002/12/13) ContentType を、x-msexcel から vnd.ms-excel に変更した。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが、ZipOutputStream に変更。
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を、 PrintWriter に戻す。
	 * @og.rev 3.6.0.2 (2004/10/04) EXCELダイレクト出力時にファイルエンドまで出力されない不具合対策
	 * @og.rev 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
	 * @og.rev 3.8.6.0 (2006/08/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 4.3.4.3 (2008/12/22) Excel出力の判別方法を前方一致に変更
	 * @og.rev 5.1.6.0 (2010/05/01) 画面帳票作成機能対応
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		table = (DBTableModel)getObject( tableId );
		if( table != null && table.getRowCount() > 0 ) {

			directory = HybsSystem.url2dir( fileURL );
			File dir = new File(directory);
			if( ! dir.exists() && ! dir.mkdirs() ) {
				String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
				throw new HybsSystemException( errMsg );
			}

			PrintWriter pw = null;
			ByteArrayOutputStream bt = null ;
			try {
				if( direct ) {
					if( zip ) {
						bt = new ByteArrayOutputStream();
						pw = new PrintWriter( bt );
						create( pw ) ;
						pw.flush();				// 3.6.0.2 (2004/10/04)
						ServletResponse response = pageContext.getResponse() ;
						((HttpServletResponse)response).addHeader( "Content-Encoding","gzip" );

						GZIPOutputStream gzip = new GZIPOutputStream(bt);
						gzip.finish() ;
						response.setContentLength( (bt).size() );

						JspWriter out = pageContext.getOut();
						out.clear();
						out.print( bt );
						out.flush();			// 3.6.0.2 (2004/10/04)
					}
					else {
						bt = new ByteArrayOutputStream();
						pw = new PrintWriter( bt );
						create( pw ) ;
						pw.flush();				// 3.6.0.2 (2004/10/04)
						ServletResponse response = pageContext.getResponse() ;
						response.setContentType( "application/vnd.ms-excel; charset=" + encode );
	//					response.setContentType( "application/x-msexcel; charset=" + encode );
	//					response.setContentType( "text/comma-separated-values ; charset=" + encode );
	//					((HttpServletResponse)response).setHeader( "contentType","application/x-msexcel; charset=Shift_JIS" );
	//					((HttpServletResponse)response).setHeader( "Content-Disposition","attachment; filename=\"" + filename + "\"" );
	//					((HttpServletResponse)response).setHeader( "Content-Disposition","inline; filename=\"" + filename + "\"" );
						// 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
						((HttpServletResponse)response).setHeader( "Content-Disposition",disposition + "; filename=\"" + filename + "\"" );

						JspWriter out = pageContext.getOut();
						out.clear();
						out.print( bt );
						out.flush();			// 3.6.0.2 (2004/10/04)
					}
				}
				else {
				// 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが、ZipOutputStream に変更。
					if( zip ) {
						bt = new ByteArrayOutputStream();
						pw = new PrintWriter( bt );
						create( pw ) ;
						pw.flush();				// 3.6.0.2 (2004/10/04)

						ZipOutputStream gzip = null;
						try {
							gzip = new ZipOutputStream(
											new FileOutputStream(
												StringUtil.urlAppend( directory,filename + ".zip" )));
							gzip.putNextEntry( new ZipEntry( filename ) );
							(bt).writeTo( gzip );
							gzip.finish() ;
						}
						finally {
							Closer.ioClose( gzip );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
						}
					}
					else {
						// if( "Excel".equalsIgnoreCase( writerClass ) ) { 
//						if( ( writerClass.toUpperCase(Locale.JAPAN) ).startsWith("EXCEL") ){ // 4.3.4.3 (2008/12/22)
						if( ( writerClass.toUpperCase(Locale.JAPAN) ).startsWith("EXCEL") // 4.3.4.3 (2008/12/22)
						||  ( writerClass.toUpperCase(Locale.JAPAN) ).startsWith("CALC") ) { // 5.1.6.0 (2010/05/01) 
							create( null );
						}
						else {
							pw = getPrintWriter();
							create( pw );
							pw.flush();				// 3.6.0.2 (2004/10/04)
						}
					}
				}
			}
			catch( IOException ex ) {
				String errMsg = "テーブル出力時に、UOエラーが発生しました。" + toString();
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
			finally {
				Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				Closer.ioClose( bt );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.5.4.2 (2003/12/15) EXCELのシート名を指定できるように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参照ファイルのURL,filenameを追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) writerClass 属性の初期値をシステムリソースより取得します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontName 属性を追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontPoint 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/08/23) directory 属性を追加します。
	 * @og.rev 4.0.0 (2005/02/28) XMLファイルにセットする、param を追加。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか useRenderer 属性追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		separator		= TableWriter.TAB_SEPARATOR;   // 項目区切り文字
		headerSequence	= null;   // ヘッダー項目の並び順
		fileURL 		= BASE_URL;
		filename		= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
		refFileURL		= BASE_URL;		// 3.5.4.3 (2004/01/05)
		refFilename 	= null;			// 3.5.4.3 (2004/01/05)
		fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
		fontPoint		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
		encode			= HybsSystem.sys( "FILE_ENCODE" );	 // ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		writerClass 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_CLASS" );		// 3.8.5.3 (2006/08/07)
		fileAppend		= false;   // ファイルをAPPENDモードで出力するか
		direct			= false;
		zip 			= false;
		table			= null;
		tableId 		= HybsSystem.TBL_MDL_KEY ;
		disposition 	= "attachment";		// 3.8.0.9 (2005/10/17) 初期値変更 inline ⇒ attachment
		sheetName		= "Sheet1";		// 3.5.4.2 (2003/12/15)
		refSheetName	= null;			// 3.5.4.2 (2003/12/15)
		param			= null ;		// 4.0.0 (2005/01/31)
		useNumber		= true;			// 3.7.0.2 (2005/02/14)
		directory		= null;			// 3.8.6.0 (2006/08/23)
		columns			= null;			// 4.0.0 (2005/12/31)
		useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01) 
	}

	/**
	 * TableWriter の実オブジェクトを生成して，<del>PrintWriter</del>OutputStream に書き込みます。
	 *
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.4.2 (2003/12/15) lang 引数も table と同時に設定します。
	 * @og.rev 3.5.4.2 (2003/12/15) TableWriter のサブクラス名変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参考ファイルを指定するように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を、 PrintWriter に戻す。
	 * @og.rev 3.5.4.5 (2004/01/23) 文字エンコードを渡すように変更する。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 3.8.5.3 (2006/08/07) EXCEL時に fontName と fontPoint 属性をセットします。
	 * @og.rev 4.0.0 (2005/01/31) キーの指定を、TableWriter. から、TableWriter_ に変更します。
	 * @og.rev 4.0.0 (2005/12/31) 外部よりカラム列（カンマ区切り）を指定できるようにする。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか設定 useRenderer 属性追加
	 *
	 * @param   out PrintWriter
	 */
	private void create( final PrintWriter out )  {
		// 5.2.1.0 (2010/10/01) writerClass で Renderer系の場合は、useRendererフラグに変更する。
		if( "ExcelRenderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Excel" ;
			useRenderer = true;
		}
		else if( "Renderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Default" ;
			useRenderer = true;
		}

		String className = HybsSystem.sys( "TableWriter_" + writerClass ) ;		// 4.0.0 (2005/01/31)
		TableWriter writer = (TableWriter)HybsSystem.newInstance( className );	// 3.5.5.3 (2004/04/09)

		writer.setDBTableModel( table,getLanguage() );
		writer.setSeparator( separator );
		writer.setHeaderSequence( headerSequence );
		writer.setEncode( encode );					// 3.5.4.5 (2004/01/23)
		writer.setAppend( fileAppend );				// 3.5.4.2 (2003/12/15)
		if( param != null ) { writer.setParam( param ); }	// 4.0.0 (2005/01/31)
		writer.setUseNumber( useNumber );			// 3.7.0.2 (2005/02/14)
	//	if( columns == null ) {
	//		columns = (String)getSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//		removeSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//	}
		writer.setColumns( columns );				// 4.0.0 (2005/12/31)
		writer.setUseRenderer( useRenderer );		// 5.2.1.0 (2010/10/01)
		if( writer.isExcel() ) {					// 3.5.4.3 (2004/01/05)
			writer.setSheetName( sheetName );
			writer.setFilename( StringUtil.urlAppend( directory , filename ) );
			if( refFileURL != null && refFilename != null ) {
				writer.setRefFilename( StringUtil.urlAppend( HybsSystem.url2dir(refFileURL) , refFilename ) );
			}
			writer.setRefSheetName( refSheetName );
			writer.setFontName( nval( fontName,null ) );			// 3.8.5.3 (2006/08/07)
			if( fontPoint != null && fontPoint.length() > 0 ) {
				writer.setFontPoint( Short.parseShort( fontPoint ) );	// 3.8.5.3 (2006/08/07)
			}
			writer.writeDBTable();
		}
		else {
			writer.writeDBTable( out );
		}
	}

	/**
	 * <del>PrintWriter</del> OutputStream を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した <del>PrintWriter</del>OutputStream を作成します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.5.4.1 (2003/12/01) PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.5.9 (2004/06/07) FileUtil.getBufferedReader を使用
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	 *
	 * @return InputStream
	 */
	private PrintWriter getPrintWriter() {
		if( filename == null ) {
			String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}

		// ※ 注意 StringUtil.urlAppend を組み込んでいる意図が不明。一旦削除していますが、注意
	 	// 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	//	out = FileUtil.getPrintWriter( StringUtil.urlAppend( directory,filename ),fileAppend,encode);
		PrintWriter out = FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );

		return out ;
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，（内部ポインタを書き換えた）DBTableModel オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param   tableId sessionに登録する時の ID
	 */
	public void setTableId( final String tableId ) {
		this.tableId = nval( getRequestParameter( tableId ), this.tableId );
	}

	/**
	 * 【TAG】可変長ファイルを作成するときの項目区切り文字をセットします(初期値:TableWriter.TAB_SEPARATOR)。
	 *
	 * @og.tag 可変長ファイルを作成するときの項目区切り文字をセットします。
	 *
	 * @param   separator 項目区切り文字
	 */
	public void setSeparator( final String separator ) {
		this.separator = nval( getRequestParameter( separator ),this.separator );
	}

	/**
	 * 【TAG】DBTableModelの出力順(LNSCD など)をセットします。
	 *
	 * @og.tag
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @param   hs 出力順 (LNSCD など）
	 */
	public void setHeaderSequence( final String hs ) {
		String[] headers = getRequestParameterValues( hs );
		if( headers != null ) {
			StringBuilder buf = new StringBuilder();
			for( int i=0; i<headers.length; i++ ) {
				if( headers[i] != null ) { buf.append( headers[i] ); }
			}
			headerSequence = buf.toString();
		}
	}

	/**
	 * 【TAG】EXCEL雛型参照ファイルのディレクトリ名をセットします。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url EXCEL雛型参照ファイルのディレクトリ名
	 */
	public void setRefFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			refFileURL = StringUtil.urlAppend( refFileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイル保存先ディレクトリ名を指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を、makeFileURL に移動。
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 保存先ディレクトリ名
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします(初期値:システムパラメータ の FILE_FILENAME)。
	 *
	 * @og.tag ファイルを作成するときのファイル名をセットします。
	 *
	 * @param   filename ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】EXCEL雛型参考ファイル名をセットします。
	 *
	 * @og.tag EXCEL雛型参考ファイル名をセットします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename EXCEL雛型参考ファイル名
	 */
	public void setRefFilename( final String filename ) {
		refFilename = nval( getRequestParameter( filename ),refFilename );
	}

	/**
	 * 【TAG】ファイルを作成するときのファイルエンコーディング名をセットします(初期値:システムパラメータ の FILE_ENCODE)。
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 *
	 * @param   enc ファイルエンコーディング名
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】実際に書き出すクラス名の略称(TableWriter_**** クラスの ****)を指定します(初期値:Default)。
	 *
	 * @og.tag
	 * 実際に書き出すクラス名（の略称）をセットします。
	 * これは、org.opengion.hayabusa.io 以下の TableWriter_**** クラスの **** を
	 * 与えます。これらは、TableWriter インターフェースを継承したサブクラスです。
	 * 属性クラス定義の {@link org.opengion.hayabusa.io.TableWriter TableWriter} を参照願います。
	 *
	 * @param   writerClass クラス名（の略称）
	 * @see		org.opengion.hayabusa.io.TableWriter  TableWriterのサブクラス
	 */
	public void setWriterClass( final String writerClass ) {
		this.writerClass = nval( getRequestParameter( writerClass ),this.writerClass );
	}

	/**
	 * 【TAG】追加モードで書き込むかどうか(true/false)を指定します(初期値:false[通常モード])。
	 *
	 * @og.tag DBTableModelのファイルを、追加モードで書き込むかどうかをセットします。
	 *
	 * @param   flag 追加モード（true)／新規（その他)
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

	/**
	 * 【廃止】結果をダイレクトにEXCEL起動するかどうか(true/false)を指定します(初期値:false[ファイル])。
	 *
	 * @og.tag 結果をダイレクトに EXCEL ファイルとして出力するかどうかをセットします。
	 *
	 * @param  flag ダイレクト（true)／ ファイル（その他)
	 * @deprecated 結果をダイレクトするとエンコード等でおかしな動きをする可能性がある。
	 */
	@Deprecated public void setDirect( final String flag ) {
		direct = nval( getRequestParameter( flag ),direct );
	}

	/**
	 * 【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、圧縮しないです。
	 *
	 * @param  flag ZIPで圧縮する（true)／ しない（その他)
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】Content-Disposition (埋め込む:inline,ダイアログを起動:attachment)を指定します(初期値:attachment)。
	 *
	 * @og.tag
	 * Content-Disposition を指定することで、ブラウザにアプリケーションデータを
	 * 埋め込むのか（inline)、セーブ/オープンのダイアログを起動するのか（attachment）
	 * 指定します。
	 * この属性が有効なのは、direct="true" で、zip="false"(初期値)のときのみです。
	 * 値は、inline 、attachment が指定できます。
	 * デフォルトは、attachment です。
	 * 注意：動作検証しましたが、inline と attachment で動作に差が現れませんでした。
	 *
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment
	 *
	 * @param  pos Content-Disposition (inline/attachment)
	 */
	public void setDisposition( final String pos ) {
		disposition = nval( getRequestParameter( pos ),disposition );
		if( !"inline".equals( disposition ) && !"attachment".equals( disposition ) ) {
			String errMsg = "disposition には、inline か attachment 以外は指定できません。"
						+ "disposition=[" + disposition + "]" ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】EXCELファイルを書き出すときのシート名を設定します。
	 *
	 * @og.tag
	 * EXCELファイルを書き出す時に、シート名を指定します。これにより、複数の形式の
	 * 異なるデータを順次書き出したり(appendモードを併用)することや、シートを指定して
	 * 新規にEXCELを作成することが可能になります。
	 * 初期値は、Sheet1 です。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheet EXCELファイルのシート名
	 */
	public void setSheetName( final String sheet ) {
		sheetName = nval( getRequestParameter( sheet ),sheetName );
	}

	/**
	 * 【TAG】EXCEL雛型参考ファイルのシート名を設定します。
	 *
	 * @og.tag
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * 初期値は、null（第一シート） です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheet EXCEL雛型参考ファイルのシート名
	 */
	public void setRefSheetName( final String sheet ) {
		refSheetName = nval( getRequestParameter( sheet ),refSheetName );
	}

	/**
	 * 【TAG】EXCEL出力時のデフォルトフォント名を設定します。(初期値：TABLE_WRITER_DEFAULT_FONT_NAME)
	 *
	 * @og.tag
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。<br />
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   name String
	 */
	public void setFontName( final String name ) {
		fontName = nval( getRequestParameter( name ),fontName );
	}

	/**
	 * 【TAG】EXCEL出力時のデフォルトフォントポイント数を設定します。<br />
	 *
	 * @og.tag
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point short
	 */
	public void setFontPoint( final String point ) {
		fontPoint = nval( getRequestParameter( point ),fontPoint );
	}

	/**
	 * 【TAG】行番号情報を、出力する(true)/しない(false)を指定します。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号を出力します。
	 * これは、#NAME 属性を使用する場合には、必ず出力する必要があります。
	 * (#NAME 属性は、読み取り時には、必須です。)
	 * この、先頭の行番号が不要な場合（つまり、他のシステムへのデータ出力、
	 * このシステムでは、#NAME 属性が出力されないため、読み込みできません。)
	 * この行番号を出力しないようにできます。
	 * なお、EXCEL 出力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true（出力する） です。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 新規追加
	 *
	 * @param   useNo 行番号情報を、出力する(true)/しない(false)を指定
	 */
	public void setUseNumber( final String useNo ) {
		useNumber = nval( getRequestParameter( useNo ),useNumber );
	}

	/**
	 * 【TAG】書き込み先ファイルのカラム列を、外部（タグ）より指定します。
	 *
	 * @og.tag
	 * この指定により、書き込むカラムを限定的に使用することが可能です。
	 * カラム名は、検索したDBTableModel上に含まれる名前で指定する必要があります。
	 * 別名で書き込みたい場合は、予めその名前で検索しておく必要があります。
	 *
	 * @og.rev 4.0.0 (2005/12/31) 新規作成
	 *
	 * @param   clms 書き込み先ファイルのカラム列（カンマ区切り文字）
	 */
	public void setColumns( final String clms ) {
		columns = nval( getRequestParameter( clms ),columns );
	}

	/**
	 * 【TAG】書込処理でコードリソースのラベル変換を行うかどうかを指定します。(初期値:USE_TABLE_WRITER_RENDERER)
	 *
	 * @og.tag
	 * コードリソースをそのままの値で出力すると、数字や記号になり何が書かれているのか
	 * 不明になります。
	 * これは、コードリソースをラベルに変換して出力するかどうかを指定します。
	 * 当然、コードはユニークですが、ラベルはユニークになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には、リスクが発生します。
	 * また、TableReader 系で読み込む場合にも、ラベルからコードを求める逆変換を行うように、
	 * setUseRenderer メソッドで指定する必要があります。
	 *
	 * 従来は、TableWriter 系に、TableWriter_Renderer 系のクラスを作って対応していましたが、
	 * このメソッドの属性値のフラグで、制御します。
	 * 初期値は、システム定数の USE_TABLE_WRITER_RENDERER で指定できます。
	 * 
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param  flag コードリソースのラベル変換を行うかどうか
	 */
	public void setUseRenderer( final String flag ) {
		useRenderer = nval( getRequestParameter( flag ),useRenderer );
	}

	/**
	 * WriteTable オブジェクトに渡すパラメータオブジェクトをセットします。
	 *
	 * @og.tag
	 * WriteTableParamTag クラスよりセットされます。
	 *
	 * @og.rev 4.0.0 (2005/02/28) 新規追加
	 *
	 * @param   entry HybsEntry
	 */
	protected void addParam( final HybsEntry entry ) {
		if( param == null ) { param = new ArrayList<HybsEntry>(); }
		param.add( entry );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "BASE_URL"		,BASE_URL		)
				.println( "separator"		,separator		)
				.println( "headerSequence"	,headerSequence	)
				.println( "fileURL"			,fileURL 		)
				.println( "filename"		,filename		)
				.println( "refFileURL"		,refFileURL		)
				.println( "refFilename"		,refFilename 	)
				.println( "encode"			,encode			)
				.println( "writerClass"		,writerClass 	)
				.println( "fileAppend"		,fileAppend		)
				.println( "direct"			,direct			)
				.println( "zip"				,zip 			)
				.println( "disposition"		,disposition	)
				.println( "tableId"			,tableId 		)
				.println( "sheetName"		,sheetName		)
				.println( "refSheetName"	,refSheetName	)
				.println( "useNumber"		,useNumber		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
