/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 文字列の開始日付属性を規定する半角文字列を扱う為の、カラム属性を定義します。
 *
 * YYYYMM01 または、00000XXXX , 9999XXXX を許可します。
 * また、入力データとして, YYYYMM 形式 は、許可しません。あらかじめ,
 * valueSet( String ) でデータを変換しておく必要があります。
 * ただし、日付としての整合性チェックは行いません。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YM01 extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final String DEF_VALUE = "00000000" ;	// データのデフォルト値

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 4.0.0 (2005/01/31) type 廃止
	 */
	public DBType_YM01() {
		super( DEF_VALUE );
	}

	/**
	 * NATIVEの型の識別コードを返します。<br />
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規作成
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType() {
		return NativeType.CALENDAR;
	}

	/**
	 * 半角スペースで固定長（半角換算の数）に変換した文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		if( value != null && value.length() >= sizeX ) {
			return value.substring( 0, sizeX );
		}

		String errMsg = "指定文字が不正です。value=[" + value + "] length=[" + sizeX + "]";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合（A,B,C など）は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。（桁上がりもあり）
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列（""）の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * valueSet( String ) する以前の文字については,整合性は保証していません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 月の加算ロジックの誤り修正。(roll ⇒ add)
	 *
	 * @param   value  String
	 * @return  String引数の文字列を＋１した文字列
	 */
	public String valueAdd( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return value; }
		Calendar cal = Calendar.getInstance();

		int year  = Integer.parseInt( value.substring( 0,4 ) );
		int month = Integer.parseInt( value.substring( 4,6 ) ) - 1;
		int date  = Integer.parseInt( value.substring( 6,8 ) );

		cal.set( year,month,date );
		cal.add( Calendar.MONTH, 1 );		// 月を１ヶ月加算します。

		DateFormat formatter = new SimpleDateFormat( "yyyyMMdd",Locale.JAPAN );
		return formatter.format( cal.getTime() );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 4.3.3.4 (2008/12/01) 0000 or 9999 時のバグ修正
	 *
	 * @param   value String （一般に編集データとして登録されたデータ）
	 * @return  修正後の文字列（一般にデータベースに登録するデータ）
	 */
	public String valueSet( final String value ) {
		if( value == null ) { return null; }
		String rtn = StringUtil.rTrim( value );			// 3.3.3.1 (2003/07/18)
		if( rtn.length() >= 6 ) {
			rtn = rtn.substring( 0,6 );
//			if(      rtn.startsWith( "0000" ) ) { rtn = "000001"; }
//			else if( rtn.startsWith( "9999" ) ) { rtn = "999912"; }
			if(      rtn.startsWith( "0000" ) ) { rtn = "00000101"; }
			else if( rtn.startsWith( "9999" ) ) { rtn = "99991231"; }
			else                                { rtn += "01"; }
		}
		return rtn;
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 * valueSet( String ) する以前の文字については,整合性は保証していません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 *
	 * @param   key String
	 * @param   value String
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   param    dbType パラメータ
	 * @return  エラー内容
	 */
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String param ) {
		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( value.length() != len ) {
			// 文字列の長さが指定の長さと異なります。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
		}

		StringBuilder val = new StringBuilder();
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt( i );
			if( ('0' > ch || '9' < ch) ) {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				val.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		String check = DBTypeCheckUtil.matcheCheck( value,param );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
