/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.html;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.io.Writer;
import java.io.IOException;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Attributes;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * ViewForm インターフェース の実装Abstractクラスです。
 * これを,共通のスーパークラスとして 各種表示フォーム（例：HTML表示等）に使います。
 *
 * このクラス は、setter/getterメソッドのデフォルト実装を提供しています。
 * 各種表示フォームに対応したサブクラス上で, create() をオーバーライドして下さい。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public abstract class AbstractViewForm implements ViewForm {
	// 5.2.1.0 (2010/10/01)
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;	// 表示フィールドの大きさ

	// 3.8.0.3 (2005/07/15) 新しいヘッダー固定用のDIV要素を分ける。
	private static final String LAYER_ST0 = "" ;
// 4.2.0.0 (2008/03/18) ビューサイズ指定のため削除
//	private static final String LAYER_ST2 =
//									"<div id=\"divPos\">"
//								+	"<div id=\"outer\" >"
//								+	"<div id=\"layer\" onscroll=\"SetScrollHeader(this);\">"
//								+	"<div id=\"divHide\">" ;
	private static final String LAYER_END0 = "" ;
	private static final String LAYER_END2 = "<a href=\"#top\" name=\"h_end\" id=\"h_end\" ></a></div></div></div></div>" ;

	// 5.1.8.0 (2010/07/01) groupClass のタイプを定義します。
	private static enum CLM_GRP { KIGO , CHAR };

	private String			name			= "";		// メニューの名前
	private DBTableModel	table			= null;
	private DBColumn[]		dbColumn		= null;
	private boolean[]		clmWritable 	= null;
	private boolean[]		writeCtrl	 	= null;		// 3.8.0.9 (2005/10/17)
	private boolean[]		clmDisplay		= null;
	private boolean[]		clmGroup		= null;		// 3.8.5.0 (2006/03/20)
	private String			groupClass		= "";		// 5.1.8.0 (2010/07/01)
	private CLM_GRP			groupType		= null;		// 5.1.8.0 (2010/07/01)
	private boolean[]		sortKeys		= null;		// 3.6.0.0 (2004/09/17)
	private boolean[]		useEventCols	= null;		// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	private boolean			useSorterKeys	= false;	// 3.6.0.0 (2004/09/17) 内部変数
	private String			sorterQuery		= "";		// 3.6.0.0 (2004/09/17) 内部変数
	private String			viewFormID		= null;
	private int 			startNo 		= 0 ;
	private int 			pageSize		= HybsSystem.sysInt( "HTML_PAGESIZE" ) ;
	private boolean 		rowWritableFlag = false;	// １カラム目が writable か？
	private ViewMarker		viewLink		= null;
	private ViewMarker		viewMarker		= null;
	private ViewMarker		editMarker		= null;		// 3.8.6.1 (2006/10/20)
	private String			selectedType	= "checkbox";
	private String			numberType		= "sequence" ;		// 3.5.1.0 (2003/10/03) 新規作成
	private int				numberTypeClm	= -1;				// 3.5.1.0 (2003/10/03) 新規作成
	private String[]		numberTypeData	= null;				// 3.5.1.0 (2003/10/03) 新規作成
	private String			optTypeAttri	= null;
	private boolean 		noMessage		= false;
	private int 			backLinkCount	= 0 ;
	private int				headerSkipCount	= 0;		// 0:通常ヘッダ、n:n回ごとに現れる
	private boolean			skip			= ViewForm.DEFAULT_SKIP;	// 3.5.3.1 (2003/10/31)
	private int				useCheckControl = 0 ;		// 3.7.0.1 (2005/01/31)
	private boolean			useTableSorter	= HybsSystem.sysBool( "VIEW_USE_TABLE_SORTER" ) ;	// 3.5.4.7 (2004/02/06)
	private boolean			numberDisplay	= true ;	// 3.5.5.0 (2004/03/12)
	private int				scrollRowNo		= -1;		// 3.7.0.3 (2005/03/01)

	// 3.5.4.6 (2004/01/30) ヘッダー文字列を定義しておきます。
	private static final String NO_HEADER = "No" ;

	// 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを受け取ります。
	private Map<String,String> viewParam = null;		// 4.3.3.6 (2008/11/15) Generics警告対応

	private int columnCount	= 0 ;	// 3.5.5.7 (2004/05/10)
	private int rowCount	= 0 ;	// 4.0.0 (2006/01/06)

	// 3.5.6.2 (2004/07/05) ゼブラ模様の指定を、ViewForm としてサポート
	// 0(ゼブラなし)、-1（ワーニング)、-2以下（エラー）、1(ゼブラ）、2以上(行数まとめ）
	private int bgColorCycle   = 1 ;
	// 行ごとに色を変更する時の、デフォルトクラス属性
	private static final String BG_COLOR_ROW0 = " class=\"row_0\"";
	// 行ごとに色を変更する時の、切り替え後クラス属性
	private static final String BG_COLOR_ROW1 = " class=\"row_1\"";

	// 行ごとに色を変更する時の、切り替え後クラス属性
	private static final String BG_COLOR_ROWSEL = " class=\"row_sel\"";		// 3.7.0.3 (2005/03/01)

	// 警告時の行ごとに色を変更する時の、デフォルトクラス属性
	private static final String BG_WARNING_COLOR_ROW0 = " class=\"row_0 row_warning\"";
	// 警告時の行ごとに色を変更する時の、切り替え後クラス属性
	private static final String BG_WARNING_COLOR_ROW1 = " class=\"row_1 row_warning\"";

	// エラー時の行ごとに色を変更する時の、デフォルトクラス属性
	private static final String BG_ERROR_COLOR_ROW0 = " class=\"row_0 row_error\"";
	// エラー時の行ごとに色を変更する時の、切り替え後クラス属性
	private static final String BG_ERROR_COLOR_ROW1 = " class=\"row_1 row_error\"";

	private String color_row0 = BG_COLOR_ROW0 ;
	private String color_row1 = BG_COLOR_ROW1 ;

	// 5.1.8.0 (2010/07/01) 行に対して、動的にクラス属性を付与するカラム名を指定します。
	private int bgColorClsClmNo  = -1;

	// 3.5.6.4 (2004/07/16)
	private boolean	useScrollBar = HybsSystem.sysBool( "VIEW_USE_SCROLLBAR" ) ;

	// 3.6.0.0 (2004/09/17)
	private boolean	firstChecked = false ;

	// 3.7.1.1 (2005/05/31) SEL_ROW機能使用時に、BG_COLOR_ROWSEL の使用 有/無を指定します。
	private boolean	useSelRowColor = false ;

	// 4.0.0 (2007/04/16) tableタグを出力するときに付与します。
	private String	clazz			= "viewTable";

	// 4.0.0.0 (2007/11/27) ResourceManagerを設定します。(クロス集計で使用)
	private ResourceManager resourceManager = null;
	
	// 4.2.0.0 (2008/03/03) ビューの幅と高さを指定できるようにします。
	private String	height		= null;
	private String	width		= null;
	
	// 4.3.1.0 (2008/09/08)
	private boolean skipNoEdit	= false;

	// 4.3.3.0 (2008/10/01) 画面遷移なしモード時に各行に出力する行番号及び改廃Cのキーを定義します。
	private static final String hiddenRowKey =		"rid";
	private static final String hiddenCdkhKey =		"kh";

	private boolean	noTransition= false;

	// 5.1.7.0 (2010/06/01) ViewFormのキャッシュ復元を画面ID単位に行う
	private String gamenId		= null;

	// 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラムを指定します。
	private boolean[]		clmBulkSet 	= null;

	/**
	 * 初期化します。
	 * ここでは、内部で使用されているキャッシュをクリアし、
	 * 新しいモデル（DBTableModel）と言語（lang） を元に内部データを再構築します。
	 * なお、テーブルモデルに対してViewFormで変更を加える場合は、変更処理を行った後に
	 * このメソッドを実行するようにして下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 * @og.rev 3.5.6.3 (2004/07/12) 呼ばれたら、必ず初期化するように修正（元に戻す）
	 * @og.rev 3.6.0.0 (2004/09/17) sortKeys 追加
	 * @og.rev 3.8.0.9 (2005/10/17) writeCtrl 追加
	 * @og.rev 3.8.5.0 (2006/03/20) clmGroup 追加
	 * @og.rev 4.0.0 (2006/01/06) rowCount 追加
	 * @og.rev 4.0.1.0 (2007/12/13) コメントの追加
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.2.0.0 (2010/09/01) 2回連続登録時に選択行にスクロールしないバグを修正。
	 * @og.rev 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラム(clmBulkSet) 追加
	 *
	 * @param	table DBTableModel
	 */
	public void init( final DBTableModel table ) {
		this.table = table;
		if( table != null ) {
			rowCount	= table.getRowCount();			// 4.0.0 (2006/01/06)
			columnCount = table.getColumnCount();
			clmWritable = new boolean[ columnCount ];
			writeCtrl	= new boolean[ columnCount ];	// 3.8.0.9 (2005/10/17)
			clmDisplay	= new boolean[ columnCount ];
			clmGroup	= new boolean[ columnCount ];	// 3.8.5.0 (2006/03/20)
			sortKeys	= new boolean[ columnCount ];	// 3.6.0.0 (2004/09/17)
			dbColumn	= new DBColumn[ columnCount ];
			useEventCols= new boolean[ columnCount ];	// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し 
			clmBulkSet  = new boolean[ columnCount ];	// 5.2.1.0 (2010/10/01)
			Arrays.fill( clmWritable,DEFAULT_CLM_WRITABLE );	// カラムの書込み許可
			Arrays.fill( writeCtrl,false );						// 書き込み制御の許可
			Arrays.fill( clmDisplay ,true );					// カラムの表示許可
			Arrays.fill( clmGroup ,false );						// 3.8.5.0 (2006/03/20) カラムのグループ化
			Arrays.fill( sortKeys ,false );						// すべてリンクしないに設定する。
			Arrays.fill( useEventCols, false );					// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
			Arrays.fill( clmBulkSet, false );					// 5.2.1.0 (2010/10/01)
			for( int i=0; i<columnCount; i++ ) {
				dbColumn[i] = table.getDBColumn( i );
			}

			// １カラム目（最初のカラム＝配列０番目）が writable か？
			rowWritableFlag = "WRITABLE".equalsIgnoreCase( getColumnName(0) );

			// 5.2.0.0 (2010/09/01) 2回連続登録時に選択行にスクロールしないバグを修正。
			scrollRowNo = -1;
			firstChecked = false;
		}
	}

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return	DBTableModel テーブルモデル
	 */
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * ViewForm の識別IDをセットします。
	 * これは、ViewFormFactory でプールする場合の識別キーになります。
	 * プールに戻すときに自分自身に この識別IDを使用します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.6.2 (2004/07/05) メソッド名がまぎらわしい為、変更します。
	 *
	 * @param	viewFormID 識別ID
	 */
	public void setId( final String viewFormID ) {
		this.viewFormID = viewFormID;
	}

	/**
	 * ViewForm の識別IDを返します。
	 * これは、ViewFormFactory でプールする場合の識別キーになります。
	 * プールに戻すときに自分自身に この識別IDを使用します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) メソッド名がまぎらわしい為、変更します。
	 *
	 * @return	識別ID
	 */
	public String getId() {
		return viewFormID;
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @return	DBTableModel から作成された HTML文字列
	 */
	public String create() {
		return create( 0, rowCount );
	}

	/**
	 * DBTableModel から View文字列を作成して、Writer にセットします。
	 * 処理内容は、create( int , int ) と同じですが、中間の文字列(StringBuilder)
	 * を作成せずに、直接、Writer に書き出します。
	 * よって、データ作成途中でエラーが発生しても、すでにいくつかのデータは
	 * クライアントに返されています。
	 *
	 * @og.rev 5.0.0.1 (2009/08/15) 直接出力用の Writer 引数追加
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 * @param  wrt		  直接登録用の Writer
	 */
	public void create( final int startNo, final int pageSize, final Writer wrt ) throws IOException {
		String errMsg = "このメソッドは、直接登録用の Writer のビューでのみ使用できます。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 内容をクリア（初期化）します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.1.0 (2003/10/03) displayNumber 廃止。numberType 新規追加。
	 * @og.rev 3.5.2.0 (2003/10/20) headerSkipCount属性を追加
	 * @og.rev 3.5.3.1 (2003/10/31) skip属性を追加
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl属性を追加
	 * @og.rev 3.5.4.3 (2004/01/05) viewFormID属性を削除（初期化しない）
	 * @og.rev 3.5.4.7 (2004/02/06) useTableSorter属性を追加
	 * @og.rev 3.5.4.7 (2004/02/06) columnMaxSize は使用されていないので削除します。
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 * @og.rev 3.5.5.0 (2004/03/12) numberType 属性の "delete" 値追加に伴なう、numberDisplay 変数の追加。
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の追加。
	 * @og.rev 3.5.6.4 (2004/07/16) useScrollBar 変数の追加。
	 * @og.rev 3.6.0.0 (2004/09/17) sortKeys , firstChecked , useSorterKeys , sorterQuery 変数の追加。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール変更( boolean ⇒ int )
	 * @og.rev 3.7.1.1 (2005/05/31) useSelRowColor 変数の追加。
	 * @og.rev 3.8.0.3 (2005/07/15) scrollBarType 変数の追加
	 * @og.rev 3.8.0.9 (2005/10/17) writeCtrl 変数の追加
	 * @og.rev 3.8.5.0 (2006/03/20) clmGroup 変数の追加
	 * @og.rev 3.8.6.1 (2006/10/20) editMarker 変数の追加
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) groupClass , groupType 変数の追加
	 * @og.rev 5.1.8.0 (2010/07/01) bgColorClsClmNo 属性を追加します。
	 * @og.rev 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラム(clmBulkSet) 追加
	 */
	public void clear() {
		name			= "";				// メニューの名前
		table			= null;
		dbColumn		= null;
		clmWritable 	= null;
		writeCtrl	 	= null;			// 3.8.0.9 (2005/10/17)
		clmDisplay		= null;
		clmGroup		= null;			// 3.8.5.0 (2006/03/20)
		groupClass		= "";			// 5.1.8.0 (2010/07/01)
		groupType		= null;			// 5.1.8.0 (2010/07/01)
		sortKeys		= null;			// 3.6.0.0 (2004/09/17)
		useSorterKeys	= false;		// 3.6.0.0 (2004/09/17)
		sorterQuery		= "";			// 3.6.0.0 (2004/09/17)
		startNo 		= 0 ;
		pageSize		= HybsSystem.sysInt( "HTML_PAGESIZE" ) ;
		rowWritableFlag = false;	// １カラム目が writable か？
		viewLink		= null;
		viewMarker		= null;
		editMarker		= null;		// 3.8.6.1 (2006/10/20)
		backLinkCount	= 0;
		selectedType	= "checkbox";
		numberType		= "sequence";
		numberTypeClm	= -1;
		numberTypeData	= null;
		headerSkipCount	= 0;	// 0:通常ヘッダ、n:n回ごとに現れる
		skip			= ViewForm.DEFAULT_SKIP;	// 3.5.3.1 (2003/10/31)
		useCheckControl = 0 ;		// 3.7.0.1 (2005/01/31)
		viewParam		= null;		// 3.5.4.8 (2004/02/23)
		numberDisplay	= true ;	// 3.5.5.0 (2004/03/12)
		columnCount		= 0;		// 3.5.5.7 (2004/05/10)
		bgColorCycle	= 1;		// 3.5.6.2 (2004/07/05)
		bgColorClsClmNo	= -1;		// 5.1.8.0 (2010/07/01)
		color_row0		= BG_COLOR_ROW0 ;		// 3.5.6.2 (2004/07/05)
		color_row1		= BG_COLOR_ROW1 ;		// 3.5.6.2 (2004/07/05)
		useScrollBar	= HybsSystem.sysBool( "VIEW_USE_SCROLLBAR" ) ;	// 3.5.6.4 (2004/07/16)
		firstChecked	= false;
		useSelRowColor	= false;	// 3.7.1.1 (2005/05/31)
		height			= null;		// 4.2.0.0 (2008/03/18)
		width			= null;		// 4.2.0.0 (2008/03/18)
		skipNoEdit		= false;	// 4.3.2.0 (2008/09/10)
		useEventCols	= null;		// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
		clmBulkSet		= null;		// 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラム(clmBulkSet) 追加
	}

	/**
	 * テーブルのヘッダーの前に、そのデータの表示範囲を示す文字列を作成します。
	 * [開始No - 終了No]/[件数] です。
	 *
	 * @og.rev 2.0.0.2 (2002/09/24) [1 - 4]/[4] 等のメッセージ出力後の &lt;br&gt; 出力を中止。
	 * @og.rev 4.3.6.0 (2008/04/01) [1 - 4]/[4]のメッセージをJavaScriptで入れ替えるように変更
	 * @og.rev 4.3.8.0 (2009/08/01) 同メッセージ複数対応及びheadでコントロール可能にする
	 * 
	 * @param  stNo    表示開始位置
	 * @param  pgSize  表示件数
	 * @return	テーブルのヘッダータグ文字列
	 */
	protected String getCountForm( final int stNo, final int pgSize ) {
		if( noMessage ) { return ""; }

		StringBuilder out = new StringBuilder();

		int lstNo = ( stNo+pgSize > rowCount ) ? rowCount : stNo+pgSize ;
		// 4.3.6.0 (2009/04/01) 件数表示の方法変更
		// out.append( "[" ).append( stNo+1 ).append( " - " ).append( lstNo ).append( "]" );
		// out.append( "/[" ).append( rowCount ).append( "]" );
		// out.append( "<div id=\"originalRowCountMessage\">" ); // 4.3.8.0 (2009/08/01) idからclassに変更
		out.append( "<div class=\"orgRowCntMsg\">" );
		out.append( "[" ).append( stNo+1 ).append( " - " ).append( lstNo ).append( "]" );
		out.append( "/[" ).append( rowCount ).append( "]" );
		out.append( "</div>" );
		// 4.3.8.0 (2009/08/01) scriptはhead読み込み方式に変更
		// out.append( "<script language=\"javascript\">" );
		// out.append( "$(function(){$('#rowCountMessage').replaceWith($('#originalRowCountMessage'));})");
		// out.append( "</script>" );
		
		return out.toString() ;
	}

	/**
	 * カラムのラベル名を返します。
	 * カラムの項目名に対して,見える形の文字列を返します。
	 * 一般には,リソースバンドルと組合せて,各国ロケール毎にラベルを
	 * 切替えます。
	 *
	 * @param	column int
	 * @return	カラムのラベル名
	 */
	protected String getColumnLabel( final int column ) {
		return dbColumn[column].getLabel();
	}

	/**
	 * カラム名を返します。
	 * データベースで検索したときのカラムの項目名を返します。
	 *
	 * @param	column int
	 * @return	カラム名
	 */
	protected String getColumnName( final int column ) {
		return dbColumn[column].getName();
	}

	/**
	 * row行，colum列 のデータの値を返します。
	 *
	 * @param	row int
	 * @param	column int
	 * @return	row行，colum列 のデータの値
	 */
	protected String getValue( final int row,final int column ) {
		return table.getValue( row,column ) ;
	}

	/**
	 * row行，colum列 のデータの値を返します。
	 * これは、データの値そのものではなく、その値のラベル文字を返します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 互換性確保のメソッド
	 *
	 * @param	row int
	 * @param	column int
	 * @return	row行，colum列 のデータの値
	 */
	protected String getRendererValue( final int row,final int column) {
		return getRendererValue( row,column,getValue( row,column ) );
	}

	/**
	 * row行，colum列 のデータの値を返します。
	 * これは、データの値そのものではなく、その値のラベル文字を返します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 追加による引数変更
	 * @og.rev 3.8.5.0 (2006/03/20) clmGroup 追加によるグループ化処理
	 * @og.rev 4.0.0 (2005/11/30) 行番号に対応した値を返すように変更します。
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) カラムグループの groupClass 対応
	 *
	 * @param	row int
	 * @param	column int
	 * @param	inVal String
	 * @return	row行，colum列 のデータの値
	 */
	protected String getRendererValue( final int row,final int column , final String inVal ) {
		// 3.8.5.0 (2006/03/20) clmGroup 追加によるグループ化処理
//		if( clmGroup[column] && row > 0 && row != startNo &&
//			inVal != null && inVal.equals( getValue( row-1,column ) ) ) {
//				return "";
//		}

		String val = dbColumn[column].getRendererValue( row,inVal );

		// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
		if( useEventCols[column] && isWritable( row ) ) {
			val = dbColumn[column].getEventColumnTag( val, getValue( row,column ), row, false );
		}
		if( viewLink != null ) {
			val = viewLink.getMarkerString( row,column,val );
		}
		if( viewMarker != null ) {
			val = viewMarker.getMarkerString( row,column,val );
		}

		// 5.1.8.0 (2010/07/01) groupClass 対応：空文字の場合を考慮し、最後に処理を行う。
		// 出力する値が、空文字列なら、前行と同じでも、空文字にしておきます。
		if( groupType != null && 
				clmGroup[column] && row > 0 && row != startNo &&
				inVal != null && inVal.equals( getValue( row-1,column ) ) &&
				val != null && val.length() > 0 ) {
			if( groupType == CLM_GRP.KIGO ) {
				val = groupClass;
			}
			else {	// groupType == CLM_GRP.CHAR
				val = "<span class=\"" + groupClass + "\">" + val + "</span>";
			}
		}

		return val;
	}

	/**
	 * row行，colum列 のデータの値をHTML文字列に変換して返します。
	 * リソースバンドルが登録されている場合は,リソースに応じた
	 * HTML文字列を作成します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 追加による引数変更
	 * @og.rev 3.8.6.1 (2006/10/20) editMarker 追加
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param	row int
	 * @param	column int
	 * @param	inVal String
	 * @return	row行，colum列 のデータの値
	 */
	protected String getEditorValue( final int row, final int column, final String inVal ) {
		String val = dbColumn[column].getEditorValue( row,inVal );
		// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
		if( useEventCols[column] ) {
			val = dbColumn[column].getEventColumnTag( val, getValue( row,column ), row, true );
		}
		if( editMarker != null ) {
			val = editMarker.getMarkerString( row,column,val );
		}

		return val;
	}

	/**
	 * row行，colum列 のデータの値をHTML文字列に変換して返します。
	 * リソースバンドルが登録されている場合は,リソースに応じた
	 * HTML文字列を作成します。
	 * リソースバンドルが登録されていない場合は,getValue( int row,int column )を
	 * 返します。
	 * 新規機能として、writableControl によるエディター・レンデラー指定が出来ます。
	 * その場合、表示/編集ともに、先頭のアンダーバーは削除されます。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl によるエディター・レンデラー指定
	 *
	 * @param	row int
	 * @param	column int
	 * @return	row行，colum列 のデータの値
	 */
	protected String getValueLabel( final int row, final int column ) {
		String val = getValue( row,column ) ;
		boolean isEdit = isColumnWritable( column ) && isWritable( row ) ;
		if( writeCtrl[ column ] && val != null && val.length() > 0 && val.charAt(0) == '_' ) {
			isEdit = false;
			val = val.substring(1);		// 先頭の '_' を削除
		}

		if( isEdit ) {
			return getEditorValue( row,column,val );		// 引数 val 追加
		}
		else {
			return getRendererValue( row,column,val );		// 引数 val 追加
		}
	}

	/**
	 * カラムのクラスを文字列にした名称を返します。
	 * これは，HTML上の各種タグに，データベース定義に応じたクラスを
	 * セットし,CSS（Cascading Style Sheet)の class="xxxxx" とする事により
	 * 各種スタイルを表現するのに使用します。
	 *
	 * ここでは, カラムリソースの DBTYPE 属性で指定の文字列（X,S9,KXなど）を返します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規作成（getColumnClassName ⇒ getColumnDbType）
	 * @og.rev 5.2.2.0 (2010/11/01) className が VARCHAR2,NUMBER以外の場合は、合わせて出力します。
	 *
	 * @param   column int
	 * @return  カラムの DBTYPE 属性
	 */
	public String getColumnDbType( final int column ) {
		// 5.2.2.0 (2010/11/01) className が BIT の場合のみ、合わせて出力
		String clsName = dbColumn[column].getClassName();
		if( clsName != null && ! "VARCHAR2".equals( clsName ) && !"NUMBER".equals( clsName ) ) {
			return clsName + " " + dbColumn[column].getDbType();
		}

		return dbColumn[column].getDbType();
	}

	/**
	 * カラムが書き込み可能かどうかを返します。
	 *
	 * @param	column int
	 * @return	書込み可能(true)／不可能（false)
	 */
	protected boolean isColumnWritable( final int column ) {
		return clmWritable[ column ];
	}

	/**
	 * カラムが書き込み可能かどうかをセットします。
	 *
	 * @param	column int
	 * @param	rw 書込み可能(true)／不可能（false)
	 */
	public void setColumnWritable( final int column,final boolean rw ) {
		clmWritable[ column ] = rw;
	}

	/**
	 * 書き込み可能カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * setColumnWritable( int column,boolean rw ) の簡易版です。
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	columnName String
	 */
	public void setColumnWritable( final String columnName ) {
		if( columnName != null ) {
			setBooleanArray( columnName,true,clmWritable );

//			Arrays.fill( clmWritable,false );		// カラムの書込み不許可
//			String[] clmNames = StringUtil.csv2Array( columnName );
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { clmWritable[ no ] = true; }
//				}
//			}
			// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
			setUseEventCols();
		}
	}

	/**
	 * 書き込み不可カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	columnName String
	 */
	public void setNoWritable( final String columnName ) {
		if( columnName != null ) {
			setBooleanArray( columnName,false,clmWritable );

//			Arrays.fill( clmWritable,true );		// カラムの書込み許可
//			String[] clmNames = StringUtil.csv2Array( columnName );
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { clmWritable[ no ] = false; }
//				}
//			}
			// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
			setUseEventCols();
		}
	}

	/**
	 * 各項目がイベントカラムをするかどうかをセットします。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成(動的プルダウン実装見直し)
	 */
	private void setUseEventCols() {
		for( int column=0; column<dbColumn.length; column++ ) {
			String evCols = dbColumn[column].getEventColumn();
			if( evCols != null && evCols.length() > 0 ) {
				String[] evColsArr = StringUtil.csv2Array( evCols );
				for( int i=0; i<evColsArr.length; i++ ) {
					String evCol = evColsArr[i];
					if( evCol.charAt(0) == '_' ) { evCol = evCol.substring( 1 ); }
					if( isColumnWritable( table.getColumnNo( evCol ) ) ) {
						useEventCols[column] = true;
						break;
					}
				}
			}
		}
	}

	/**
	 * 行指定の書込み許可を返します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) writable カラムが "2" のときも、書き込み許可とする
	 * @og.rev 3.5.5.7 (2004/05/10) 判定ロジックを若干見直します。
	 *
	 * @param	row int
	 * @return	書込み可能(true)／不可能（false)
	 */
	protected boolean isWritable( final int row ) {
		boolean rtn = table.isRowWritable( row );
		if( rtn && rowWritableFlag ) {
			String val = table.getValue( row,0 );
			rtn = (	"TRUE".equalsIgnoreCase( val ) ||
					"1".equalsIgnoreCase( val )    ||
					"2".equalsIgnoreCase( val ) );

			if( ! rtn &&
				! "FALSE".equalsIgnoreCase( val ) &&
				! "0".equalsIgnoreCase( val ) &&
				! "".equalsIgnoreCase( val ) ) {
					String errMsg = "writable は、TRUE,FALSE,0,1,2,null 以外指定できません。" +
									"  row=[" + (row+1) + "]  val=[" + val + "]";
					throw new HybsSystemException( errMsg );
			}
		}
		return rtn;
	}

	/**
	 * 書き込み可能な行（rowWritable == true）のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) writable カラムが "2" のとき、チェックを付ける。
	 *
	 * @param	row 行の指定
	 * @return	チェックON(true)／チェックOFF（false)
	 */
	protected boolean isChecked( final int row ) {
		boolean rtn = table.isRowChecked( row );
		if( rowWritableFlag ) {
			String val = table.getValue( row,0 );
			rtn = rtn || ( "2".equalsIgnoreCase( val ) );
		}
		return rtn;
	}

	/**
	 * チェック済みの行の先頭に、フォーカスを当てる処理で、チェックの一つ前の
	 * 行番号欄にダミーのリンクを作成する為の判定を行います。
	 * つまり、指定の行番号の次の値が、チェックされているかどうかを判断します。
	 * さらに、これは、一番上位にチェックされている番号の時のみ
	 * 返します。（表示テーブル中、最高一回のみ、true が返る。）
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 * @og.rev 3.7.0.3 (2005/03/01) setScrollRowNo の対応
	 * @og.rev 3.8.8.3 (2007/02/09) 最初の選択行 より指定行番号へのフォーカスを優先
	 * @og.rev 4.2.3.1 (2008/06/11) 1件目がチェックされている場合は、対象外とする。
	 * @og.rev 4.2.4.0 (2008/06/25) 1件目対象外の判定で、指定行番号へのフォーカスを優先する。
	 * @og.rev 5.1.1.1 (2009/12/02) 選択行アンカーは、自身の行に出す(default.js#focus2()も合わせて変更)
	 *
	 * @param	row  int 行の指定
	 * @return	チェックON(true)／チェックOFF（false)
	 */
	protected boolean isFirstChecked( final int row ) {

		// 最初の１回のみ、true を返す。よって、元がtrue なら、false を返す。
		if( firstChecked ) { return false; }
	
		// 指定の次の行を見るため、範囲オーバーしていないか確認
//		if( row+1 >= rowCount ) { return false; }
		if( row >= rowCount ) { return false; } // 5.1.1.1 (2009/12/02)

		final boolean rtn;
		// 3.8.8.3 (2007/02/09) 最初の選択行 より指定行番号へのフォーカスを優先
		if( scrollRowNo >= 0 ) {
//			rtn = ( row+1 == scrollRowNo );
			rtn = ( row == scrollRowNo ); // 5.1.1.1 (2009/12/02)
		}
		else {
			// 1件目がチェックされている場合は、対象外とする。4.2.3.1 (2008/06/11)
			if( row == 0 && table.isRowChecked( row ) ) {
				firstChecked = true;
				return false;
			}

//			rtn = table.isRowChecked( row+1 );
			rtn = table.isRowChecked( row ); // 5.1.1.1 (2009/12/02)
		}

		// 最初の１回のみ、true を返す。よって、元がtrue なら、false を返す。
		if( rtn ) { firstChecked = true; }
		return rtn;
	}

	/**
	 * その行が、処理対象かどうかをチェックします。
	 * 処理対象かどうかは、書き込み可能な行（rowWritable == true）
	 * で且つ チェックされた行（rowChecked == true) とします。
	 * 例えば,NEXT/PREVでスキップ等の処理を行う場合は､ 処理対象以外を
	 * スキップすることで実現できます。
	 *
	 * @param	row int
	 * @return	 処理対象(true)／処理対象でない（false)
	 */
	public boolean isMarked( final int row ) {
		return ( isWritable( row ) && isChecked( row ) );
	}

	/**
	 * カラムが表示可能かどうかを返します。
	 * もし，表示不可の場合は,このカラムの全データが，表示対象から外されます。
	 *
	 * @param	column int
	 * @return	表示可能(true)／不可能（false)
	 */
	protected boolean isColumnDisplay( final int column ) {
		if( rowWritableFlag && column == 0 ) { return false; }
		return clmDisplay[ column ];
	}

	/**
	 * 表示可能なカラムの数を返します。
	 *
	 * @return	int 表示可能なカラム数
	 */
	protected int getColumnDisplayCount() {
		int rtn = 0;
		for( int i=0; i<columnCount; i++ ) {
			if( isColumnDisplay(i) ) { rtn++ ; }
		}

		return rtn;
	}

	/**
	 * カラムが表示可能かどうかをセットします。
	 * DBColumnのROLE属性による表示可否は、ViewFormTag で設定されます。
	 *
	 * @param	column int
	 * @param	rw 表示可能(true)／不可能（false)
	 */
	public void setColumnDisplay( final int column,final boolean rw ) {
		clmDisplay[ column ] = rw;
	}

	/**
	 * 表示可能カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * setColumnDisplay( int column,boolean rw ) の簡易版です。
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	columnName String
	 */
	public void setColumnDisplay( final String columnName ) {
		setBooleanArray( columnName,true,clmDisplay );

//		if( columnName != null ) {
//			String[] clmNames = StringUtil.csv2Array( columnName );
//			Arrays.fill( clmDisplay ,false );	// すべて表示しないに設定する。
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { clmDisplay[ no ] = true; }
//				}
//			}
//		}
	}

	/**
	 * 表示不可カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	columnName String
	 */
	public void setNoDisplay( final String columnName ) {
		setBooleanArray( columnName,false,clmDisplay );

//		if( columnName != null ) {
//			String[] clmNames = StringUtil.csv2Array( columnName );
//			Arrays.fill( clmDisplay ,true );	// すべて表示するに設定する。
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { clmDisplay[ no ] = false; }
//				}
//			}
//		}
	}

	/**
	 * 同一表示データをまとめる（表示しない）カラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムのグループ化指定を行います。
	 * 同じカラム番号で一つ上の行番号と同じ内容の場合に、表示しないというものです。
	 * 対応は、表示(Renderer)時のみとします。
	 * カラム単位なので、新しいゼブラによる色分けは行いません。(任意のカラムに適用できる為)
	 * また、ファイル出力などのデータ自身は変更されていませんので、そのままデータが落ちます。
	 * また、全カラムについて、有効にする場合は、group="*" を設定します。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	group String
	 */
	public void setColumnGroup( final String group ) {
		if( group != null ) {
			if( groupType == null ) { groupType = CLM_GRP.KIGO ; }
			setBooleanArray( group,true,clmGroup );

//			String[] clmNames = StringUtil.csv2Array( group );
//			Arrays.fill( clmGroup ,false );	// すべてグループ化しないに設定する。
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { clmGroup[ no ] = true; }
//				}
//			}
		}
	}

	/**
	 * 同一表示データをまとめないカラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムのグループ化指定で、同一でもまとめないカラムを指定するほうが簡単な場合に
	 * 使用します。例えば、キー情報以外は、グループ化指定したい場合などに便利です。
	 * 動作は、columnGroup の動きと同じです。（まとめないカラムを指定するだけです。）
	 *
	 * これは、同一表示データをまとめるカラム名の指定(columnGroup)と同時にセットする
	 * ことは出来ません。
	 * また、全カラムについて、有効にする場合は、group="*" を設定します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param	group String
	 */
	public void setNoGroup( final String group ) {
		if( group != null ) {
			if( groupType == null ) { groupType = CLM_GRP.KIGO ; }
			setBooleanArray( group,false,clmGroup );
		}
	}

	/**
	 * 同一表示データをまとめる場合の、表示方法を指定します。
	 *
	 * これは、カラムのグループ化指定(columnGroup)を行う場合の、まとめ表示の方法を指定します。
	 * なにも指定しない場合は、まとめカラムは、表示しない(ゼロ文字列)になります。
	 * その場合、先の行と同じ場合も、変更されたカラムが、NULL(ゼロ文字列)であった場合も、
	 * 同じ用に表示されるため、区別が付きません。
	 * そこで、前の行と同じデータの場合に、特殊な処理を行うことで、区別できるようにします。
	 * 指定方法が特殊なので、注意が必要です。
	 *   記号の場合：
	 *       記号(-- , → , ↓ , * など）が指定された場合は、そのままの文字に置き換えられます。
	 *   アルファベットの場合：
	 *       アルファベット(a-z,A-Z)の場合は、&lt;span class="アルファベット"&gt;元の文字&lt;/span&gt;
	 * 例えば、バックカラーに色を付ける、文字を薄い灰色にする、などできます。
	 * ただし、データ量が圧倒的に増えるため、大量のデータ等で使用するのは避けたほうが良いと思います。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 *
	 * @param	grpCls String
	 */
	public void setGroupClass( final String grpCls ) {
		if( grpCls != null ) {
			groupClass = grpCls ;
			if( groupClass.length() == 0 ) {	// ゼロ文字列の場合
				groupType = CLM_GRP.KIGO ;
			}
			else {
				char ch = groupClass.charAt(0);
				if( ( ch >= 'a' && ch <= 'z' ) || ( ch >= 'a' && ch <= 'z' ) ) {
					groupType = CLM_GRP.CHAR ;
				}
				else {
					groupType = CLM_GRP.KIGO ;
				}
			}
		}
	}

	/**
	 * カラム名リンクソートを表示するカラム名を、カンマ区切りで与えます。
	 *
	 * ヘッダーにソート用リンクを作成する useTableSorter 属性 に対して、
	 * カラム個別に作成する場合のカラム名をカンマ区切り文字で指定します。
	 * この tableSorterKeys 属性は、useTableSorter 属性 と無関係に、指定した
	 * カラムのみ、リンクを表示します。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param	columnName String
	 */
	public void setTableSorterKeys( final String columnName ) {
		if( columnName != null ) {
			setBooleanArray( columnName,true,sortKeys );

			useSorterKeys = true;				// 使用したことを記憶
//			String[] clmNames = StringUtil.csv2Array( columnName );
//			for( int i=0; i<clmNames.length; i++ ) {
//				if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//					int no = table.getColumnNo( clmNames[i] );
//					if( no >= 0 ) { sortKeys[ no ] = true; }
//				}
//			}
		}
	}

	/**
	 * 各カラムのフィールドのデータ長を返します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBColumn の size と maxlength の 意味を変更
	 *
	 * @param	column int
	 * @return	カラム数
	 */
	protected int getColumnSize( final int column ) {
		return dbColumn[column].getTotalSize();	// 4.0.0 (2005/01/31) メソッド名変更
	}

	/**
	 * カラム数を返します。
	 *
	 * @return	カラム数
	 */
	protected int getColumnCount() {
		return columnCount ;
	}

	/**
	 * 行数を返します。
	 *
	 * @return	行数
	 */
	protected int getRowCount() {
		return rowCount;
	}

	/**
	 * リストボックスを作成する場合の name をセットします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param  name 属性
	 */
	protected void setName( final String name ) {
		this.name = name;
	}

	/**
	 * リストボックスを作成する場合の name を返します。
	 *
	 * @return	name 属性
	 */
	protected String getName() {
		return name;
	}

	/**
	 * カラム名をもとに、そのカラム番号を返します。
	 * カラム名が存在しない場合は､ HybsSystemException を throw します。
	 *
	 * @param	columnName	 カラム名
	 * @return	カラム番号
	 */
	protected int getColumnNo( final String columnName ) {
		return table.getColumnNo( columnName );
	}

	/**
	 * 表示開始位置を返します。
	 *
	 * @return	startNo    表示開始位置
	 */
	public int getStartNo() {
		return startNo;
	}

	/**
	 * 表示開始位置をセットします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param  no	 表示開始位置
	 */
	public void setStartNo( final int no ) {
		if( no >= 0 ) { startNo = no; }
	}

	/**
	 * 先頭へ戻るリンク間隔を返します。
	 *
	 * @return	backLinkCount	 リンク間隔
	 */
	public int getBackLinkCount() {
		return backLinkCount;
	}

	/**
	 * 先頭へ戻るリンク間隔をセットします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param  no	リンク間隔
	 */
	public void setBackLinkCount( final int no ) {
		if( no >= 0 ) { backLinkCount = no; }
	}

	/**
	 * ヘッダーを出力する間隔を取得します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @return	ヘッダーの表示間隔 （ 0:通常ヘッダ、n:n回ごとに現れる ）
	 */
	public int getHeaderSkipCount() {
		return headerSkipCount;
	}

	/**
	 * ヘッダーを出力する間隔をセットします。
	 *
	 * 0を指定すると、繰り返しません（つまり、ヘッダーを最初に表示するだけです。）
	 * 数字を指定すると、その回数毎に、ヘッダーをデータの行に挿入します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @param	hsc ヘッダーの表示間隔 （ 0:通常ヘッダ、n:n回ごとに現れる ）
	 */
	public void setHeaderSkipCount( final int hsc ) {
		headerSkipCount = hsc;
	}

	/**
	 * 表示件数を取得します。
	 *
	 * @return	表示件数
	 */
	public int getPageSize() {
		return pageSize;
	}

	/**
	 * 表示件数をセットします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param	psize 表示件数
	 */
	public void setPageSize( final int psize ) {
		if( psize > 0 ) { pageSize = psize; }
	}

	/**
	 * フォーマットを設定します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 * @param list List
	 */
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		String errMsg = "このメソッドは、フォーマット関係のビューでのみ使用できます。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 表示時の選択用オブジェクトのタイプを指定します。
	 * ・複数選択可能時は "checkbox" を指定します。
	 * ・一つだけ選ばせる場合は, "radio" を指定します。
	 * ・隠しフィールドで全件を選ぶ場合は、"hidden" を指定します。
	 * ・Noカラムそのものを表示させない場合は、"none" を指定します。
	 * デフォルトは、"checkbox" です。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 選択用オブジェクトのタイプとして"hidden" を追加
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.1.0 (2003/10/03) displayNumber 廃止。numberType 新規追加。
	 *
	 * @param	type 選択用オブジェクトのタイプ（ "checkbox"/"radio"/"hidden"/"null" ）
	 */
	public void setSelectedType( final String type ) {
		if( type != null ) {
			if( "checkbox".equalsIgnoreCase( type  ) ||
				"radio".equalsIgnoreCase( type	) ||
				"hidden".equalsIgnoreCase( type	) ) {
					selectedType = type;
			}
			else {
				selectedType = null;
			}
		}
	}

	/**
	 * 表示時の行番号の表示方法を指定します。
	 * ・sequenceは、１から始まる連番です。
	 * ・none を指定すると、番号は表示されません。
	 * ・delete を指定すると、行番号表示そのもののタグを取り除きます。(3.5.5.0 追加)
	 * ・skip=カラム名を指定すると、そのカラムの値が NULL の場合は、番号をスキップします。
	 * ・view=カラム名を指定すると、そのカラムの値が、使用されます。（ラベルは付きません）
	 * skip=XX と、view=XX は、=の前後にスペースを入れないで下さい。
	 * デフォルトは、"sequence" です。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 * @og.rev 3.5.5.0 (2004/03/12) numberType 属性の "delete" 値追加に伴なう、numberDisplay 変数の追加。
	 * @og.rev 5.3.0.0 (2010/12/01) 左右分割スクロール(SplitViewTag)対応。numberType 再設定時のnumberDisplay初期化
	 *
	 * @param	type 行番号の表示方法のタイプ（ /sequence/none/delete/skip=カラム名/view=カラム名/ ）
	 */
	public void setNumberType( final String type ) {

		numberType = type ;
		numberDisplay = true;	// 5.3.0.0 (2010/12/01) numberType 再設定時のnumberDisplay初期化
		if( type != null && !type.startsWith( "seq" ) && !type.startsWith( "none" ) ) {
			if( type.startsWith( "skip=" ) ) {
				numberTypeClm = table.getColumnNo( type.substring( 5 ) );
				numberTypeData = new String[rowCount];
				int cnt = 1;
				for( int i=0; i<rowCount; i++ ) {
					String tmp = table.getValue( i,numberTypeClm );
					if( tmp != null && tmp.length() > 0 ) {
						numberTypeData[i] = String.valueOf( cnt );
						cnt++ ;
					}
					else {
						numberTypeData[i] = "" ;
					}
				}
			}
			else if( type.startsWith( "view=" ) ) {
				numberTypeClm = getColumnNo( type.substring( 5 ) );
			}
			// 3.5.5.0 (2004/03/12)
			else if( type.startsWith( "delete" ) ) {
				numberDisplay = false;
			}
			else {
				// 3.5.5.0 (2004/03/12) 不正な値をエラーチェック
				String errMsg = "numberType属性の値が不正です。numberType=[" + numberType + "]" +
						HybsSystem.CR +
						"設定できるのは、sequence/none/delete/skip=カラム名/view=カラム名/ です。";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 表示時の選択用オブジェクトのタイプを返します。
	 * 複数選択可能時は "checkbox"	一つだけ選ばせる場合は, "radio" を指定します。
	 * デフォルトは、"checkbox" です。
	 * "checkbox"/"radio"/"hidden" 以外の文字列の場合は, null を返します。
	 *
	 * @return	 選択用オブジェクトのタイプ（ "checkbox"/"radio"/"hidden"/"null" ）
	 */
	protected String getSelectedType() {
		return selectedType ;
	}

	/**
	 * No カラムの文字列を取得します。
	 * setNumberType で指定した、（sequence / none / skip=カラム名 / view=カラム名）に
	 * 応じた値(レンデラーの値)を返します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 *
	 * @param   row 行番号
	 * @return	 No カラムの文字列
	 */
	protected String getNumberData( final int row ) {
		String rtn = null;

		if( numberType == null || numberType.startsWith( "se" ) ) {	// sequence
			rtn = String.valueOf( row + 1 );
		}
		else {
			if( numberType.startsWith( "no" ) ) {			// none
				rtn = "";
			}
			else if( numberType.startsWith( "vi" ) ) {		// view
				rtn = getRendererValue( row,numberTypeClm );
			}
			else if( numberType.startsWith( "sk" ) ) {		// skip
				rtn = numberTypeData[row];
			}
		}

		return rtn ;
	}

	/**
	 * No カラムのヘッダー文字列を取得します。
	 * setNumberType で指定した、（sequence / none / skip=カラム名 / view=カラム名）に
	 * 応じた値(ラベル)を返します。
	 * 具体的には、none 以外は、"No" という文字を、none の時は、""(ゼロストリング)を返します。
	 *
	 * @og.rev 3.5.4.6 (2004/01/30) 新規作成
	 *
	 * @return	 No カラムのヘッダー文字列
	 */
	protected String getNumberHeader() {
		String rtn = NO_HEADER;

		if( numberType.startsWith( "no" ) ) {			// none
			rtn = "";
		}

		return rtn ;
	}

	/**
	 * テーブル等のチェックボックスに属性を付加します
	 * JavaScript などの HTML基本タグ以外の属性を、そのまま
	 * チェックボックス/ラジオボタン等に使用します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param	option オプション属性文字列
	 */
	public void setOptionTypeAttributes( final String option ) {
		optTypeAttri = option;
	}

	/**
	 * テーブル等のチェックボックスに属性を付加します
	 * JavaScript などの HTML基本タグ以外の属性を、そのまま
	 * チェックボックス/ラジオボタン等に使用します。
	 *
	 * @return	 String オプション属性
	 */
	protected String getOptionTypeAttributes() {
		return optTypeAttri ;
	}

	/**
	 * 最終表示番号を取得します。
	 * 最終表示番号は、表示開始位置 + 表示件数 で求まります。
	 * lastNo = startNo +  pageSize;
	 *
	 * ただし、最終表示番号 > データ件数 の場合は､ 最終表示番号 = データ件数
	 * とします。
	 * なお、表示件数 が -1 の場合は､デフォルトの表示件数を使用します。
	 *
	 * @param	startNo    表示開始位置
	 * @param	pageSize   表示件数
	 * @return	最終表示番号
	 */
	protected int getLastNo( final int startNo, final int pageSize ) {
		int lastNo = startNo +	(( pageSize < 0 ) ? getPageSize() : pageSize );
		if( lastNo > rowCount ) {
			lastNo = rowCount ;
		}

		if( rowCount < 0 || startNo < 0 || lastNo < 0 ) {
			String errMsg = "startNo lastNo の範囲が不正です。" + HybsSystem.CR
						+ " startNo=" + startNo + " , lastNo=" + lastNo + " , RowCount=" + rowCount;
			throw new HybsSystemException( errMsg );
		}

		return lastNo ;
	}

	/**
	 * ビューフォームのタイプを返します。
	 * これは、ViewFormFactory で、org.opengion.hayabusa.html.ViewForm_ + 『type』
	 * で作成されるサブクラスの 『type』 部分を返します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) viewFormID を使用するように変更します。
	 *
	 * @return	ビューフォームのタイプ
	 */
	public String getViewFormType() {
		return viewFormID ;
	}

	/**
	 * ビューリンクオブジェクトを設定します。
	 * これは、ViewLink タグで作成された キー（カラム名）とリンク文字列を
	 * 持っている Attributes オブジェクトを受け取り、内部でリンク表示に
	 * 使用します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) table が null の場合は、セットしません。
	 *
	 * @param  link ViewLink ビューリンクオブジェクト
	 */
	public void setViewLink( final ViewMarker link ) {
		viewLink = link;
		if( viewLink != null && table != null ) {
			viewLink.setDBTableModel( table ) ;
		}
	}

	/**
	 * ビューマーカーオブジェクトを設定します。
	 * これは、ViewMarker タグで作成された キー（カラム名）とマーカー文字列を
	 * 持っている Attributes オブジェクトを受け取り、内部でマーカー表示に
	 * 使用します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) table が null の場合は、セットしません。
	 *
	 * @param  marker ViewMarker ビューマーカーオブジェクト
	 */
	public void setViewMarker( final ViewMarker marker ) {
		viewMarker = marker;
		if( viewMarker != null && table != null ) {
			viewMarker.setDBTableModel( table ) ;
		}
	}

	/**
	 * エディットマーカーオブジェクトを設定します。 <br />
	 * これは、ViewMarker タグで、isRendere="false" で作成された エディットキー（カラム名）と
	 * マーカー文字列を持っている Attributes オブジェクトを受け取り、内部でマーカー表示に
	 * 使用します。<br />
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  marker ViewMarker ビューマーカーオブジェクト
	 */
	public void setEditMarker( final ViewMarker marker ) {
		editMarker = marker;
		if( editMarker != null && table != null ) {
			editMarker.setDBTableModel( table ) ;
		}
	}

	/**
	 * 検索結果メッセージを表示する/しないを設定します
	 * デフォルトは、表示する（false)です。
	 *
	 * @param	noMessage boolean (true:表示しない / false:表示する。）
	 */
	public void setNoMessage( final boolean noMessage ) {
		this.noMessage = noMessage;
	}

	/**
	 * DBColumn オブジェクトを返します。
	 *
	 * @og.rev 3.1.8.0 (2003/05/16) DBColumn オブジェクト取得用のメソッド追加
	 *
	 * @param	column int
	 * @return  DBColumn オブジェクト
	 */
	protected DBColumn getDBColumn( final int column ) {
		return dbColumn[column];
	}

	/**
	 * チェックの入った行のみを表示させるかどうか指定します。
	 *
	 * "true" で、チェックの入った行のみを表示させます。
	 * 従来は、TextField系のViewに対して、NEXT,PREVでチェックの
	 * 入った行のみを表示させる機能でしたが、Table系のViewに対しても、
	 * 同様に機能するように、しました。
	 * デフォルトは、ViewForm.DEFAULT_SKIP です。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 新規追加
	 *
	 * @param	skp チェックの入った行のみを表示させるかどうか（true:のみ表示/false:前件表示）
	 */
	public void setSkip( final boolean skp ) {
		skip = skp;
	}

	/**
	 * チェックの入った行のみを表示させるかどうか（スキップするかどうか）を返します。
	 *
	 * skip=true で、かつ、チェックの入っていない行の場合に、trueを返します。
	 * つまり、skip=trueの場合は、チェックの入っていない行は、スキップするという
	 * 判断を行います。
	 * skip属性のデフォルトは、ViewForm.DEFAULT_SKIP です。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 新規追加
	 *
	 * @param	row int
	 * @return	スキップする(true)／スキップしない（false)
	 */
	protected boolean isSkip( final int row ) {
		return skip && ! isChecked( row );
	}

	/**
	 * チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを指定します。
	 *
	 * 1 で、コントロール用のチェックボックスが現れて、この機能を使用することができるようになります。
	 * 0 は、従来どおりです。
	 * 使用するにあたり、jsp/commpn/default.js にJavaScriptを設定しておきます。(設定済み）
	 * デフォルトは、システムパラメータ の VIEW_USE_CHECK_CONTROL です。(0:使用しない)
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール変更( boolean ⇒ int )
	 *
	 * @param	chCtrl チェックボックスの全チェックを選択する機能を使用するかどうか
	 *			（1:使用する/0:使用しない/2:初期値チェック済み）
	 */
	public void setUseCheckControl( final int chCtrl ) {
		useCheckControl = chCtrl;
	}

	/**
	 * チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを返します。
	 *
	 * 内部的には、（1:使用する/0:使用しない/2:初期値チェック済み）の値を保持していますが、
	 * タグを作成する場合には、まず、使用するかどうかを指定する必要があるため、
	 * 1:使用する/2:初期値チェック済み は、true:使用する、0 は、false:使用しない
	 * を返します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	useCheckControl チェックボックスの全チェックを選択する機能を使用するかどうか（true:使用する/false:使用しない）
	 */
	protected boolean isUseCheckControl() {
		return useCheckControl > 0;
	}

	/**
	 * チェックボックスの全チェックを選択する タグ文字列を返します。
	 *
	 * 内部的には、（1:使用する/0:使用しない/2:初期値チェック済み）の値を
	 * 保持していますので、それに対応した文字列を返します。
	 *
	 * @og.rev 3.7.0.1 (2005/01/31) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) 全チェックを押した状態で画面遷移した場合にチェック状態を保持する。
	 *
	 * @return	チェックボックスの全チェックを選択する機能を使用するかどうか
	 *			（1:使用する/0:使用しない/2:初期値チェック済み）
	 */
	protected String getAllCheckControl() {
		final String rtn;
		switch( useCheckControl ) {
			case 1: rtn = "<input type=\"checkbox\" name=\"ALL_CHECK\" value=\"2\" onClick=\"checkbox(this);\" />" ;
					break;
			case 2: rtn = "<input type=\"checkbox\" name=\"ALL_CHECK\" value=\"2\" onClick=\"checkbox(this);\" checked=\"checked\" />" ;
					break;
			default : rtn = "" ;
		}
		return rtn ;
	}

	/**
	 * ヘッダーにソート用リンクを作成するかどうかを指定します。
	 *
	 * "true" で、ヘッダーにソート用リンクを作成します。
	 * false は、作成しません。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規追加
	 *
	 * @param	flag ヘッダーにソート用リンクを作成するかどうか（true:作成する/false:作成しない）
	 */
	public void setUseTableSorter( final boolean flag ) {
		useTableSorter = flag;
	}

	/**
	 * ヘッダーにソート用リンクを作成する時の、リクエスト引数のMapを設定します。
	 *
	 * ソート用URLに、リクエスト時の引数を設定する必要があります。
	 * そのため、リクエスト時のキーと値のセットをMapで指定します。
	 * このMap は、リクエストキャッシュより取り出します。
	 * 実装では、このMapを元に、URLのQuery部文字列を作成します。処理過程において、
	 * このMap を書き換えることは行いません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 * @og.rev 3.6.0.0 (2004/09/22) 引数が null の時の処理追加
	 * @og.rev 4.0.0 (2005/01/31) Map.keySet より Map.entrySet を使用するように変更
	 *
	 * @param	map tableSorter 時のリンクに追加するリクエスト変数のキャッシュMap
	 */
	public void makeTableSorterQuery( final Map<?,?> map ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		if( map == null ) { return; }			// 3.6.0.0 (2004/09/22)

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		Iterator<?> ite = map.entrySet().iterator() ;	// 4.0.0 (2005/01/31)
		while( ite.hasNext() ) {
			Map.Entry<?,?> entry = (Map.Entry<?,?>)ite.next();	// 4.3.3.6 (2008/11/15) Generics警告対応
			String key = (String)entry.getKey();
			if( key != null
					 &&	! key.equalsIgnoreCase( "command" )
					 &&	! key.equalsIgnoreCase( HybsSystem.SORT_COLUMNS ) ) {
				String[] vals = (String[])entry.getValue();	// 4.0.0 (2005/01/31)
				if( vals != null ) {
					for( int i=0; i<vals.length; i++ ) {
						buf.append( "&" ).append( key ).append( "=" );
						buf.append( StringUtil.urlEncode( vals[i] ) );
					}
				}
			}
		}

		if( buf.length() > 0 ) {
			sorterQuery = buf.toString();
		}
	}

	/**
	 * カラムソート機能（リンク）の付いたラベル名を返します。
	 * カラムの項目名に対して,見える形の文字列を返します。
	 * 一般には,リソースバンドルと組合せて,各国ロケール毎にラベルを
	 * 切替えます。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規追加
	 * @og.rev 3.6.0.0 (2004/09/17) sortKeys の使用によるカラム個別のリンク有無追加
	 * @og.rev 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用する場合の処理を追加
	 * @og.rev 5.2.3.0 (2010/12/01) チェックボックスでは、onChange ではなく、onClick でイベントを発生させます。
	 *
	 * @param	column int
	 * @return	ソート機能（リンク）の付いたラベル名
	 */
	protected String getSortedColumnLabel( final int column ) {
		String rtn = getColumnLabel( column ) ;

		if( ( useSorterKeys && sortKeys[column] ) ||
			( ! useSorterKeys && useTableSorter ) ) {
//			return "<a href=\"?command=VIEW&" +
//						HybsSystem.SORT_COLUMNS + "=" + dbColumn[column].getName() +
//						sorterQuery + "\">" +
//						getColumnLabel( column ) + "</a>" ;		// 4.0.0 (2005/01/31)
			rtn = "<a href=\"?command=VIEW&" +
					HybsSystem.SORT_COLUMNS + "=" + dbColumn[column].getName() +
					sorterQuery + "\">" +
					rtn + "</a>" ;
		}
//		else {
//			return getColumnLabel( column ) ;		// 4.0.0 (2005/01/31)
//		}

		// 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能対応
		if( isClmBulkSet( column ) ) {
			org.opengion.hayabusa.db.DBColumnConfig config = dbColumn[column].getConfig();
			config.setAddNoValue( true );
			if( "TEXT".equals( config.getEditor() ) ) {
				String mlen = config.getMaxlength();
				if( mlen != null && Integer.parseInt(mlen) < COLUMNS_MAXSIZE ) {
					config.setViewLength( mlen );
				}
				config.setMaxlength( "1000" );
			}
			else if( "RADIO".equals( config.getEditor() ) ) {
				config.setEditor( "MENU" );
			}

			Attributes attri = new Attributes();
			// 5.2.3.0 (2010/12/01) チェックボックスでは、onChange ではなく、onClick でイベントを発生させます。
			if( "CHBOX".equals( config.getEditor() ) ) {
				attri.add( "onClick",  "bulkSet(this);" );
			}
			else {
				attri.add( "onChange",  "bulkSet(this);" );
				attri.add( "ondblclick","bulkPaste(this);" );
			}
			attri.add( "onkeydown", "ctrlCV(this);" );
			config.setEditorAttributes( attri );

			String key = config.getName();
			config.setName( "h_" + key );
			DBColumn clm = new DBColumn( config );

			rtn = rtn + "<br />" + clm.getEditorValue( null ) ;
		}

		return rtn ;
	}

	/**
	 * ViewForm のサブクラスに渡すパラメータマップを設定します。
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 *
	 * @param	map Map
	 */
	public void setParam( final Map<String,String> map ) {
		viewParam = map ;
	}

	/**
	 * ViewForm のサブクラスに渡すパラメータマップの値を返します。
	 * パラメータが 存在しない（null）か、値が 存在しない（null）の場合は、
	 * 初期値を返します。
	 *
	 * @og.rev 3.5.5.9 (2004/06/07) 新規追加
	 *
	 * @param key String パラメータの取り出すキー
	 * @param def String パラメータが存在しない場合の初期値
	 * @return	パラメータ値
	 */
	protected String getParam( final String key, final String def ) {
		if( viewParam == null ) { return def; }
//		String rtn = (String)viewParam.get( key );
		String rtn = viewParam.get( key );		// 4.3.3.6 (2008/11/15) Generics警告対応

		return ( rtn != null ) ? rtn : def ;
	}

	/**
	 * No 欄そのものを作成するかどうかを返します。
	 *
	 * numberType 属性に、"delete" という値を設定した場合は、No 欄そのものを
	 * 作成しません。それ以外は、作成します。
	 * デフォルトは、作成する（true)です。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) 新規追加
	 *
	 * @return	numberDisplay No 欄そのものを作成するかどうか（true:作成する/false:作成しない）
	 */
	protected boolean isNumberDisplay() {
		return numberDisplay;
	}

	/**
	 * マーカーオブジェクト（リンク、マーカー）を設定します。
	 * ここでは、旧 ViewForm 属性を 新ViewForm に直接セットします。
	 *
	 * @og.rev 3.5.6.1 (2004/06/25) 新規追加
	 * @og.rev 3.8.6.1 (2006/10/20) editMarker 追加
	 *
	 * @param	view ViewForm
	 */
	public void markerSet( final ViewForm view ) {
		if( view instanceof AbstractViewForm ) {
			viewLink   = ((AbstractViewForm)view).viewLink ;
			if( viewLink != null ) { viewLink.setDBTableModel( table ); }
			viewMarker = ((AbstractViewForm)view).viewMarker;
			if( viewMarker != null ) { viewMarker.setDBTableModel( table ); }
			editMarker = ((AbstractViewForm)view).editMarker;
			if( editMarker != null ) { editMarker.setDBTableModel( table ); }
		}
		else {
			String errMsg = "AbstractViewForm 以外の view は、サポートしていません。"
					+ " view=[" + view + "]" ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします。
	 * 0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 * 初期値は、1(ゼブラ)です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) ViewForm_HTMLTable より移動
	 *
	 * @param  sycle  0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 */
	public void setBgColorCycle( final int sycle ) {
		bgColorCycle	= 1;		// 強制的に設定

		if( sycle > 0 ) {						// 1(ゼブラ)、2以上(行数まとめ)
			color_row0		= BG_COLOR_ROW0 ;
			color_row1		= BG_COLOR_ROW1 ;
			bgColorCycle	= sycle;			// このケースのみ引数を設定
		}
		else if( sycle == -1 ) {					// -1(ワーニング)
			color_row0		= BG_WARNING_COLOR_ROW0 ;
			color_row1		= BG_WARNING_COLOR_ROW1 ;
		}
		else if( sycle < -1 ) {					// -2以下(エラー)
			color_row0		= BG_ERROR_COLOR_ROW0 ;
			color_row1		= BG_ERROR_COLOR_ROW1 ;
		}
		else {											// 0(ゼブラなし)
			color_row0		= "" ;
			color_row1		= "" ;
		}
	}

	/**
	 * テーブルのバックグラウンドカラーのクラスに指定するカラム名を指定します。
	 *
	 * 通常のゼブラ模様は、tr 属性のクラス指定として、row_0、row_1 が指定されます。
	 * その代わりに、ここで指定されたカラムの値がクラス属性として設定されます。
	 * （指定するのはカラム名です。行単位にクラス属性の値を変えることが可能です。）
	 * 選択行(row_sel)は、優先して使用されます。
	 * 出力されるクラス名は、"row_" + 属性値 になります。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param  clsClm ゼブラ模様の替わりに指定するクラスを格納したカラム名
	 */
	public void setBgColorClsClm( final String clsClm ) {
		if( clsClm != null ) {
			bgColorClsClmNo = table.getColumnNo( clsClm );
		}
	}

	/**
	 * テーブルのバックグラウンドカラーの値をセットします。
	 * これは、DBTableModele が指定の行番号の時に、BG_COLOR_ROWSEL を
	 * 返します。それ以外は、通常の getBgColorCycleClass( int )と
	 * 同じ結果を返します。
	 * BG_COLOR_ROWSEL の使用有無は、システムパラメータの
	 * VIEW_USE_SELROW_COLOR 属性で指定できます。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規追加
	 * @og.rev 3.7.1.1 (2005/05/31) SEL_ROW機能使用時に、BG_COLOR_ROWSEL の使用 有/無を指定
	 *
	 * @param	indx 先頭からの連番（ ０から始める ）
	 * @param	row 行番号
	 * @return	行の色を指定する class 属性（ cssファイルで指定 ）
	 * @see	#getBgColorCycleClass( int )
	 */
	protected String getBgColorCycleClass( final int indx,final int row ) {
		if( useSelRowColor && ( scrollRowNo == row ) ) {
			return BG_COLOR_ROWSEL;
		}
		else {
			return getBgColorCycleClass( indx );
		}
	}

	/**
	 * テーブルのバックグラウンドカラーの値をセットします。
	 * 行番号は, ０から始まるので、偶数を HTML_BG_COLOR_ROW0 、
	 * 奇数行を HTML_BG_COLOR_ROW1 とします。
	 * setBgColorCycle で、設定値変換しています。
	 * bgColorClsClm が指定されている場合は、その値を利用したクラス属性を返します。
	 * クラス名は、"row_" + 指定カラムの値 です。
	 * 指定カラムの値 が、null または、空文字列の場合は、従来のゼブラ模様が優先されます。
	 * また、行選択があれば、そちらが最優先されます。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) ViewForm_HTMLTable より移動
	 * @og.rev 5.1.8.0 (2010/07/01) bgColorClsClmNo 対応
	 *
	 * @param	row 行番号（ ０から始める ）
	 * @return	行の色を指定する class 属性（ cssファイルで指定 ）
	 * @see		#setBgColorClsClm( String )
	 */
	protected String getBgColorCycleClass( final int row ) {
		String rtn = null ;

		// 5.1.8.0 (2010/07/01) bgColorClsClmNo 対応
		if( bgColorClsClmNo >= 0 ) {
			String val = table.getValue( row,bgColorClsClmNo );
			if( val != null && val.length() > 0 ) {
				rtn = " class=\"row_" + val + "\"";
			}
		}

		if( rtn == null ) {
			if( (row/bgColorCycle) % 2 == 0 ) {
				rtn = color_row0;		// 偶数の場合
			}
			else {
				rtn = color_row1;		// 奇数の場合
			}
		}

		return rtn ;
	}

	/**
	 * スクロールバー用のDIV要素を出力するかどうか(初期値はシステムパラメータ)
	 *
	 * スクロールバー対応する為、テーブルの先頭に、DIV要素を出力します。
	 * 初期値は、システムパラメータ の VIEW_USE_SCROLLBAR です。
	 * ※ 互換性の関係より、false になっています。
	 * ※ 互換性の関係より、新しいタイプのヘッダー固定を、TYPE2 とします。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) 新規追加
	 * @og.rev 3.8.0.3 (2005/07/15) barType 変数の追加
	 * @og.rev 4.0.0 (2007/04/10) ヘッダー固定のスクロールタイプは、TYPE2 のみにする。
	 *
	 * @param  useBar  スクロールバー用のDIV要素を出力するかどうか(true/false)
	 * <del>  barType String ヘッダー固定のタイプをセットします(null/TYPE2)</del>
	 */
	public void setUseScrollBar( final boolean useBar ) {
		useScrollBar  = useBar;
	}

	/**
	 * スクロールバー用の開始DIV要素返します。
	 *
	 * スクロールバー対応する為、テーブルの先頭に、DIV要素を出力します。
	 * 初期値は、システムパラメータ の VIEW_USE_SCROLLBAR で指定されたDIV要素です。
	 * DIV要素の開始タグになります。
	 *
	 * @og.rev 3.8.0.3 (2005/07/15) 新規追加
	 * @og.rev 4.2.0.0 (2008/03/18) outerにwidthとheightを出すように修正
	 *
	 * @return  LAYER_ST スクロールバー用の開始DIV要素
	 */
	protected String getScrollBarStartDiv() {
		String layerStart = LAYER_ST0 ;
		if( useScrollBar ) {
//			layerStart = LAYER_ST2 ;
			layerStart = "<div id=\"divPos\" style=\"";
			if( height != null ){
				layerStart += "height:" + height + ";";
			}
			if( width != null ){
				layerStart += "width:" + width + ";";
			}
			layerStart += "\" >"
				+	"<div id=\"outer\">"
				+	"<div id=\"layer\" onscroll=\"SetScrollHeader(this);\">"
				+	"<div id=\"divHide\">" ;
		}
		return layerStart;
	}

	/**
	 * スクロールバー用の終了DIV要素返します。
	 *
	 * スクロールバー対応する為、テーブルの終了に、DIV要素を出力します。
	 * 初期値は、システムパラメータ の VIEW_USE_SCROLLBAR で指定されたDIV要素です。
	 * DIV要素の終了タグになります。
	 *
	 * @og.rev 3.8.0.3 (2005/07/15) 新規追加
	 *
	 * @return  スクロールバー用の終了DIV要素
	 */
	protected String getScrollBarEndDiv() {
		String layerEnd = LAYER_END0 ;
		if( useScrollBar ) {
			layerEnd = LAYER_END2 ;
		}
		return layerEnd;
	}

	/**
	 * 指定の行番号まで画面をスクロールさせる場合の行番号を設定します。
	 *
	 * 画面をスクロール後、リンク等で他画面を表示後、戻ってきた場合に、
	 * 先のスクロール位置まで戻します。
	 * ただし、厳密に戻すことはできないため、大体のあたりに戻します。
	 * 指定しない場合(クリアする場合)は、-1 をセットしてください。
	 * useSelRowColor は、選択行に色づけするかどうかを指定します。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規追加
	 * @og.rev 3.7.1.1 (2005/05/31) 選択行マーカーの使用有無
	 *
	 * @param  rowNo int 指定の行番号まで画面をスクロールさせる場合の行番号
	 * @param  useSelRowColor boolean 選択行マーカーの使用有無
	 */
	public void setScrollRowNo( final int rowNo, final boolean useSelRowColor ) {
		scrollRowNo = rowNo;
		firstChecked = false;
		this.useSelRowColor = useSelRowColor;
	}

	/**
	 * 設定値に "_" が含まれている場合にレンデラーを使用するカラムをCSV形式で指定します。
	 *
	 * これは、従来の カラム定義の WRITABLE エディターと同等の働きを行うように
	 * カラム属性を指定します。
	 * WRITABLE エディターは、設定値にアンダーバー "_" が含まれている場合に、
	 * その値を書込み禁止にする機能です。これは、エディター自身が値を判断して
	 * 書き込み許可か禁止かを判断しています。
	 * この動きを汎用的にするため、指定のカラムをカンマ区切り文字(CSV)で指定
	 * することにより、レンデラーとエディターを設定値によって動的に切り替える
	 * 機能を実現します。
	 * その場合、表示/編集ともに、先頭のアンダーバーは削除されます。
	 * また、全カラムについて、有効にする場合は、writableControl="*" を設定します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) 内部ロジック変更(setBooleanArray)
	 *
	 * @param  wrtCtrl 書き込み制御を行いたいカラムをCSV形式で指定
	 */
	public void setWritableControl( final String wrtCtrl ) {
		setBooleanArray( wrtCtrl,true,writeCtrl );

//		if( wrtCtrl != null ) {
//			// 書き込み制御の許可 をカラム単位で設定。* の場合は、全カラム許可
//			if( "*".equals( wrtCtrl ) ) {
//				Arrays.fill( writeCtrl,true );
//			}
//			else {
//				Arrays.fill( writeCtrl,false );
//				String[] clmNames = StringUtil.csv2Array( wrtCtrl );
//				for( int i=0; i<clmNames.length; i++ ) {
//					if( clmNames[i] != null && clmNames[i].length() > 0 ) {
//						int no = table.getColumnNo( clmNames[i] );
//						if( no >= 0 ) { writeCtrl[ no ] = true; }
//					}
//				}
//			}
//		}
	}

	/**
	 * CSV形式で指定されたカラムに、true/false の初期設定を行います。
	 *
	 * 従来は、各クラスで実装されていた処理を、一箇所にまとめます。
	 * これにより、各種機能をあらかじめ実装しておきます。
	 * 指定のカラムが、null の場合は、何も処理を行いません。（つまり、初期値のまま）
	 * 指定のカラムが、* の場合は、すべてのカラムが指定されたとみなし、配列に値を設定します。
	 * 指定のカラムが、""(ゼロ文字列)の場合は、なにもカラムが指定されていないものとみなされ、
	 * 初期値の逆で埋められます。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param  clsm    指定のカラムをCSV形式で指定
	 * @param  def     設定する値
	 * @param  arrays  設定する boolean 配列
	 */
	private void setBooleanArray( final String clsm , final boolean def , final boolean[] arrays ) {
		if( clsm != null ) {
			// 書き込み制御の許可 をカラム単位で設定。* の場合は、全カラム許可
			if( "*".equals( clsm ) ) {
				Arrays.fill( arrays,def );
			}
			else {
				Arrays.fill( arrays,!def );
				String[] clmNames = StringUtil.csv2Array( clsm );
				for( int i=0; i<clmNames.length; i++ ) {
					if( clmNames[i] != null && clmNames[i].length() > 0 ) {
						int no = table.getColumnNo( clmNames[i] );
						if( no >= 0 ) { arrays[ no ] = def; }
					}
				}
			}
		}
	}

	/**
	 * ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  rtnKeys ogPopupで値を返すカラム文字列(CSV形式)
	 */
	public void setPopupReturnKeys( final String rtnKeys ) {
		// このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
		// いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
		// するようにします。(エラーにしません)
	}

	/**
	 * table要素に対して class 属性を設定します。
	 *
	 * 従来の システムリソースでのテーブルレイアウトの設定を廃止し、
	 * CSSファイルで、指定するように変更しています。
	 * これに伴い、CSSファイルのキーとして、クラス属性を出力します。
	 * view(または、出力されるtableタグ)のレイアウトは、このクラス属性で
	 * 指定することが可能になります。
	 * 初期値は、viewTable です。
	 *
	 * @og.rev 4.0.0 (2007/04/16) 新規追加
	 *
	 * @param   cls String class属性を表す文字列
	 */
	public void setTableClass( final String cls ) {
		if( cls != null ) {
			clazz = cls;
		}
	}

	/**
	 * table要素に対して class 属性を返します。
	 *
	 * @og.rev 4.0.0 (2007/04/16) 新規追加
	 *
	 * @return   String class属性を表す文字列
	 */
	protected String getTableClass() {
		return clazz;
	}
	
	/**
	 * outerのIDを持つDIV要素にheightを指定します
	 *
	 * ビューの高さ、幅を指定できるようにします。
	 * これによって１画面に２つのビューを出力する事も可能となります。
	 * 初期値はnullです。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param   high String ビューの高さ
	 */
	public void setHeight( final String high ) {
		height = high;
	}
	
	/**
	 * outerのIDを持つDIV要素にwidthを指定します
	 *
	 * ビューの高さ、幅を指定できるようにします。
	 * これによって１画面に２つのビューを出力する事も可能となります。
	 * 初期値はnullです。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param   wide String ビューの高さ
	 */
	public void setWidth( final String wide ) {
		width = wide;
	}

	/**
	 * リソースマネージャを設定します。
	 * クロス集計時に、useColumnHeader="true"とした場合のみ設定されます。
	 *
	 * @og.rev 4.0.0.0 新規作成
	 *
	 * @param	res リソースマネージャー
	 */
	public void setResourceManager( final ResourceManager res ) {
		resourceManager = res;
	}

	/**
	 * リソースマネージャを取得します。
	 *
	 * @og.rev 4.0.0.0 新規作成
	 *
	 * @return	ResourceManager リソースマネージャー
	 */
	protected ResourceManager getResourceManager() {
		return resourceManager;
	}

	/**
	 * 改廃Cのついていない行をスキップするかどうか指定します。
	 *
	 * "true" で、改廃Cのついた行をスキップします。
	 * デフォルトはfalseです。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) 新規追加
	 *
	 * @param	sne 改廃Cのついていない行をスキップするかどうか（true:スキップする/false:スキップしない）
	 */
	public void setSkipNoEdit( final boolean sne ) {
		skipNoEdit = sne;
	}

	/**
	 * 改廃Cのついていない行をスキップするかどうか指定します。
	 *
	 * skipNoEdit="true"でかつ、編集されていない(改廃Cがついていない)場合のみ
	 * trueを返します。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) 新規追加
	 *
	 * @param	row int
	 * @return	スキップ対象行か
	 */
	protected boolean isSkipNoEdit( final int row ) {
		return skipNoEdit && ( table.getModifyType( row ) == null || table.getModifyType( row ).length() == 0 );
	}

	/**
	 * 画面遷移なしモードに対応した形で処理を行うかを指定します。
	 *
	 * "true" で、画面遷移なしモードに対応します。
	 * デフォルトはfalseです。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 *
	 * @param	flag 画面遷移なしモードに対応するかどうか（true:対応する/false:対応しない）
	 */
	public void setNoTransition( final boolean flag ) {
		noTransition = flag;
	}

	/**
	 * 画面遷移なしモードに対応した形で処理を行うかを返します。
	 *
	 * "true" で、画面遷移なしモードに対応します。
	 * デフォルトはfalseです。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 *
	 * @return	画面遷移なしモードに対応するかどうか（true:対応する/false:対応しない）
	 */
	protected boolean isNoTransition() {
		return noTransition;
	}

	/**
	 * 該当行の行番号と改廃Cを出力します。
	 * 出力形式は、 rid="[行番号]" kh="[改廃C]" です。
	 * 改廃Cが付加されていない場合は、改廃Cがnullの場合は、kh属性は出力されません。
	 * (画面遷移なしモードで使用します)
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 *
	 * @param	row 行番号（ ０から始める ）
	 * @return	該当行の行番号と改廃C
	 */
	protected String getHiddenRowValue( final int row ) {
		String kh = "";
		if( table.getModifyType( row ) != null && table.getModifyType( row ).length() > 0 ) {
			kh = " " + hiddenCdkhKey + "=\"" + table.getModifyType( row ) + "\"";
		}
		return " " + hiddenRowKey + "=\"" + row + "\"" + kh;
	}

	/**
	 * ビューで表示したカラムの一覧をカンマ区切りで返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return String ビューで表示したカラムの一覧
	 */
	public String getViewClms() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		for( int i=0; i<clmDisplay.length; i++ ) {
			if( clmDisplay[i] && !( rowWritableFlag && i==0 ) ) {
				if( buf.length() > 0 ) { buf.append( ',' ); }
				buf.append( dbColumn[i].getName() );
			}
		}
		return buf.toString();
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return boolean 表示項目の編集(並び替え)が可能かどうか(true:可能)
	 */
	public boolean isEditable() {
		return true;
	}

	/**
	 * このViewFormが作成された画面IDをセットします。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @param gamenId 画面ID
	 */
	public void setGamenId( final String gamenId ) {
		this.gamenId = gamenId;
	}

	/**
	 * このViewFormが作成された画面IDを返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @return String 画面ID
	 */
	public String getGamenId() {
		return gamenId;
	}

	/**
	 * カラムが一括入力アシスト機能を利用するかどうかを返します。
	 *
	 * 条件は、一括入力アシストカラムで、かつ、書き込み許可(isColumnWritable(clm)==true)の場合とする。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	column int
	 * @return	利用する(true)／しない（false)
	 */
	protected boolean isClmBulkSet( final int column ) {
		return clmBulkSet[ column ] && clmWritable[ column ];
	}

	/**
	 * 一括入力カラムのアシスト機能を利用するカラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムの書き込み時に、一括入力可能なカラムをヘッダーに表示し、
	 * 一括登録できる機能を提供します。
	 * この機能は、jsp/common/bulkSet.js JavaScript とともに提供されます。
	 * IE については、クリップボードも利用できます。Ctrl-C と Ctrl-V でそれぞれ
	 * 一連のカラムの値の取り出しと書き込みが可能になります。
	 * "*" を指定すると、すべてのカラムを(columnBulkSet)指定したことになります。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	columnName String 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 */
	public void setColumnBulkSet( final String columnName ) {
		setBooleanArray( columnName,true,clmBulkSet );
	}
}
