/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import java.io.FileInputStream;
import java.io.IOException;

import javax.mail.internet.MimeUtility;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.KanaFilter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * サーバー管理ファイルをダウンロードする場合に使用する、サーブレットです。
 *
 * 引数（URL)に指定のファイルをサーバーからクライアントにダウンロードさせます。
 * file には、サーバーファイルの物理アドレスを指定します。相対パスを使用する場合は、
 * コンテキストルート(通常、Tomcatでは、G:\webapps\dbdef2\ など)からのパスと判断します。
 * name には、クライアントに送信するファイル名を指定します。ファイル名を指定しない場合は、
 * サーバーの物理ファイルのファイル名が代わりに使用されます。
 * 日本語ファイル名は、すべて UTF-8化して処理します。指定するファイルに日本語が含まれる
 * 場合は、URLエンコードを行ってください。
 * ファイル名の指定に、Content-disposition 属性を使用していますが、Internet Explorer の
 * 仕様で、attachment で指定すると開くボタンが使えません。("ファイル名" が見つかりません。)
 * この実装では、inline を使用するため、新たな画面で開きたい場合は、リンクにターゲット属性を
 * 指定してください。
 * http://support.microsoft.com/?scid=kb;ja;436605&spid=2073&sid=269
 *
 * 一般的なサーブレットと同様に、デプロイメント・ディスクリプタ WEB-INF/web.xml に、
 * servlet 要素と そのマッピング(servlet-mapping)を定義する必要があります。
 *
 *     &lt;servlet&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;servlet-class&gt;org.opengion.hayabusa.servlet.FileDownload&lt;/servlet-class&gt;
 *     &lt;/servlet&gt;
 *
 *     &lt;servlet-mapping&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;url-pattern&gt;/jsp/fileDownload&lt;/url-pattern&gt;
 *     &lt;/servlet-mapping&gt;
 *
 * 一般には、http://:ポート/システムID/jsp/fileDownload?file=サーバー物理ファイル&name=ファイル名
 * 形式のURL でアクセスします。
 *
 * @og.rev 3.8.1.1 (2005/11/21) 新規追加
 * @og.group その他機能
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class FileDownload extends HttpServlet {
	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	// 拡張子contentType対応テーブル
	private final static String CONTENT_TYPE_TABLE[][] = {
		{"jpg", "image/pjpeg"	},
		{"gif", "image/gif"		},
		{"txt", "text/plain"	},
//		{"xls", "application/vnd.ms-excel"}
		// OpenDocument追加
		{"xls", "application/vnd.ms-excel"},
		{"odp", "application/vnd.oasis.opendocument.presentation"}, // 4.3.5.5 (2008/03/08)
		{"ods", "application/vnd.oasis.opendocument.spreadsheet"}, // 4.3.5.5 (2008/03/08)
		{"odt", "application/vnd.oasis.opendocument.text"} // 4.3.5.5 (2008/03/08)
	};
	private final static int EXTENTION	 = 0;
	private final static int CONTENT_TYPE= 1;

	/**
	 * GET メソッドが呼ばれたときに実行します。
	 *
	 * 処理は、doPost へ振りなおしています。
	 *
	 * @param request HttpServletRequest
	 * @param response HttpServletResponse
	 *
	 * @og.rev 3.8.1.2 (2005/12/19) 半角カナ-全角カナ変換機能の追加
	 *
	 * @throws ServletException
	 * @throws IOException
	 */
	public void doGet( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {
		doPost( request,response );
	}

	/**
	 * POST メソッドが呼ばれたときに実行します。
	 *
	 * file 引数の サーバー物理ファイルを、クライアントにストリーム化して返します。
	 * name 引数があれば、その名前のファイル名でクライアントがファイルセーブできるように
	 * します。name 引数がなければ、そのまま物理ファイル名が使用されます。
	 * サーバー物理ファイル名が、相対パスの場合、コンテキストルートに対する相対パスになります。
	 * (例：G:\webapps\dbdef2\ など)
	 * 
	 * @og.rev 5.3.2.0 (2011/02/01) 日本語ファイル名が正しく処理できないバグを修正
	 *
	 * @param request HttpServletRequest
	 * @param response HttpServletResponse
	 *
	 * @throws ServletException
	 * @throws IOException
	 */
	public void doPost( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {

		// 3.8.1.2 (2005/12/19) 半角カナ-全角カナ変換機能の追加
		boolean hanzenFlag = HybsSystem.sysBool( "USE_FILEDOWNLOAD_HAN_ZEN" );

		String reqFilename = request.getParameter( "file" );
		String newFilename = request.getParameter( "name" );

		// クライアント側の文字エンコーディングをUTF-8に変換
		reqFilename = new String( reqFilename.getBytes("ISO-8859-1"), "UTF-8" );

		// 相対パスを絶対パスに変換。ファイルセパレータも正規化されています。
		reqFilename = HybsSystem.url2dir( reqFilename );

		// 拡張子からcontentTypeを獲得
		String contentType = getContentType( reqFilename );
		// contentTypeを出力
		response.setContentType( contentType );

		// 表示ファイル名の指定
		if( newFilename == null || newFilename.length() == 0 ) {
			newFilename = getFileName( reqFilename );
		}
		else {
			newFilename = new String( newFilename.getBytes("ISO-8859-1"), "UTF-8" );
		}

		// 3.8.1.2 (2005/12/19) 半角カナを全角カナに置き換えます。ファイルダイアログの文字化け仮対応
		if( hanzenFlag ) {
			newFilename = KanaFilter.han2zen( newFilename );
		}

		// 5.3.2.0 (2011/02/01) 日本語ファイル名が正しく処理できないバグを修正
//		newFilename = StringUtil.urlEncode( newFilename );
		if( request.getHeader( "User-Agent" ).indexOf( "MSIE" ) == -1 ) {
			newFilename = MimeUtility.encodeWord( newFilename, "UTF-8", "B" );
		}
		else {
			newFilename = StringUtil.urlEncode( newFilename );
		}

		// ファイル名の送信( attachment部分をinlineに変更すればインライン表示 )
		response.setHeader( "Content-disposition", "inline; filename=\"" + newFilename + "\"" );

		// ファイル内容の出力
		FileInputStream     fin = null;
		ServletOutputStream out = null;
		try {
			fin = new FileInputStream( reqFilename );
			out = response.getOutputStream();

			// ファイル読み込み用バッファ
			byte buffer[]  = new byte[4096];
			int size;
			while((size = fin.read(buffer))!=-1) {
				out.write(buffer,0, size);
				out.flush();
			}
		}
		finally {
			Closer.ioClose( fin );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			Closer.ioClose( out );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}

	/**
	 * アドレス名から拡張子を取り出します。
	 *
	 * アドレス名の後ろから、"." 以降を拡張子として切り取ります。
	 * 拡張子が存在しない場合（指定のファイル名に "." が含まれない場合)は
	 * ゼロ文字列("")を返します。
	 *
	 * @param fileAddress String アドレス名
	 * @return String 拡張子
	 */
	private String getExtention( final String fileAddress ) {
		int idx = fileAddress.lastIndexOf('.');
		if( idx!=-1 ) { return fileAddress.substring( idx+1 ); }
		return "";
	}

	/**
	 * アドレス名からファイル名を取り出します。
	 *
	 * アドレス名の後ろから、ファイルセパレータ以降をファイル名として切り取ります。
	 * ファイルセパレータが存在しない場合はアドレス名をそのまま返します。
	 * ここでは、OS毎に異なるファイルセパレータを統一後に処理してください。
	 *
	 * @param fileAddress String アドレス名
	 * @return String ファイル名
	 */
	private String getFileName( final String fileAddress ) {
		int idx = fileAddress.lastIndexOf( HybsSystem.FS );
		if( idx!=-1 ) { return fileAddress.substring( idx+1 ); }
		return fileAddress;
	}

	/**
	 * アドレス名から対応するコンテンツタイプを取り出します。
	 *
	 * アドレス名から、ファイル拡張子を取り出し、対応するコンテンツタイプを返します。
	 * コンテンツタイプは、CONTENT_TYPE_TABLE 配列に定義している中から検索して返します。
	 * 存在しない場合は、"application/octet-stream" を返します。
	 *
	 * @param fileAddress String アドレス名
	 * @return String コンテンツタイプ
	 */
	private String getContentType( final String fileAddress ) {
		String extention = getExtention( fileAddress );
		for( int j=0; j<CONTENT_TYPE_TABLE.length; j++ ) {
			if( CONTENT_TYPE_TABLE[j][EXTENTION].equalsIgnoreCase( extention ) ) {
				return CONTENT_TYPE_TABLE[j][CONTENT_TYPE];
			}
		}
		return "application/octet-stream";
	}
}
