/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem ;
import org.opengion.hayabusa.db.DBColumn;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * query.jsp での検索ボタンを表示します。
 * 検索ボタン以外に、pageSize、maxRowCount、prevNext、timeView などの
 * 項目も表示可能です。
 * また、BODY部に登録した値は、table 要素として、td 部に追記されます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryButton /&gt;
 * ●body：あり
 *
 * ●使用例
 *    &lt;og:queryButton
 *        command     = "NEW"      NEW:通常の検索(初期値)／ENTRY:エントリ登録時
 *        type        = "ENTRY"    null:標準 , SELECT:検索ボタンのみ , ENTRY:登録ボタンのみ
 *        pageSize    = "100"      pageSize の初期値設定(初期値:HTML_PAGESIZE)
 *        maxRowCount = "100"      maxRowCount の初期値設定(初期値:DB_MAX_ROW_COUNT)
 *        useCrear    = "true"     true:JavaScriptによる初期化／false:通常のresetボタン
 *        scope       = "request"  null(初期値)／session:セッション登録／request:リクエスト登録
 *        msg         = "MSG0010"  検索ボタンに表示するメッセージID（初期値:MSG0010）
 *        accesskey   = "F"        ショートカットキー文字(初期値:F)
 *        usePrevNext = "true"     Prev-Nextリンクを使用するかどうか(初期値:true)
 *        usePageSize = "true"     pageSize 引数を使用するかどうか(初期値:true)
 *        useRowCount = "true"     maxRowCount 引数を使用するかどうか(初期値:true)
 *        useTimeView = "true"     timeView表示(実行時間表示)を使用するかどうか(初期値:true)
 *    /&gt;
 *
 *  標準的な検索ボタン
 *    &lt;og:queryButton /&gt;
 *
 *  標準的な登録ボタン(command="NEW" で、query.jsp で使用するボタン)
 *    &lt;og:queryButton type="ENTRY" /&gt;
 *
 *  BODY 部に追加要素を表示
 *    &lt;og:queryButton &gt;
 *        &lt;td&gt;&lt;jsp:directive.include file="/jsp/common/timeViewImage.jsp" /&gt;&lt;/td&gt;
 *    &lt;/og:queryButton &gt;
 *
 * @og.rev 4.0.0 (2005/01/31) 新規作成
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class QueryButtonTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String CR = HybsSystem.CR;

	private static final String COMMAND	=
		"<input type=\"hidden\" name=\"command\" value=\"NEW\" id=\"h_cmd\" />";

	private static final String JSP = HybsSystem.sys( "JSP" );

	// 4.2.1.0 (2008/04/07) TDにIDを追加
	// 4.3.1.0 (2008/09/03) 戻る・進むボタンをsubmitにより実行する。
	private static final String PREV_NEXT =
		"<td id=\"prev-next\"><a href=\"forward.jsp?command=FIRST\" target=\"RESULT\" >"		+ CR +
		"  <img src=\"" + JSP + "/image/FIRST.gif\" alt=\"FIRST\" border=\"0px\" " + CR +
		"    onclick=\"return submitPrevNextForm('FIRST');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=PREV\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/PREV.gif\" alt=\"PREV\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('PREV');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=NEXT\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/NEXT.gif\" alt=\"NEXT\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('NEXT');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=LAST\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/LAST.gif\" alt=\"LAST\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('LAST');\"/>"	+ CR +
		"</a></td>" + CR ;

	private static final String TIME_VIEW =		// 4.2.1.0 (2008/04/07) 直接色指定をやめる。
		"<td><table cellspacing=\"0\" cellpadding=\"0\">"	+ CR +
		"<tr>"												+ CR +
		"<td width=\"20px\"> </td>"							+ CR +
		"<td width=\"0px\" id=\"queryTbl\" title=\"\" ></td>" + CR +	// #ff0000
		"<td width=\"0px\" id=\"viewTbl\"  title=\"\" ></td>" + CR +	// #0000ff
		"<td width=\"0px\" id=\"pageTbl\"  title=\"\" ></td>" + CR +	// #00ff00
		"<td width=\"10px\"></td>"						+ CR +
		"<td style=\"background-color:transparent;\" id=\"totalTbl\"></td>" + CR +
		"</tr>"											+ CR +
		"</table></td>" + CR ;
	
	// 4.3.6.0 (2009/04/01) 件数をボタンで開閉させる
	private static final String COUNT_SWITCH=
		"<td>" +
		"  <img id=\"queryHide\" src=\"" + JSP + "/image/expand_r.png\" alt=\"Expand\" border=\"0px\" "+
				" onClick=\"hideClass( event, this, 'hide', 'expand_r', 'shrink_r' )\"> "	+ CR +
		"</td>";

	private String	command		= "NEW";
	private String	pageSize	= null;
	private String	maxRowCount	= null;
	private String	accesskey	= "F";
	private boolean	useCrear	= true;
	private String	type		= null;		// null:標準 , SELECT:検索ボタンのみ , ENTRY:登録ボタンのみ
	private String	scope		= null;
	private boolean	usePrevNext	= true;
	private boolean	usePageSize	= true;
	private boolean	useRowCount	= true;
	private boolean	useTimeView	= true;
	private String	body		= null;
	private String	clearAcskey	= "C";

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 *
	 * @return  int 後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = nval( getBodyString(),body );

		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append( getHiddenScope() ).append( CR );

		buf.append( "<table border=\"0\" frame=\"border\" summary=\"submit\" ><tr>" );
		buf.append( CR );
		buf.append( "<td>" ).append( getSubmit()	).append( "</td>" ).append( CR );
		buf.append( "<td>" ).append( getReset()		).append( "</td>" ).append( CR );
		buf.append( "<td>" ).append( getCommand()	).append( "</td>" ).append( CR );
		buf.append( "<td>" ).append( getGamenId()	).append( "</td>" ).append( CR );
		if( usePrevNext ) { buf.append( PREV_NEXT		).append( CR ); }
		buf.append( "<td>" ).append( "<span id=\"query-button-append\"></span>").append( "</td>" ).append( CR ); //4 3.6.0 (2009/04/01)
		// if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
		// if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
		if( body != null ) {
			buf.append( "<td>" ).append( body ).append( "</td>" ).append( CR );
		}
		if( usePageSize || useRowCount){ buf.append( COUNT_SWITCH ).append( CR ); } // 4.3.6.0 (2009/04/01)
		if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
		if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
		if( useTimeView ) { buf.append( TIME_VIEW ).append( CR ); }
		 buf.append( "</tr></table>" ).append( CR );

		jspPrint( buf.toString() );
		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= "NEW";
		pageSize	= null;
		maxRowCount	= null;
		accesskey	= "F";
		useCrear	= true;
		type		= null;
		scope		= null;
		usePrevNext	= true;
		usePageSize	= true;
		useRowCount	= true;
		useTimeView	= true;
		clearAcskey	= "C"; // 5.3.5.0 (2011/05/01) クリアボタンアクセスキー対応
	}

	/**
	 * サブミットボタンを作成します。
	 *
	 * @return String サブミットボタン
	 */
	private String getSubmit() {
		String msglbl = getLongLabel();		// 4.0.0 (2005/01/31)
		if( msglbl == null ) {
			if( "ENTRY".equalsIgnoreCase( type ) ) {
				msglbl = getLabel( "ENTRY" );
			}
			else {
				msglbl = getLabel( "QUERY" );
			}
		}

		String rtn = "<button type=\"submit\" accesskey=\"" + accesskey + "\">" +
					msglbl + "(" + accesskey + ")</button>" ;

		return rtn;
	}

	/**
	 * リセットボタンを作成します。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 5.0.2.0 (2009/11/01) 戻るボタンが出力されている状態で、クリアボタンを押すと、戻るボタンが消える不具合を修正
	 * @og.rev 5.3.5.0 (2011/05/01) クリアボタンにもアクセスキーが付加できるように対応
	 *
	 * @return String リセットボタン
	 */
	private String getReset() {
		final String rtn ;

		if( useCrear ) {
			rtn = "<button onClick=\"window.open('" + JSP + "/" + getGUIInfoAttri("ADDRESS") +
//				"/query.jsp?GAMENID=" + getGUIInfoAttri("KEY") + "','QUERY')\" >" +
				"/query.jsp?GAMENID=" + getGUIInfoAttri("KEY") + "&command=NEW','QUERY')\" " +
				"accesskey=\"" + clearAcskey + "\">" +
//				getResource().getMessage( "MSG0011" ) + "</button>" ;
//				getResource().getLabel( "MSG0011" ) + "</button>" ;
				getResource().getLabel( "MSG0011" ) + "(" + clearAcskey + ")" + "</button>" ;
		}
		else {
			rtn = "<button type=\"reset\" accesskey=\"" + clearAcskey + "\">" +
//				getResource().getMessage( "MSG0011" ) + "</button>" ;
				getResource().getLabel( "MSG0011" ) + "(" + clearAcskey + ")" + "</button>" ;
		}

		return rtn;
	}

	/**
	 * コマンドを出力する hidden を作成します。
	 *
	 * @return String コマンドを出力する hidden
	 */
	private String getCommand() {
		final String rtn ;

		if( "NEW".equals( command ) ) {
			rtn = COMMAND;
		}
		else {
			rtn = "<input type=\"hidden\" name=\"command\" value=\"" +
						command + "\" id=\"h_cmd\" />" ;
		}

		return rtn;
	}

	/**
	 * スコープを出力する hidden を作成します。
	 *
	 * @return String スコープを出力する hidden
	 */
	private String getHiddenScope() {
		String rtn = "";

		if( scope != null ) {
			rtn = "<input type=\"hidden\" name=\"scope\" value=\"" +
						scope + "\" />" ;
		}

		return rtn;
	}

	/**
	 * 画面IDを出力する hidden を作成します。
	 *
	 * @return String 画面IDを出力する hidden
	 */
	private String getGamenId() {
		String rtn = "<input type=\"hidden\" name=\"GAMENID\" value=\"" +
						getGUIInfoAttri("KEY") + "\" />";

		return rtn;
	}

	/**
	 * ページサイズプルダウンメニューを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 * @return String ページサイズプルダウンメニュー
	 */
	private String getPageSize() {
//		String rtn = "";

//		if( usePageSize ) {
			if( pageSize == null ) {
				String PAGESIZE  = sys( "HTML_PAGESIZE" );
				pageSize = nval( getRequestValue( "pageSize" ),PAGESIZE ) ;
			}

			DBColumn column = getDBColumn( "pageSize" );
			String tag = column.getEditorValue( pageSize );

			// return ( "<td id=\"label\">" +
			return ( "<td class=\"label pageSize hide\">" + // 4.3.6.0 (2008/04/01) class出力
						column.getLongLabel() +
						":</td><td class=\"pageSize hide\">" + tag + "</td>" ) ;
//		}

//		return rtn;
	}

	/**
	 * 検索時の最大検索数のプルダウンメニューを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 * @return String 最大検索数のプルダウンメニュー
	 */
	private String getMaxRowCount() {
//		String rtn = "";

//		if( useRowCount ) {
			if( maxRowCount == null ) {
				String ROW_COUNT = sys( "DB_MAX_ROW_COUNT" );
				maxRowCount = nval( getRequestValue( "maxRowCount" ), ROW_COUNT ) ;
			}

			DBColumn column = getDBColumn( "maxRowCount" );
			String tag = column.getEditorValue( maxRowCount );

			// return ( "<td id=\"label\">" +
			return ( "<td class=\"label maxRowCount hide\">" + // 4.3.6.0 (2008/04/01) class出力
						column.getLongLabel() +
						":</td><td class=\"maxRowCount hide\">" + tag + "</td>" ) ;
//		}
//		return rtn;
	}
	
	/**
	 * 【TAG】リクエストとして送信するコマンドを指定します(初期値:NEW)。
	 *
	 * @og.tag
	 *  NEW:通常の検索(初期値)／ENTRY:エントリ登録時
	 *
	 * @param   cmd String 送信するコマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】JavaScriptによる初期化ボタンを使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常の初期化ボタンでは、HTMLが作成された状態に戻すだけです。例えば、
	 * 引数付きで設定された場合、その引数が設定されている状態へ戻します。
	 * このJavaScriptによる初期化ボタンの場合(trueの場合)は、JavaScriptにより
	 * INIT状態でもう一度QUERYフレームを呼び出しますので、指定された引数も
	 * クリアされます。
	 * 初期値は、true:JavaScriptによる初期化です。
	 *
	 * @param   flag String true:JavaScriptによる初期化／false:通常のresetボタン
	 */
	public void setUseCrear( final String flag ) {
		useCrear = nval( getRequestParameter( flag ),useCrear );
	}

	/**
	 * 【TAG】典型的なボタンの形状に設定します(初期値:null)。
	 *
	 * @og.tag
	 * クエリーボタンでは、検索ボタン以外に、pageSize、maxRowCount、
	 * prevNext、timeView などの項目も表示可能です。
	 * これらを、個別に設定可能ですが、通常使用するタイプとして、３タイプを
	 * 用意します。これは、設定状態の初期値といえます。
	 * null  :通常の検索ボタンで、フル装備の指定です。
	 * SELECT:１件のみ検索するようなケースで使用します。
	 *        検索ボタンとクリアボタンのみを表示します。
	 * ENTRY :いきなり登録するようなケースで使用します。
	 *        登録ボタンとクリアボタンのみを表示します。(command="NEW" です。)
	 *
	 * @param   tp String タイプ指定(null,SELECT,ENTRY)
	 */
	public void setType( final String tp ) {
		type = nval( getRequestParameter( tp ),type );

		if( "SELECT".equalsIgnoreCase( type ) ) {
			accesskey	= "F";
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
		else if( "ENTRY".equalsIgnoreCase( type ) ) {
			accesskey	= "W";
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
	}

	/**
	 * 【TAG】リクエストとして送信するスコープを指定します(初期値:null)。
	 *
	 * @og.tag
	 * リクエスト時に、付与する引数情報のうちの、スコープを指定します。
	 * これは、検索専用画面の場合は、scope="request" にする事で、メモリの
	 * 無駄な確保を行わずに、検索結果を表示できます。
	 * ただし、その結果に対して、NEXT-PREV や、ヘッダーソートリンクなどの
	 * 機能が使えなくなります。
	 * 通常は、何も指定しない状態(null)で、RESULT画面の設定に任せるのが良いでしょう。
	 *
	 * @param   scp String session:セッション登録(初期値)／request:リクエスト登録
	 */
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * 【TAG】検索ボタンに使用するショートカットキー文字を指定します(初期値:F)。
	 *
	 * @og.tag
	 * ボタンを、Alt-XX で実行できるようにするショートカットキーを設定します。
	 * 標準では、検索ボタンは、"F" に設定していますので、Alt-F となります。
	 * 現仕様では、ショートカットキーを無くすことは出来ません。
	 * 初期値は、Find の "F" です。
	 *
	 * @param   accsKey String ショートカットキー文字(初期値:F)
	 */
	public void setAccesskey( final String accsKey ) {
		accesskey = nval( getRequestParameter( accsKey ),accesskey );
	}

	/**
	 * 【TAG】クリアボタンに使用するショートカットキー文字を指定します(初期値:C)。
	 *
	 * @og.tag
	 * ボタンを、Alt-XX で実行できるようにするショートカットキーを設定します。
	 * 標準では、検索ボタンは、"C" に設定していますので、Alt-C となります。
	 * 現仕様では、ショートカットキーを無くすことは出来ません。
	 * 初期値は、Clear の "C" です。
	 *
	 * @param   accsKey String クリアボタンのショートカットキー文字(初期値:C)
	 */
	public void setClearAccesskey( final String accsKey ) {
		clearAcskey = nval( getRequestParameter( accsKey ),clearAcskey );
	}

	/**
	 * 【TAG】進む／戻るのprev_next リンクを表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 進む／戻るリンクは、RESULT画面にコマンド(FIRST,PREV,NEXT,LAST) という
	 * コマンドを送る事で実現している画像リンクです。
	 * PREV_NEXT が存在しない画面では、消去しておく必要があります。
	 * 初期値は、true(表示する)です。
	 *
	 * @param   flag String prev_next リンクを表示するかどうか
	 */
	public void setUsePrevNext( final String flag ) {
		usePrevNext = nval( getRequestParameter( flag ),usePrevNext );
	}

	/**
	 * 【TAG】pageSize 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * pageSize 引数は、RESULT画面のviewタグで使用する 一回の表示に
	 * 何件のデータを表示させるかを指定する引数です。
	 * この件数が多いと、多くのデータを見ることが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、PREV_NEXT を多用することになり、
	 * 目的の行を探し出しにくくなります。
	 * pageSize の値は、pageSizeパラメータで画面単位に指定するか、システム
	 * パラメータの HTML_PAGESIZE でシステム単位に指定できます。
	 * 初期値は、true(使用する)です。
	 *
	 * @param   flag String pageSize引数を使用するかどうか
	 * @see     #setPageSize( String )
	 */
	public void setUsePageSize( final String flag ) {
		usePageSize = nval( getRequestParameter( flag ),usePageSize );
	}

	/**
	 * 【TAG】maxRowCount 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * maxRowCount 引数は、RESULT画面のqueryタグで使用する 一回の検索時に
	 * 何件のデータを取得させるかを指定する引数です。
	 * この件数が多いと、多くのデータを検索することが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、目的の行まで探し出せないかもしれません。
	 * maxRowCount の値は、rowCountパラメータで画面単位に指定するか、システム
	 * パラメータの DB_MAX_ROW_COUNT でシステム単位に指定できます。
	 *
	 * 初期値は、true(使用する)です。
	 *
	 * @param   flag String maxRowCount引数を使用するかどうか
	 * @see     #setMaxRowCount( String )
	 */
	public void setUseRowCount( final String flag ) {
		useRowCount = nval( getRequestParameter( flag ),useRowCount );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するバーイメージが表示されます。
	 * これは、DB検索、APサーバー処理、画面表示の各処理時間をバーイメージで
	 * 表示させる機能です。処理時間の目安になります。
	 * 初期値は、true です。
	 *
	 * @param   flag String TimeView表示をするかどうか(初期値:true)
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * 【TAG】pageSize プルダウンメニューの初期値を指定します(初期値:HTML_PAGESIZE)。
	 *
	 * @og.tag
	 * プルダウンメニューを表示する場合の初期値を指定します。
	 * pageSize の初期値設定(初期値:HTML_PAGESIZE)
	 *
	 * @param   pSize String プルダウンメニューの初期値
	 * @see     #setUsePageSize( String )
	 */
	public void setPageSize( final String pSize ) {
		pageSize = nval( getRequestParameter( pSize ),pageSize );
	}

	/**
	 * 【TAG】maxRowCount プルダウンメニューの初期値を指定します(初期値:DB_MAX_ROW_COUNT)。
	 *
	 * @og.tag
	 * プルダウンメニューを表示する場合の初期値を指定します。
	 * maxRowCount の初期値設定(初期値:DB_MAX_ROW_COUNT)
	 *
	 * @param   maxSize String maxRowCountプルダウンメニューの初期値
	 * @see     #setUseRowCount( String )
	 */
	public void setMaxRowCount( final String maxSize ) {
		maxRowCount = nval( getRequestParameter( maxSize ),maxRowCount );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "command"		,command		)
				.println( "pageSize"	,pageSize		)
				.println( "maxRowCount"	,maxRowCount	)
				.println( "accesskey"	,accesskey		)
				.println( "scope"		,scope			)
				.println( "useCrear"	,useCrear		)
				.println( "usePrevNext"	,usePrevNext	)
				.println( "usePageSize"	,usePageSize	)
				.println( "useRowCount"	,useRowCount	)
				.println( "useTimeView"	,useTimeView	)
				.println( "body"		,body			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
