/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

/**
 * Doclet を処理するプログラムで共通して使用される簡易メソッド群（ユーティリティクラス）です。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DocletUtil {

	/**
	 * すべてが staticメソッドなので、コンストラクタを呼び出さなくしておきます。
	 *
	 */
	private DocletUtil() {}

	/**
	 * 入力文字列の HTML 文字をフィルタリングします。
	 * "&" は、"&amp;#38;" 、"<" は、"&amp;lt;" 、">" は、"&amp;gt;" に
	 * 変換します。
	 *
	 * @param input String
	 * @return 変換後文字列
	 */
	public static String htmlFilter( final String input ) {
		if( input == null ) { return ""; }

		int len = input.length();

		StringBuilder rtn = new StringBuilder( len + 50 );
		char ch;
		for(int i=0; i<len; i++) {
			ch = input.charAt(i);
			switch( ch ) {
				case '<'  : rtn.append("&lt;");   break;
				case '>'  : rtn.append("&gt;");   break;
				case '"'  : rtn.append("&quot;"); break;
				case '\'' : rtn.append("&#39;");  break;
				case '&'  : rtn.append("&amp;");  break;
				case ';'  : rtn.append("&#59;");  break;
				default   : rtn.append(ch);
			}
		}
		return( rtn.toString() );
	}

	/**
	 * target 文字列に含まれる from 文字列を to 文字列に置き換えます。
	 *
	 * @param target String
	 * @param from String
	 * @param to String
	 * @return 変換後文字列
	 */
	public static String replace( final String target,final String from,final String to ) {
		if( target == null || from == null || to == null ) { return target; }
		StringBuilder strBuf = new StringBuilder( 200 );

		int start = 0;
		int end   = target.indexOf( from,start );
		while( end >= 0  ) {
			strBuf.append( target.substring( start,end ) );
			strBuf.append( to );
			start = end + from.length();
			end   = target.indexOf( from,start );
		}
		strBuf.append( target.substring( start ) );

		return strBuf.toString();
	}

	/**
	 * コメント部の文字列を取得します。
	 *
	 * @param cmnt String
	 * @return コメント文字列
	 */
	public static String commentText( final String cmnt ) {
		if( cmnt == null ) { return ""; }

		String rtn = cmnt;
		int indx = cmnt.indexOf( "{@value}" );
		if( indx >= 0 ) {
			rtn = cmnt.substring( indx+8 );	// {@value} 以前を削除
		}
		return htmlFilter( rtn );
	}

	/**
	 * セッターメソッドの setXXXX の set を削除し、次の文字を小文字化します。
	 * つまり、セッターメソッドから属性値を推測します。
	 * (超特殊処理)セッターメソッドのset以下２文字目が大文字の場合は、
	 * １文字目も大文字と考えて小文字化を行いません。
	 * 例えば、setSYS や setUSER など、RequestValueTag.javaに使用するケースです。
	 *
	 * @param target String
	 * @return オプション文字列
	 */
	public static String removeSetter( final String target ) {
		if( target != null && target.startsWith( "set" ) ) {
			char[] chs = target.toCharArray();
			if( chs.length > 4 && !( chs[4] >= 'A' && chs[4] <= 'Z' ) ) {
				chs[3] = Character.toLowerCase( chs[3] ) ;
			}
			return new String( chs,3,chs.length-3 );
		}
		return target;
	}

	/**
	 * オプション配列文字列より、指定のキーに対応するオプション値を返します。
	 *
	 * @param key String
	 * @param options String[][]
	 * @return オプション文字列
	 */
	public static String getOption( final String key , final String[][] options ) {
		String rtn = "";
		if( key == null || options == null ) { return rtn; }

		for( int i=0; i<options.length; i++ ) {
			if( key.equalsIgnoreCase( options[i][0] ) ) {
				rtn = options[i][1];
				break ;
			}
		}
		return rtn ;
	}
}
