/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.servlet.multipart.MultipartParser;
import org.opengion.hayabusa.servlet.multipart.Part;
import org.opengion.hayabusa.servlet.multipart.FilePart;
import org.opengion.hayabusa.servlet.multipart.ParamPart;

import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;

/**
 * ファイルをサーバーにアップロードする場合に使用されるマルチパート処理サーブレットです。
 *
 * 通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、取り出すことが可能です。
 *
 * ファイルをアップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * @og.group その他機能
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class MultipartRequest {
	private static volatile int dumyNewFileCnt = 1 ;	// 3.8.1.4 (2006/03/17)

	private final Map<String,List<String>> parameters	= new HashMap<String,List<String>>();
	private final Map<String,UploadedFile> files	= new HashMap<String,UploadedFile>();

	/**
	 * MultipartRequest オブジェクトを構築します。
	 *
	 * 引数として、ファイルアップロード時の保存フォルダ、最大サイズ、エンコード、
	 * 新しいファイル名などを指定できます。新しいファイル名は、アップロードされる
	 * ファイルが一つだけの場合に使用できます。複数のファイルを同時に変更したい
	 * 場合は、アップロードルールにのっとり、リクエストパラメータで指定してください。
	 *
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 *
	 * @param request HttpServletRequest
	 * @param saveDirectory String ファイルアップロードがあった場合の保存フォルダ名
	 * @param maxPostSize int  ファイルアップロード時の最大ファイルサイズ
	 * @param encoding String  ファイルのエンコード
	 * @param inputFilename String アップロードされたファイルの新しい名前
	 * @throws IOException
	 */
	public MultipartRequest(final HttpServletRequest request,
							final String saveDirectory,
							final int maxPostSize,
							final String encoding,
							final String inputFilename ) throws IOException {

		if(request == null) {
			throw new IllegalArgumentException("request cannot be null");
		}
		if(saveDirectory == null) {
			throw new IllegalArgumentException("saveDirectory cannot be null");
		}
		if(maxPostSize <= 0) {
			throw new IllegalArgumentException("maxPostSize must be positive");
		}

		// Save the dir
		File dir = new File(saveDirectory);

		// Check saveDirectory is truly a directory
		if(!dir.isDirectory()) {
			throw new IllegalArgumentException("Not a directory: " + saveDirectory);
		}

		// Check saveDirectory is writable
		if(!dir.canWrite()) {
			throw new IllegalArgumentException("Not writable: " + saveDirectory);
		}

		// Parse the incoming multipart, storing files in the dir provided,
		// and populate the meta objects which describe what we found
		MultipartParser parser = new MultipartParser(request, maxPostSize);
		if(encoding != null) {
			parser.setEncoding(encoding);
		}

		List<String> list = new ArrayList<String>();

		Part part;
		while ((part = parser.readNextPart()) != null) {
			String name = part.getName();
			if( part.isParam() && part instanceof ParamPart ) {
				ParamPart paramPart = (ParamPart)part;
				String value = paramPart.getStringValue();
				List<String> existingValues = parameters.get(name);
				if(existingValues == null) {
					existingValues = new ArrayList<String>();
					parameters.put(name, existingValues);
				}
				existingValues.add(value);
			}
			else if( part.isFile() && part instanceof FilePart ) {
				FilePart filePart = (FilePart)part;
				String fileName = filePart.getFilename();
				if(fileName != null) {
					list.add( name );		// 3.5.6.5 (2004/08/09)
					// 3.8.1.2 (2005/12/19) 仮ファイルでセーブする。
					String newName = String.valueOf( dumyNewFileCnt++ ) ;	// 3.8.1.4 (2006/03/17)
					filePart.setFilename( newName );
					filePart.writeTo(dir);
					files.put(name,
							  new UploadedFile( dir.toString(),
												newName,		// 3.8.1.2 (2005/12/19)
												fileName,
												filePart.getContentType()));
				}
				else {
					files.put(name, new UploadedFile(null, null, null, null));
				}
			}
			else {
				String errMsg = "Partオブジェクトが、ParamPartでもFilePartでもありません。"
							+ " class=[" + part.getClass() + "]";
				throw new RuntimeException( errMsg );
			}
		}

		// 3.5.6.5 (2004/08/09) 登録後にファイルをリネームします。
		int size = list.size();
		for( int i=0; i<size; i++ ) {
			String name = list.get(i);
			File file = getFile( name );

			String newName = inputFilename;
			if( newName == null && name != null ) {
				int adrs = name.lastIndexOf( HybsSystem.JOINT_STRING );	// カラム__行番号 の __ の位置
				if( adrs < 0 ) {
					newName = getParameter( name + "_NEW" );
				}
				else {
					newName = getParameter( name.substring( 0,adrs ) + "_NEW" + name.substring( adrs ) );
				}
			}

			// 3.8.1.3 (2006/02/06) 新ファイル名に拡張子がないとき
			// 旧ファイル名から拡張子取得し新ファイル名に文字列連結
			if( newName != null && newName.length() > 0 ) {
				// 新ファイル名から拡張子取得
				String newExt = getExtension( newName );
				if( newExt == null || newExt.length() == 0 ) {
					String oldExt = getExtension( getOriginalFileName( name ) );
					newName = new StringBuilder().append( newName ).append( "." ).append( oldExt ).toString();
				}
			}
			else {
				newName = getOriginalFileName( name );
			}

			// 3.8.1.2 (2005/12/19) 基本的にはすべてのケースでファイル名変更が発生する。
			if( file != null && newName != null && newName.length() > 0 ) {
				File newFile = new File( dir,newName );
				if( newFile.exists() && !newFile.delete() ) {
					String errMsg = "既存のファイル[" + newName + "]が削除できませんでした。";
					throw new RuntimeException( errMsg );
				}
//				file.renameTo( newFile );
				if( !file.renameTo( newFile ) ) {
					String errMsg = "所定のファイルをリネームできませんでした。[" + file + "]" ;
					throw new RuntimeException( errMsg );
				}
				UploadedFile fup = files.get( name );
				fup.setFilesystemName( newName );
			}
		}
	}

	/**
	 * リクエストパラメータの名前配列を取得します。
	 *
	 * @return String[]	リクエストパラメータの名前配列
	 */
	public String[] getParameterNames() {
		Set<String> keyset = parameters.keySet();
		return keyset.toArray( new String[keyset.size()] );
	}

	/**
	 * ファイルアップロードされたファイル群のファイル名配列を取得します。
	 *
	 * @return String[]	アップロードされたファイル名配列
	 */
	public String[] getFilenames() {
		Set<String> keyset = files.keySet();
		return keyset.toArray( new String[keyset.size()] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を取得します。
	 *
	 * @param name String	リクエストパラメータ名
	 * @return String	パラメータの値
	 */
	public String getParameter( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}
		return values.get(values.size() - 1);
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列型式で取得します。
	 * 
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 *
	 * @param name String	リクエストパラメータ名
	 * @return String[]	パラメータの値(配列)
	 */
	public String[] getParameters( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}
		return values.toArray( new String[0] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列(int)型式で取得します。
	 * 
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 * @og.rev 5.3.6.0 (2011/06/01) 配列値が""の場合にNumberFormatExceptionが発生するバグを修正
	 *
	 * @param name String	リクエストパラメータ名
	 * @return int[]	パラメータの値(int配列)
	 */
	public int[] getIntParameters( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}

//		int[] rtn = new int[values.size()];
//		for( int i=0; i<values.size(); i++ ) {
//			rtn[i] = Integer.valueOf( values.get(i) );
//		}

		// 5.3.6.0 (2011/06/01) ゼロストリング("")はint変換対象から予め除外する
		List<Integer> intVals = new ArrayList<Integer>();
		for( int i=0; i<values.size(); i++ ) {
			String s = values.get(i);
			if( s != null && s.length() > 0 ) {
				intVals.add( Integer.parseInt( s ) );
			}
		}
		if( intVals.isEmpty() ) {
			return null;
		}

		int[] rtn = new int[intVals.size()];
		for( int i=0; i<intVals.size(); i++ ) {
			rtn[i] = intVals.get(i).intValue();
		}

		return rtn;
	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから 登録されるファイル名を取得します。
	 *
	 * 登録されるファイル名とは、新たに書き換えられたファイル名のことです。
	 *
	 * @param name String	キー情報
	 * @return String	新たに書き換えられたファイル名
	 */
	public String getFilesystemName( final String name ) {
		UploadedFile file = files.get(name);
		return file.getFilesystemName();  // may be null
	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから アップロードされたファイル名を取得します。
	 *
	 * アップロードされたファイル名とは、オリジナルのファイル名のことです。
	 *
	 * @param name String	キー情報
	 * @return String	オリジナルのファイル名
	 */
	public String getOriginalFileName( final String name ) {
		UploadedFile file = files.get(name);
		return file.getOriginalFileName();  // may be null
	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから File オブジェクトを取得します。
	 *
	 * @param name String	キー情報
	 * @return File		File オブジェクト
	 */
	public File getFile( final String name ) {
		UploadedFile file = files.get(name);
		return file.getFile();  // may be null
	}

	/**
	 * ファイル名から 拡張子を取得します。
	 *
	 * @param fileName String	ファイル名
	 * @return String	拡張子
	 */
	private String getExtension( final String fileName ) {
		int index = fileName.lastIndexOf('.');
		if(index!=-1) {
			return fileName.substring(index + 1, fileName.length());
		}
		return "";
	}
}

/**
 * ファイルをサーバーにアップロードする場合に使用されるファイル管理内部クラスです。
 *
 * @og.group その他機能
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class UploadedFile {

	private String filename;
	private final String dir;
	private final String original;
	private final String type;

	/**
	 * アップロードファイルの管理オブジェクトを作成します。
	 *
	 * @param dir String	ファイルを保管するフォルダ
	 * @param filename String	ファイル名(置き換え後)
	 * @param original String	ファイル名(オリジナル)
	 * @param type String	コンテントタイプ
	 */
	UploadedFile( final String dir, final String filename, final String original, final String type) {
		this.dir = dir;
		this.filename = filename;
		this.original = original;
		this.type = type;
	}

	/**
	 * コンテントタイプを取得します。
	 *
	 * @return String	コンテントタイプ
	 */
	public String getContentType() {
		return type;
	}

	/**
	 * ファイル名(置き換え後)を取得します。
	 *
	 * @return String	ファイル名(置き換え後)
	 */
	public String getFilesystemName() {
		return filename;
	}

	/**
	 * ファイル名(置き換え後)をセットします。
	 *
	 * @param name String	ファイル名(置き換え後)
	 */
	public void setFilesystemName( final String name ) {
		filename = name;
	}

	/**
	 * ファイル名(オリジナル)を取得します。
	 *
	 * @return String	ファイル名(オリジナル)
	 */
	public String getOriginalFileName() {
		return original;
	}

	/**
	 * ファイル名(置き換え後)の File オブジェクトを取得します。
	 *
	 * @return File	File オブジェクト
	 */
	public File getFile() {
		if(dir == null || filename == null) {
			return null;
		}
		else {
			return new File(dir + File.separator + filename);
		}
	}
}
