/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.io.IOException;

import javax.servlet.http.HttpServletRequest;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.servlet.MultipartRequest;

/**
 * クライアントのファイルをサーバーにアップロードするタグです。
 *
 * アップロード後の属性は、DBTableModel に格納することにより、通常のデータと
 * 同様の取り出し方が可能です。
 * また、通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、{&#064;XXXX} 変数で取り出すことが可能です。
 *
 * この upload タグでは、アップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * 新ファイル名に拡張子が設定されていない場合は、オリジナルファイル名の拡張子をセット
 * します。
 *
 * @og.formSample
 * ●形式：&lt;og:upload fileURL="…" maxPostSize="…" /&gt;
 * ●body：なし
 *
 * ●使用例 :
 *
 * 【query.jsp】
 *       &lt;form method="POST" action="result.jsp" enctype="multipart/form-data" target="RESULT"&gt;
 *       &lt;table summary="layout" &gt;
 *       &lt;tr&gt;&lt;og:input type="text" name="submitter" value="{&#064;USER.JNAME}" size="20" msg="MSG0014" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;
 *           &lt;og:input type="file" name="file1"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file1_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file2"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file2_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file3"      size="30" msg="MSG0015" /&gt;
 *           &lt;og:input             name="file3_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:column name="writable"     value="false"  /&gt;
 *       &lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * 【result.jsp】
 *       &lt;og:upload
 *           fileURL     = "{&#064;USER.ID}"
 *       /&gt;
 *       &lt;br /&gt;
 *       &lt;og:message msg="MSG0003" comment="ファイルの登録が完了しました。" /&gt;
 *
 *       &lt;og:view
 *           command      = "NEW"
 *           viewFormType = "HTMLTable"
 *           writable     = "{&#064;writable}"
 *       /&gt;
 *
 *       &lt;table&gt;
 *       &lt;tr&gt;&lt;og:input name="submitter" value="{&#064;submitter}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="writable"  value="{&#064;writable}"  /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="directory" value="{&#064;directory}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1"     value="{&#064;file1}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1_NEW" value="{&#064;file1_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file1_ORG" value="{&#064;file1_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2"     value="{&#064;file2}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2_NEW" value="{&#064;file2_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file2_ORG" value="{&#064;file2_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3"     value="{&#064;file3}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3_NEW" value="{&#064;file3_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file3_ORG" value="{&#064;file3_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileUploadTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;		// 4.0.0 (2005/01/31)

	// 3.5.2.0 (2003/10/20) カラム名に、ISFILEを追加。
	private static final String[] names = new String[] { "KEY","VALUE","ISFILE" };
	private static final String   ENCODE = "UTF-8";			// 3.5.2.0 (2003/10/20) 廃止
	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String  filename    = null;						// 3.5.4.2 (2003/12/15)

	private int 	maxPostSize = 10*1024*1024;				// 最大ファイル容量  10MB
	private String	tableId 	= HybsSystem.TBL_MDL_KEY ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちのエンコード情報を取得する。
	 * @og.rev 3.5.2.0 (2003/10/20) scope 属性を追加。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 * @og.rev 5.3.7.0 (2011/07/01) エラーメッセージ内容変更
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
		HttpServletRequest request = (HttpServletRequest)getRequest();

		try {
			String directory = HybsSystem.url2dir( fileURL );
			File dir = new File(directory);
			if( ! dir.exists() && ! dir.mkdirs() ) {
				String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
				throw new HybsSystemException( errMsg );
			}

			// 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
			MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename );
			DBTableModel table = makeDBTable( multi );

			// 3.5.2.0 (2003/10/20) scope 属性を追加。
			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "FileUploadTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return (SKIP_PAGE);
			}

		} catch(IOException ex) {
//			String errMsg = "ファイルの取り扱い中にエラーが発生しました。"
//					+ toString() + HybsSystem.CR
//					+ "FileURL=" + fileURL + HybsSystem.CR
//					+ ex.getMessage();				// 5.1.8.0 (2010/07/01) errMsg 修正
			String errMsg = "ファイル登録エラー！"
				+ ex.getMessage() + HybsSystem.CR	// 5.3.7.0 (2011/07/01) errMsg 修正
				+ "(" + toString() + HybsSystem.CR
				+ "FileURL=" + fileURL + ")";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.1.1 (2003/03/06) columns を廃止
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 		= HybsSystem.sys( "FILE_URL" );
		maxPostSize		= 10*1024*1024;				// 最大ファイル容量  10MB
		tableId 		= HybsSystem.TBL_MDL_KEY ;
		filename		= null;						// 3.5.4.2 (2003/12/15)
	}

	/**
	 * ファイルアップロードの実行結果を DBTableModel に記述します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.0.1.1 (2003/03/06) request 情報から{&#064;XXXX} で値が取り出せる様に修正。
	 * @og.rev 3.5.2.0 (2003/10/20) カラム名（KEY,VALUE）に ISFILE を追加
	 * @og.rev 3.5.6.5 (2004/08/09) MultipartRequest 変更に伴なう修正(Enum変更、元ファイル名取得)
	 * @og.rev 3.5.6.6 (2004/08/23) 上記変更時のバグ修正。
	 * @og.rev 3.5.6.6 (2004/08/23) 元ファイルのキーを、XXXX_ORG にします。
	 * @og.rev 4.0.0.0 (2007/10/12) テーブルモデルの登録方法を変更
	 * @og.rev 5.3.2.0 (2011/02/01) チェック行のパラメーターはint配列側に変換して復元する。パラメーター名を保存する。
	 *
	 * @param	 multi	   MultipartRequest オブジェクト
	 * @return	 DBTableModel テーブルモデル
	 */
	private DBTableModel makeDBTable( final MultipartRequest multi ) {

		DBTableModel table = DBTableModelUtil.newDBTable();

		table.init( names.length );

		for( int i=0; i<names.length; i++ ) {
			DBColumn dbColumn = getDBColumn( names[i] );
			table.setDBColumn( i,dbColumn );
		}

		String[] values ;		// 4.0.0.0 (2007/10/12)

		String[] files = multi.getFilenames();		// 3.6.0.0 (2004/09/17)
		for( int i=0; i<files.length; i++ ) {
			String name = files[i];
			File fin = multi.getFile(name);
			if( fin != null ) {
				String val = multi.getFilesystemName(name);

				// "KEY","VALUE","ISFILE" の順にデータを作成します。
				values = new String[] { name, val, "1" };
				table.addColumnValues( values );
				setRequestAttribute( name,val );

				String orgName = name + "_ORG" ;
				val  = multi.getOriginalFileName(name);	// 注意：取得は、送信名

				// "KEY","VALUE","ISFILE" の順にデータを作成します。
				values = new String[] { orgName, val, "2" };
				table.addColumnValues( values );
				setRequestAttribute( orgName,val );

			}
		}

		// "KEY","VALUE","ISFILE" の順にデータを作成します。
		values = new String[] { "directory", fileURL, "0" };
		table.addColumnValues( values );
		setRequestAttribute( "directory",fileURL );

		String[] params = multi.getParameterNames();
		for( int i=0; i<params.length; i++ ) {
			String name = params[i];
			// 5.3.2.0 (2011/02/01) チェック行のパラメーターはint配列側に変換
			if ( HybsSystem.ROW_SEL_KEY.equals( name ) ) {
				setRequestAttribute( name,multi.getIntParameters(name) );
			}
			else {
				String val = multi.getParameter(name);

				// "KEY","VALUE","ISFILE" の順にデータを作成します。
				values = new String[] { name, val, "0" };
				table.addColumnValues( values );
				setRequestAttribute( name,val );
			}
		}

		// 5.3.2.0 (2011/02/01) パラメーター名を保存する。
		String[] names = new String[ files.length + params.length ];
	    System.arraycopy( files, 0, names, 0, files.length );
	    System.arraycopy( params, 0, names, files.length, params.length );
	    setParameterNames( names );

		return table ;
	}

	/**
	 * 【TAG】ファイルをアップロードするディレクトリを指定します(初期値:FILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、アップロードされたファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルURL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】最大転送サイズ(Byte)を指定します(初期値:10485760)。
	 *
	 * @og.tag
	 * 最大転送サイズを指定します。初期値は、10*1024*1024 = 10MB です。
	 * 指定は、Byte 単位で指定します。
	 *
	 * @og.rev 3.0.1.1 (2003/03/06) maxPostSize の設定バグ修正。
	 *
	 * @param	maxPS 最大転送サイズ
	 */
	public void setMaxPostSize( final String maxPS ) {
		maxPostSize = nval( getRequestParameter( maxPS ),maxPostSize );
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) nvalメソッドを適用するように変更。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】(通常使いません)ファイルを作成するときのファイル名をセットします。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * これは、複数同時にアップロードファイル名を変更する時に使用できません。
	 * 通常、アップロードされたファイル名を指定する場合、アップロードするinput タグの
	 * name 属性に指定する名称 ＋ "_NEW" というリクエスト値を同時に送信すれば、
	 * 内部的に関連付けて、ファイル名を更新します。
	 * その場合、クライアントより指定したファイル名は、name属性＋"_ORG" という
	 * リクエスト値として取得することが可能になります。
	 * name属性 には、最終的に設定されたファイル名がセットされています。
	 * いずれの値も、{&#064;name属性＋"_ORG"} や、{&#064;name属性＋"_NEW"}として、
	 * アップロードのオリジナルと変更後のファイル名を取得することが出来ます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 *
	 * @param   filename ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "upload" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "names"		,names		)
				.println( "ENCODE"		,ENCODE		)
				.println( "fileURL" 	,fileURL	)
				.println( "filename"    ,filename	)
				.println( "maxPostSize" ,maxPostSize)
				.println( "tableId" 	,tableId	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
