/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;

import java.util.Map;
import java.util.WeakHashMap;
import java.util.Collections ;

/**
 * systemId に対応したカラムデータを作成するデータロードクラスです。
 *
 * カラムデータは、項目(CLM)に対して、各種カラム情報を持っています。
 * エンジン内部で使用している DBColumn オブジェクトは、RENDERER や EDITOR など
 * 実際にはオブジェクトで管理していますが、この ColumnData では、それらのキーとなる
 * 文字列を持っています。実際に DBColumn オブジェクトの構築時に、各属性オブジェクトを
 * 生成（または、キャッシュから取り出し）ます。
 *
 * カラムデータを作成する場合は、同一カラムで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つコードを使用します。
 * 作成区分(KBSAKU)='0' のデータは、マスタリソースとして、エンジンとともに
 * 配布されるリソースになります。
 *
 * カラムデータには、３つのレベルのオブジェクト作成方法が適用されます。
 * エンジン内部のカラムリソースファイル(org.opengion.hayabusa.common.data.ColumnResource)は、
 * 初期作成されるカラムリソースです。エンジンの更新に対応して、このリソースも同時に
 * 更新されます。このカラムは、最も優先順位の低いリソースで、同一キー情報で他の形式の
 * カラムがあれば、そちらが使用されます。
 *
 * 読込フラグ(FGLOAD)='1'のカラムリソースは、すべて初期起動時に一括読み込みされます。
 * 読込フラグが、'1' 以外のデータは、初期起動時には、メモリにキャッシュされず
 * 実際に使用されるまで、オブジェクトが作成されません。
 * これは、使用されるかどうか判らないカラムデータを、予め作成しないことで、メモリの
 * 節約を図っています。
 *
 * SYSTEM_ID='**' は、共通リソースです。
 * これは、システム間で共通に使用されるリソース情報を登録しておきます。
 *
 * @og.rev 4.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class ColumnDataLoader {
	// リソースの接続先を、取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// ＤＢリソースの初期一括読込のクエリー
 //private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
 //									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
 //									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
 //									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
 //									+ " and FGJ='1' and FGLOAD = '1'"
 //									+ " order by SYSTEM_ID,CLM,KBSAKU" ;

	// 4.3.5.7 (2009/03/22) FGLOADの影響で個別システムのリソースが読まれない問題の対応
	private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES,"
									+ "FGLOAD"
									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
									+ " and FGJ='1'"
									+ " order by SYSTEM_ID,CLM,KBSAKU" ;

	// ＤＢリソースの個別読込時のクエリー
	private static final String QUERY2 = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
									+ " and CLM=? and FGJ='1'"
									+ " order by SYSTEM_ID,KBSAKU" ;

	private final Map<String,ColumnData> pool = Collections.synchronizedMap( new WeakHashMap<String,ColumnData>() );	// キャッシュ用プール
	private final String  SYSTEM_ID ;		// システムID

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 *  SystemId 毎に ファクトリオブジェクトを作成します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param systemId システムID
	 * @param initLoad リソースデータの先読み可否(true:先読みする)
	 */
	ColumnDataLoader( final String systemId,final boolean initLoad ) {
		SYSTEM_ID = systemId;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "ColumnDataLoader",null,null );
		}
		else {
			appInfo = null;
		}

		// ApplicationInfo の設定が終わってから実行します。
		if( initLoad ) { loadDBResource(); }
	}

	/**
	 * ＤＢリソースより カラムデータを取得、設定します。
	 * 同一キー(CLM)に対して、複数の作成区分(KBSAKU)を持つデータが
	 * 検索される場合は、作成区分(KBSAKU)の大きな値が使用されます。
	 * つまり、より、ローカライズなキーほど、作成区分(KBSAKU)に大きな値を
	 * 使用するようにします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.3.5.7 (2009/03/22) FGLOADの影響でシステム個別リソースが読まれない問題対応
	 *
	 */
	private void loadDBResource() {
		String[] args = new String[] { SYSTEM_ID };

		String[][] vals = DBUtil.dbExecute( QUERY,args,appInfo,DBID );

		int len = vals.length;
		for( int i=0; i<len; i++ ) {
			if( "1".equals( vals[i][ColumnData.FG_LOAD] ) ) { // 4.3.5.7 (2009/03/22)
				pool.put( vals[i][0],new ColumnData( vals[i] ) );
			}
			// より上の作成区分で、FGLOAD='0'(個別読込)が来た場合は、下位のFGLOAD='1'(一括読込)を破棄
			else if( pool.get( vals[i][0]) != null ){
				pool.remove( vals[i][0] );
			}
		}

		System.out.println( "  ColumnDataLoader [" + len + "] loaded" );
	}

	/**
	 * ColumnData オブジェクトを取得します。
	 * 作成したColumnDataオブジェクトは，内部にプールしておき，同じリソース要求が
	 * あったときは，プールの ColumnDataを返します。
	 * 読込フラグ(FGLOAD)が '1' のデータは、起動時に先読みします。
	 * それ以外のデータは、ここでキー要求が発生した時点で読み込みます。
	 * 読込フラグ(FGLOAD) のマーカー設定モード(useAutoSetting)を使用する(true)場合は、
	 * 追加読み込み（先読みされていないカラム）に対して、読込フラグ(FGLOAD)を '2' に
	 * 自動設定します。（次回起動時の、初期読み込みは行いません。）
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param   key         カラムのキー
	 * @return  ColumnData  カラムオブジェクト
	 */
	public ColumnData getColumnData( final String key ) {
		ColumnData column = pool.get( key ) ;
		if( column == null ) {
			String[] args = new String[] { SYSTEM_ID,key };
			String[][] vals = DBUtil.dbExecute( QUERY2,args,appInfo,DBID );		// 個別検索

			if( vals.length > 0 ) {
				column = new ColumnData( vals[vals.length-1] );		// 最後の検索結果が有効
				pool.put( key,column );
			}
		}

		return column ;
	}

	/**
	 * ColumnData オブジェクトのキャッシュを個別にクリアします。
	 * リソースデータの更新など、一部分の更新時に、すべてのキャッシュを
	 * 破棄するのではなく、指定の分のみ破棄できる機能です。
	 *
	 * @param   key         カラムのキー
	 */
	public void clear( final String key ) {
		pool.remove( key );
	}

	/**
	 * ColumnData オブジェクトのキャッシュをクリアします。
	 *
	 */
	public void clear() {
		pool.clear();
	}
}
