/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * Excelでの文字変換関数 =T("値") という文字列で書き出すクラスです。
 *
 * DefaultTableWriter を継承して，データの出力部のみオーバーライドして，
 * 文字列カラム（クラス名VARCHAR2のカラム）に、=T("値") という文字列で出力する
 * ファイルの出力機能を実現しています。
 *
 * @og.rev 3.1.4.1 (2003/04/21) 新規作成
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_T extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/** 文字列タイプ名	{@value} */
	public static final String CLASS_VARCHAR2	= "VARCHAR2";

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 行番号情報を、出力する(true)/しない(false)を指定
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows = table.getRowCount();
		String SEPARATOR = getSeparator();
		boolean useNumber = isUseNumber();
		boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int row=0; row<numberOfRows; row++ ) {
			if( useNumber ) {
				writer.print( String.valueOf( row+1 ) );
				writer.print( SEPARATOR );
			}
			for( int i=0; i<numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue(row,clm);
//				if( "NVAR".equals( dbColumn[clm].getDbType()) ) {
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					val = StringUtil.spanCut( dbColumn[clm].getRendererValue( val ) );
				}

				if( i != 0 ) { writer.print( SEPARATOR ); }
				// 開始日などの 00000000 を文字列タイプで渡す
				if( CLASS_VARCHAR2.indexOf( dbColumn[clm].getClassName() ) < 0 ) {
					writer.print( val );
				}
				else {
					writer.print( excelT( val ) );
				}
			}
			writer.println();
		}
	}

	/**
	 * EXCELの文字列変換関数である T を適用します。
	 * 値を、 =T("値") という文字列に変換します。
	 * 値にダブルクオートが存在する場合は、その直前にもう一つダブルクオートを
	 * 入れて、エスケープします。
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) JDK1.3 対応に修正。
	 *
	 * @param	 data 元のString文字列
	 * @return	 T関数を適用した文字列
	 */
	private String excelT( final String data ) {
		if( data == null ) { return "" ; }

		StringBuilder strBuf = new StringBuilder( data );
		int len = data.lastIndexOf( '\"' );
		while( len >= 0 ) {
			strBuf.insert( len,"\"" ) ;
			len = data.lastIndexOf( "\"",len-1 );
		}

		strBuf.insert( 0,"=T(\"" ).append( "\")" );

		return strBuf.toString();
	}
}
