/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.awt.color.ColorSpace;
import java.awt.color.ICC_ColorSpace;
import java.awt.color.ICC_Profile;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.awt.image.ColorConvertOp;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import javax.media.jai.JAI;

import javax.imageio.ImageIO;

import com.sun.media.jai.codec.FileSeekableStream;

/**
 * ImageResizer は、画像ファイルのリサイズを行うためのクラスです。
 * ここでの使い方は、初期化時に、オリジナルの画像ファイルを指定し、
 * 変換時に各縮小方法に対応したメソッドを呼び出し、画像を変換します。
 * 変換方法としては、以下の3つがあります。
 * ①最大サイズ(px)指定による変換
 *   縦横の最大サイズ(px)を指定し、変換を行います。
 *   横長の画像については、変換後の横幅=最大サイズとなり、縦幅については、横幅の
 *   縮小率に従って決定されます。
 *   逆に縦長の画像については、変換後の縦幅=最大サイズとなり、横幅については、縦幅の
 *   縮小率に従って決定されます。
 * ②縦横サイズ(px)指定による変換
 *   縦横の変換後のサイズ(px)を個別に指定し、変換を行います。
 * ③縮小率指定による変換
 *   "1"を元サイズとする縮小率を指定し、変換を行います。
 *   縮小率は、縦横で同じ縮小率が適用されます。
 * 入力フォーマットとしてはJPEG/PNG/GIFに、出力フォーマットとしてはJPEG/PNGに対応しています。
 * 出力フォーマットについては、出力ファイル名の拡張子より自動的に決定されますが、一般的には
 * サイズが小さくなるjpegファイルを推奨します。
 * 入出力フォーマットについて、対応していないフォーマットが指定された場合は例外が発生します。
 * また、縦横の出力サイズが入力サイズの縦横よりも両方大きい場合、変換は行われず、入力ファイルが
 * そのままコピーされて出力されます。(拡大変換は行われません)
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ImageResizer {
	private static final long serialVersionUID = 5390 ;	// 5.3.9.0 (2011/09/01)
	private static final String ICC_PROFILE = "org/opengion/fukurou/util/ISOcoated_v2_eci.icc"; // 5.4.3.5 (2012/01/17)

	private final File inFile;
	private final BufferedImage inputImage; // 入力画像オブジェクト

	private final int inSizeX;				// 入力画像の横サイズ
	private final int inSizeY;				// 入力画像の縦サイズ

	/**
	 * 入力ファイル名を指定し、画像縮小オブジェクトを初期化します。
	 * 
	 * @og.rev 5.4.3.5 (2012/01/17)  CMYK対応
	 * @og.rev 5.4.3.7 (2012/01/20)  FAIでのファイル取得方法変更
	 * @og.rev 5.4.3.8 (2012/01/24)  エラーメッセージ追加
	 * @param in 入力ファイル名
	 */
	public ImageResizer( final String in ) {
		String inSuffix = getSuffix( in );
		BufferedImage bi;
		if( "|jpeg|jpg|png|gif|".indexOf( inSuffix ) < 0 ) {
			String errMsg = "入力ファイルは(JPEG|PNG|GIF)のいずれかの形式のみ指定可能です。" + "File=[" + in + "]";
			throw new RuntimeException( errMsg );
		}

		inFile = new File( in );
		try {
			// inputImage = ImageIO.read( inFile );
			bi = ImageIO.read( inFile );
		}
		catch (javax.imageio.IIOException ex){ // 5.4.3.5 (2012/01/17) 決めうち
			FileSeekableStream fsstream = null;
			try{
				// 5.4.3.7 (2012/01/20) ファイルの開放がGC依存なので、streamで取得するように変更
				// bi = cmykToSRGB(JAI.create("FileLoad",inFile.toString()).getAsBufferedImage(null,null));
				fsstream = new FileSeekableStream(inFile.getAbsolutePath());
				bi = cmykToSRGB(JAI.create("stream",fsstream).getAsBufferedImage(null,null));

			}
			catch( IOException ioe ){
				String errMsg = "イメージファイルの読込(JAI)に失敗しました。" + "File=[" + in + "]";
				throw new RuntimeException( errMsg,ioe );
			}
			catch( Exception oe ){ // 5.4.3.8 (2012/01/23) その他エラーの場合追加
				String errMsg = "イメージファイルの読込(JAI)に失敗しました。ファイルが壊れている可能性があります。" + "File=[" + in + "]";
				throw new RuntimeException( errMsg,oe );
			}
			finally{
				Closer.ioClose(fsstream);
			}
			
		}
		catch( IOException ex ) {
			String errMsg = "イメージファイルの読込に失敗しました。" + "File=[" + in + "]";
			throw new RuntimeException( errMsg,ex );
		}
		
		inputImage = bi;
		inSizeX = inputImage.getWidth();
		inSizeY = inputImage.getHeight();
	}

	/**
	 * 縦横の最大サイズ(px)を指定し、変換を行います。
	 * 横長の画像については、変換後の横幅=最大サイズとなり、縦幅については、横幅の
	 * 縮小率に従って決定されます。
	 * 逆に縦長の画像については、変換後の縦幅=最大サイズとなり、横幅については、縦幅の
	 * 縮小率に従って決定されます。
	 * 
	 * @param out 出力ファイル名
	 * @param maxSize 変換後の縦横の最大サイズ
	 */
	public void resizeByPixel( final String out, final int maxSize ) {
		int sizeX = 0;
		int sizeY = 0;
		if( inSizeX > inSizeY ) {
			sizeX = maxSize;
			sizeY = inSizeY * maxSize / inSizeX;
		}
		else {
			sizeX = inSizeX * maxSize / inSizeY;
			sizeY = maxSize;
		}
		convert( inputImage, out, sizeX, sizeY );
	}

	/**
	 * 縦横の変換後のサイズ(px)を個別に指定し、変換を行います。
	 * 
	 * @param out 出力ファイル名
	 * @param sizeX 変換後の横サイズ(px)
	 * @param sizeY 変換後の縦サイズ(px)
	 */
	public void resizeByPixel( final String out, final int sizeX, final int sizeY ) {
		convert( inputImage, out, sizeX, sizeY );
	}

	/**
	 * "1"を元サイズとする縮小率を指定し、変換を行います。
	 *  縮小率は、縦横で同じ縮小率が適用されます。
	 * 
	 * @param out 出力ファイル名
	 * @param ratio 縮小率
	 */
	public void resizeByRatio( final String out, final double ratio ) {
		int sizeX = (int)( inSizeX * ratio );
		int sizeY = (int)( inSizeY * ratio );
		convert( inputImage, out, sizeX, sizeY );
	}

	/**
	 * 画像の変換を行うための内部共通メソッドです。
	 * 
	 * @og.rev 5.4.1.0 (2011/11/01) 画像によってgetTypeが0を返し、エラーになる不具合を修正
	 * 
	 * @param inputImage 入力画像オブジェクト
	 * @param out 出力ファイル名
	 * @param sizeX 横サイズ(px)
	 * @param sizeY 縦サイズ(px)
	 */
	private void convert( final BufferedImage inputImage, final String out, final int sizeX, final int sizeY ) {
		String outSuffix = getSuffix( out );
		if( "|jpeg|jpg|png|".indexOf( outSuffix ) < 0 ) {
			String errMsg = "出力ファイルは(JPEG|PNG)のいずれかの形式のみ指定可能です。" + "File=[" + out + "]";
			throw new RuntimeException( errMsg );
		}

		File outFile = new File( out );
		// 変換後の縦横サイズが大きい場合はコピーして終わり(変換しない)
		if( sizeX > inSizeX && sizeY > inSizeY ) {
			FileUtil.copy( inFile, outFile );
			return;
		}

		// 5.4.1.0 (2011/11/01) 画像によってgetTypeが0を返し、エラーになる不具合を修正
		int type = inputImage.getType();
		BufferedImage resizeImage = null;
		if( type == 0 ) {
			resizeImage = new BufferedImage( sizeX, sizeY, BufferedImage.TYPE_4BYTE_ABGR_PRE );
		}
		else {
			resizeImage = new BufferedImage( sizeX, sizeY, inputImage.getType() );
		}
		AffineTransformOp ato = null;
		ato = new AffineTransformOp(
				AffineTransform.getScaleInstance(
						(double)sizeX/inSizeX, (double)sizeY/inSizeY ), null );
		ato.filter( inputImage, resizeImage );

		try {
			ImageIO.write( resizeImage, outSuffix, outFile );
		}
		catch( IOException ex ) {
			String errMsg = "イメージファイルの作成に失敗しました。" + "File=[" + out + "]";
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * ファイル名から拡張子(小文字)を求めます。
	 * 
	 * @param fileName
	 * @return 拡張子(小文字)
	 */
	private static String getSuffix( final String fileName ) {
		String suffix = null;
		if( fileName != null ) {
			int sufIdx = fileName.lastIndexOf( '.' );
			if( sufIdx >= 0 ) {
				suffix = fileName.substring( sufIdx + 1 ).toLowerCase( Locale.JAPAN );
				if( suffix.length() > 5 ) {
					suffix = null;
				}
			}
		}
		return suffix;
	}
	
	
	/**
	 * BufferedImageをISOCoatedのICCプロファイルで読み込み、RGBにした結果を返します。
	 * (CMYKからRBGへの変換、ビット反転)
	 * 
	 * @og.rev 5.4.3.5 (2012/01/17)
	 * @param readImage
	 * @return 変換後のBufferedImage
	 * @throws IOException 
	 */
	 public BufferedImage cmykToSRGB(BufferedImage readImage) throws IOException {
		 ClassLoader loader = Thread.currentThread().getContextClassLoader();
		 InputStream icc_stream = loader.getResourceAsStream( ICC_PROFILE );
		 ICC_Profile p =	ICC_Profile.getInstance(icc_stream);//変換プロファイル
		 ColorSpace cmykCS = new ICC_ColorSpace(p);
		 // 遅いので標準のスペースは使えない
		 //ColorSpace cmykCS = SimpleCMYKColorSpace.getInstance();
		 BufferedImage rgbImage = new BufferedImage(readImage.getWidth(),
				 readImage.getHeight(), BufferedImage.TYPE_INT_RGB);
		 ColorSpace rgbCS = rgbImage.getColorModel().getColorSpace();
		 ColorConvertOp cmykToRgb = new ColorConvertOp(cmykCS, rgbCS, null);
		 cmykToRgb.filter(readImage, rgbImage);
		 
		int w = rgbImage.getWidth();
		int h = rgbImage.getHeight();
		// 反転が必要
		for (int i=0;i<w;i++) {
			for (int j=0;j<h;j++) {
				int rgb = rgbImage.getRGB(i, j);
				int r = (rgb & 0xff0000) >> 16;
				int g = (rgb & 0x00ff00) >> 8;
				int b = (rgb & 0x0000ff);
				rgb = ((Math.abs(r - 255) << 16) + (Math.abs(g - 255) << 8) + (Math.abs(b - 255)));
				rgbImage.setRGB(i, j, rgb);
			}
		}

		 return rgbImage;
	 }

	/**
	 * メイン処理です。
	 * java org.opengion.fukurou.util.ImageResizer [Input Filename] [OutputFilename] [ResizeScale]
	 *
	 * @param  args  引数文字列配列 入力ファイル、出力ファイル、縦横最大サイズ
	 */
	public static void main( final String[] args ) {
		if( args.length < 3 ) {
			LogWriter.log( "Usage: java org.opengion.fukurou.util.ImageResizer [Input Filename] [OutputFilename] [ResizeScale]" );
			return ;
		}

		ImageResizer ir = new ImageResizer( args[0] );
		ir.resizeByPixel( args[1], Integer.parseInt( args[2] ),Integer.parseInt( args[2] ) );
//		ir.resizeByRatio( args[1], Double.parseDouble( args[2] ) );
	}
}
