/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer;

import java.io.PrintWriter;
import java.io.IOException;
import java.io.File;
import java.io.ByteArrayOutputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletResponse;

/**
 * FileFilter で使用する、File圧縮するServletOutputStreamクラスです。
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileResponseStream extends ServletOutputStream {
	/** 内部出力ストリーム */
	protected ByteArrayOutputStream filestream = null;
	/** クローズ判定 */
	protected boolean isClosed = false;
	/** レスポンスオブジェクト */
//	protected ServletResponse response = null;		// 5.5.2.6 (2012/05/25)
	/** サーブレット出力ストリーム */
	protected ServletOutputStream output = null;

	private final String filename ;

	/**
	 * コンストラクター
	 *
	 * @param response ServletResponse レスポンス
	 * @param filename String ファイル名
	 * @throws IOException
	 */
	public FileResponseStream( final ServletResponse response,final String filename ) throws IOException {
		// 4.3.4.4 (2009/01/01)
//		super();
		this.filename = filename;
		isClosed = false;
//		this.response = response;		// 5.5.2.6 (2012/05/25)
		this.output = response.getOutputStream();
		filestream = new ByteArrayOutputStream();
	}

	/**
	 * このストリームを閉じ、このストリームに関連するすべてのシステムリソースを解放します。
	 *
	 * close の汎用規約では、close は出力ストリームを閉じます。閉じられたストリームは
	 * 出力処理を実行できません。また、それを開き直すことはできません。
	 *
	 * @throws IOException
	 */
	public void close() throws IOException {
		if(isClosed) {
//			System.out.println( "already been closed !" );
//			throw new IOException("This output stream has already been closed");
			return ;
		}
		try {
			filestream.flush();

			String msg = filestream.toString( "UTF-8" );
			FileResponseTransform trans = new FileResponseTransform();
			msg = trans.replace( filename,msg );

			PrintWriter writer = FileUtil.getPrintWriter( new File( filename ), "UTF-8" );
			writer.print( msg );
			Closer.ioClose( writer );
			System.out.println( filename + " Saved" );

			output.flush();
		}
		finally {
			isClosed = true;
			Closer.ioClose( filestream );
			Closer.ioClose( output );
		}
	}

	/**
	 * この出力ストリームをフラッシュし、バッファに入っている出力バイトをすべて強制的書き込みますに。
	 *
	 * flush の汎用規約では、それまでに書き込まれたバイトが出力ストリームの
	 * 実装によってバッファに入れられている場合に flush を呼び出すと、それらのバイトは
	 * ただちにその目的の転送先に書き込まれます。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) isClosed == true の場合に Exception でなく、return にする。
	 * @throws IOException
	 */
	public void flush() throws IOException {
		if(isClosed) {
//			throw new IOException("Cannot flush a closed output stream");
			return ;
		}
		filestream.flush();
		output.flush();
	}

	/**
	 * この出力ストリームに指定されたバイトを書き込みます。
	 *
	 * write の汎用規約では、1 バイトが
	 * 出力ストリームに書き込まれます。書き込まれるバイトは、引数 b の下位 8 ビットです。
	 * b の上位 24 ビットは無視されます。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) isClosed == true の場合に Exception でなく、return にする。
	 *
	 * @param bt int byteデータ
	 * @throws IOException
	 */
	public void write(final int bt) throws IOException {
		if(isClosed) {
//			throw new IOException("Cannot write to a closed output stream");
			return ;
		}
		filestream.write((byte)bt);
		output.write((byte)bt);
	}

	/**
	 * 指定されたバイト配列からこの出力ストリームに b.length バイトを書き込みます。
	 *
	 * write(b) の汎用規約では、write(b) の効果は write(b, 0, b.length) を呼び出す
	 * 場合とまったく同じです。
	 *
	 * @param bt byte[] データ
	 * @throws IOException
	 */
	public void write(final byte[] bt) throws IOException {
		write(bt, 0, bt.length);
	}

	/**
	 * オフセット off から始まる指定のバイト配列からこの出力ストリームに len バイトを書き込みます。
	 *
	 * write(b, off, len) の汎用規約では、配列 b 内の一定のバイトが出力ストリームに順番に
	 * 書き込まれます。この処理で最初に書き込まれるバイトは要素 b[off]、最後に書き込まれる
	 * バイトは要素 b[off+len-1] です。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) isClosed == true の場合に Exception でなく、return にする。
	 *
	 * @param bt byte[]
	 * @param off int
	 * @param len int
	 * @throws IOException
	 */
	public void write(final byte bt[], final int off, final int len) throws IOException {
		if(isClosed) {
//			throw new IOException("Cannot write to a closed output stream");
			return ;
		}
		output.write(bt, off, len);
		filestream.write(bt, off, len);
	}

	/**
	 * すでにストリームが閉じられているかどうかを返します。
	 *
	 * @return boolean すでにストリームが閉じられているかどうか
	 */
	public boolean closed() {
		return isClosed;
	}
}
