/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.FileUtil;
// import org.opengion.fukurou.util.CSVTokenizer ;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer ;
// import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.model.ArrayDataModel;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * 指定のファイルを直接データベースに登録するデータ入力タグです。
 *
 * 通常の readTable などは、DBTableModel オブジェクトを介して全件メモリに
 * ロードしてから表示させる為、大量データ処理ができません。
 * このタグでは、直接ファイルを読み取りながらデータベース登録するので
 * 大量データをバッチ的に登録する場合に使用します。
 *
 * 読み取るファイルは、先頭(または実データが現れるまでに) #NAME 行が必要です。
 * これは、ファイルデータのカラム名を指定しています。また、columns 属性を使用すれば、
 * ファイルの#NAME 行より優先して(つまり存在していなくても良い)データのカラム名を
 * 指定することが出来ます。
 * この#NAME 行は、ファイルのセパレータと無関係に必ずタブ区切りで用意されています。
 * タグのBODY部に、実行するSQL文を記述します。
 * このSQL文は、
 * INSERT INTO GE41 (CLM,NAME_JA,SYSTEM_ID,FGJ,DYSET)
 * VALUES ([CLM],[NAME_JA],[SYSTEM_ID],'1','{&#064;USER.YMDH}')
 * と、いう感じで、ファイルから読み込んだ値は、[カラム名]に割り当てられます。
 * もちろん、通常の固定値(FGJに'1'をセット)や、リクエスト変数(DYSETの{&#064;USER.YMDH})
 * なども使用できます。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:directTableInsert filename="[･･･]" ･･･ &gt;INSERT INTO ･･･ &lt;/og:directTableInsert &gt;
 * ●body：あり
 *
 * ●使用例
 *     &lt;og:directTableInsert
 *         dbid         = "ORCL"                接続データベースID(初期値：DEFAULT)
 *         separator    = ","                   ファイルの区切り文字（初期値：タブ）
 *         fileURL      = "{&#064;USER.ID}"     読み取り元ディレクトリ名
 *         filename     = "{&#064;filename}"    読み取り元ファイル名
 *         encode       = "Shift_JIS"           読み取り元ファイルエンコード名
 *         displayMsg   = "MSG0040"             登録完了後のメッセージ
 *         columns      = "CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG"
 *                                              #NAME の代わりに使用するカラム列名
 *         commitBatch  = "100"                 この件数ずつコミットを発行(初期値：無制限)
 *         useColumnCheck  = "true"             カラムチェックを行うかどうか(初期値:false)
 *         useColumnAdjust = "true"             カラム変換を行うかどうか(初期値:false)
 *         nullCheck       = "CLM,SYSTEM_ID"    NULLチェックを実行します。
 *     &gt;
 *        &lt;jsp:text&gt;
 *          INSERT INTO GE41
 *              (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *                 FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *          VALUES
 *              ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *                '1','{&#064;USER.YMDH}','{&#064;USER.YMDH}','{&#064;USER.ID}','{&#064;USER.ID}','{&#064;GUI.KEY}')
 *        &lt;/jsp:text&gt;
 *     &lt;/og:directTableInsert &gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectTableInsertTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String TAB_SEPARATOR = "\t" ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	  dbid			= "DEFAULT";
	private String	  dbid			= null;
	private String    separator		= TAB_SEPARATOR;   // 項目区切り文字
	private String    fileURL		= HybsSystem.sys( "FILE_URL" );		// 4.0.0 (2005/01/31)
	private String    filename		= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
	private String    encode		= HybsSystem.sys( "FILE_ENCODE"   );   // ファイルエンコーディング  "JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private String    displayMsg	= "MSG0040";	// 　件登録しました。
	private String[]  columns		= null;
	private String[]  clmKeys		= null;		// SQL文の[カラム名]配列
	private String	  sql			= null;
	private int		  commitBatch	= 0;		// コミットするまとめ件数
	private boolean   useColumnCheck  = false;	// 3.6.0.2 (2004/10/04)
	private boolean   useColumnAdjust = false;	// 3.6.0.2 (2004/10/04)
	private String[]  nullCheck		= null;		// 3.8.0.2 (2005/06/30) nullチェック確認
	private long	  dyStart		= 0;	// 実行時間測定用のDIV要素を出力します。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();		// 時間測定用
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) SQL文の [カラム] 対応とパーサー機能追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 *
	 * @return  int 後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString();
		if( sql == null || sql.length() == 0 ) {
			String errMsg = "BODY 部の登録用 Insert/Update文は、必須です。";
			throw new HybsSystemException( errMsg );
		}
		// 3.6.0.2 (2004/10/04) SQL文の [カラム] 対応とパーサー機能追加
//		sql = makeFormat( sql.trim() );		// ここで、clmLeys 配列もセットされる。

		return(SKIP_BODY);				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		BufferedReader pw = getBufferedReader();
		int executeCount = create( pw );
		Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視

		// 実行件数の表示
		// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
		if( displayMsg != null && displayMsg.length() > 0 ) {
//			String status = executeCount + getResource().getMessage( displayMsg ) ;
			String status = executeCount + getResource().getLabel( displayMsg ) ;
			jspPrint( status + HybsSystem.BR );
		}

		// 時間測定用の DIV 要素を出力
		long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( executeCount,dyTime,sql ); }

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) useColumnCheck,useColumnAdjust 属性追加
	 * @og.rev 3.8.0.2 (2005/06/30) nullCheck 属性追加
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 */
	@Override
	protected void release2() {
		super.release2();
		dbid			= null;
		separator		= TAB_SEPARATOR;   // 項目区切り文字
		fileURL			= HybsSystem.sys( "FILE_URL" );		// 4.0.0 (2005/01/31)
		filename		= HybsSystem.sys( "FILE_FILENAME"        );   // ファイル名
		encode			= HybsSystem.sys( "FILE_ENCODE"          );   // ファイルエンコーディング  "JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		displayMsg		= "MSG0040";	// 　件登録しました。
		columns			= null;		// 3.5.4.5 (2004/01/23)
		useColumnCheck	= false;	// 3.6.0.2 (2004/10/04)
		useColumnAdjust	= false;	// 3.6.0.2 (2004/10/04)
		nullCheck		= null;		// 3.8.0.2 (2005/06/30)
	}

	/**
	 * BufferedReader より読み込み、データベースに書き込みます。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) カラムオブジェクトのDBType属性の整合性チェック
	 * @og.rev 3.8.0.2 (2005/06/30) nullチェック確認
	 * @og.rev 3.8.5.1 (2006/05/08) 取込データが name 列より少ない場合の対応を追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) CheckColumnDataクラス static 化、引数にResourceManager追加
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.2.0 (2010/11/01)) ""で囲われているデータに改行が入っていた場合の対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) pstmt.setObject で、useParamMetaData の判定を避けるため、pstmt.setString で代用(PostgreSQL対応)
	 *
	 * @param   reader BufferedReader
	 * @return  取り込み件数
	 */
	private int create( final BufferedReader reader )  {

		String[] names = readName( reader );	// ファイルのカラム名
		int   nameLen = names.length ;			// 3.8.5.1 (2006/05/08) 追加
//		int[] clmNos = getColumnNos( names, clmKeys );	// バインド変数のアドレス求め
//		int   clmNosLen = clmNos.length ;

		ArrayDataModel nmdata = new ArrayDataModel( names );
		Formatter format = new Formatter( nmdata );
		format.setFormat( sql.trim() );
		sql = format.getQueryFormatString();
		int[] clmNos = format.getClmNos();
		int   clmNosLen = clmNos.length ;
		clmKeys = format.getClmKeys();

		CheckColumnData checkClass = new CheckColumnData( clmNos,clmKeys,getResource() );

		ArrayDataModel nullData = new ArrayDataModel( names );
		int[] nullClmNos = nullData.getColumnNos( nullCheck );	// バインド変数のアドレス求め

		// 3.8.0.2 (2005/06/30) nullチェック確認
//		int[] nullClmNos = getColumnNos( names, nullCheck );	// バインド変数のアドレス求め
		int   nullClmNosLen = nullClmNos.length ;

		int    executeCount = 0;
		int    commitCount  = 0;
		char   sep  = separator.charAt(0);
		boolean errFlag  = true;
//		Connection  conn = null;
		Transaction tran = null;	// 5.1.9.0 (2010/08/01) Transaction 対応
		PreparedStatement pstmt	= null ;
		String[] data	= null ;
		try {
			// 5.1.9.0 (2010/08/01) Transaction 対応
			TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
//				tran = new TransactionReal( dbid,getApplicationInfo() );
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}
//			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			
//			try {
				Connection conn = tran.getConnection( dbid );		// 5.1.9.0 (2010/08/01) Transaction 対応
				pstmt = conn.prepareStatement( sql );
				String line ;
				while((line = reader.readLine()) != null) {
					if( line.length() == 0 || line.charAt( 0 ) == '#' ) { continue; }
					else {
						// 5.2.2.0 (2010/11/01) ""で囲われているデータに改行が入っていた場合の対応
						int quotCount = StringUtil.countChar( line, '"' );
						if( quotCount % 2 != 0 ) {
							String addLine = null;
							StringBuilder buf = new StringBuilder( line );
							while(quotCount % 2 != 0 && (addLine = reader.readLine()) != null) {
								if( line.length() == 0 || line.charAt( 0 ) == '#' ) { continue; }
								buf.append( HybsSystem.CR ).append( addLine );
								quotCount += StringUtil.countChar( addLine, '"' );
							}
							line = buf.toString();
						}

						// 3.8.5.1 (2006/05/08) 取込データが name 列より少ない場合の対応を追加
						data = StringUtil.csv2Array( line , sep , nameLen );

						// 3.6.0.2 (2004/10/04) カラム変換
						if( useColumnAdjust ) {
							data = checkClass.adjustData( data );
						}

						// 3.6.0.2 (2004/10/04) カラムチェック
						if( useColumnCheck ) {
							ErrorMessage errMsg = checkClass.checkData( executeCount, data );
							if( !errMsg.isOK() ) {
//								conn.rollback();
								tran.rollback();			// 5.1.9.0 (2010/08/01) Transaction 対応
								jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg,getResource() ) );
								return commitCount;
							}
						}

						// 3.8.0.2 (2005/06/30) nullチェック確認
						if( nullClmNosLen > 0 ) {
							ErrorMessage errMsg = new ErrorMessage( "Null Check Columns Error!" );

							for( int i=0; i<nullClmNosLen; i++ ) {
								int clm = nullClmNos[i];
								if( data[clm] == null || data[clm].length() == 0 ) {
									String label = getResource().getLabel( nullCheck[i] );
									// ERR0012 : 指定のデータがセットされていません。（NULLエラー）。key={0}
									errMsg.addMessage( executeCount+1,ErrorMessage.NG,"ERR0012",label );
								}
							}
							if( !errMsg.isOK() ) {
//								conn.rollback();
								tran.rollback();			// 5.1.9.0 (2010/08/01) Transaction 対応
								jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg,getResource() ) );
								return commitCount;
							}
						}

						for( int i=0; i<clmNosLen; i++ ) {
							String val = data[clmNos[i]];
							if( val != null && val.startsWith( "'0" ) ) {
								val = val.substring( 1 );
							}
//							pstmt.setObject( i+1,val );		// 5.3.8.0 (2011/08/01) 処理の簡素化
							pstmt.setString( i+1,val );
						}

						pstmt.execute();
						if( commitBatch > 0 && ( executeCount%commitBatch == 0 ) ) {
//							conn.commit();
							tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
							commitCount = executeCount;
						}
						executeCount++ ;
					}
				}
//				conn.commit();
				tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
				commitCount = executeCount ;
				errFlag = false;		// エラーではない
//			}
//			finally {
//				Closer.stmtClose( pstmt );
//			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + reader.toString() + "]"
						+ " 行番号=[" + executeCount +"]"
						+ " 登録件数=[" + commitCount + "]"  ;
			throw new HybsSystemException( errMsg,ex );
		}
		catch (SQLException ex) {
			String errMsg = "sql=[" + sql + "]" + HybsSystem.CR
						+	"names=[" + StringUtil.array2csv( names ) + "]" + HybsSystem.CR
						+	"vals =[" + StringUtil.array2csv( data ) + "]" + HybsSystem.CR
						+ " 行番号=[" + executeCount +"]"
						+ " 登録件数=[" + commitCount + "]"  + HybsSystem.CR
						+ " errorCode=[" + ex.getErrorCode() + "] State=[" +
						ex.getSQLState() + "]" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			Closer.stmtClose( pstmt );
//			tran.close( errFlag );			// 5.1.9.0 (2010/08/01) Transaction 対応
			if( tran != null ) {				// 5.5.2.6 (2012/05/25) findbugs対応
				tran.close( errFlag );			// 5.1.9.0 (2010/08/01) Transaction 対応
			}
//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
//			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//			conn = null;
		}

		return commitCount;
	}

	/**
	 * BufferedReader より、#NAME 行の項目名情報を読み取ります。
	 * データカラムより前に、項目名情報を示す "#Name" が存在する仮定で取り込みます。
	 * この行は、ファイルの形式に無関係に、TAB で区切られています。
	 * #NAME 配列の先頭(行番号にあたる個所)は、ROW_NO というキーを割り当てます。
	 * columns が設定されている場合は、#NAME 行ではなく、columns を優先します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) #NAME 行が見つからない場合は、エラーとします。
	 * @og.rev 3.6.0.2 (2004/10/04) columns が設定されている場合は、それを返します。
	 *
	 * @param 	reader PrintWriter
	 * @return	String[] カラム名配列
	 */
	private String[] readName( final BufferedReader reader ) {
		if( columns != null && columns.length > 0 ) {
			return columns ;
		}

		try {
			String line;
			while((line = reader.readLine()) != null) {
				if( line.length() == 0 ) { continue; }
				if( line.charAt(0) == '#' ) {
					if( line.length() >= 5 &&
						"#NAME".equalsIgnoreCase( line.substring( 0,5 ) ) ) {
						String[] rtn = StringUtil.csv2Array( line ,TAB_SEPARATOR.charAt(0) );
						rtn[0] = "ROW_NO";	// 先頭カラムにカラム名を与える。
						return rtn ;
					}
					else { continue; }
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。" + HybsSystem.CR 
							+ " LINE=" + line;			// 5.1.8.0 (2010/07/01) errMsg 修正
					throw new HybsSystemException( errMsg );
				}
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + reader.toString() + "]"  ;
			throw new HybsSystemException( errMsg,ex );
		}

		String errMsg = "#NAME が見つかりませんでした。";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * SQLにバインドされる変数のファイルに対するアドレスの配列を返します。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) 処理ロジック変更
	 * @og.rev 3.8.0.2 (2005/06/30) clms の null時対応
	 *
	 * @param   names   String[] ファイルヘッダーの名称配列
	 * @param   clms    String[] SQLにバインドされる名称のCSV文字列
	 * @return  int[] SQLにバインドされる変数のファイルに対するアドレスの配列
	 */
//	private int[] getColumnNos( final String[] names,final String[] clms ) {
//		int[] clmNos = new int[0];
//
//		if( clms != null ) {
//			clmNos = new int[clms.length];
//			for( int i=0; i<clms.length; i++ ) {
//				boolean flag = true;
//				for( int j=0; flag && j<names.length; j++ ) {
//					if( clms[i].equalsIgnoreCase( names[j] ) ) {
//						clmNos[i] = j;
//						flag = false;
//					}
//				}
//				// 見つからない場合はエラー
//				if( flag ) {
//					String errMsg = "指定のカラム名は、ファイルのカラム列に一致しません。"
//								+ " カラム=[" + clms[i] + "]"
//								+ " カラム列=[" + StringUtil.array2csv( names ) + "]" ;
//					throw new HybsSystemException( errMsg );
//				}
//			}
//		}
//		return clmNos;
//	}

	/**
	 * BufferedReader を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した BufferedReader を作成します。
	 *
	 * @return   BufferedReader 指定の読み取り用BufferedReader
	 */
	private BufferedReader getBufferedReader() {
		if( filename == null ) {
			String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}
		String directory = HybsSystem.url2dir( fileURL );
		File file = new File( StringUtil.urlAppend( directory,filename ) );

		BufferedReader out = FileUtil.getBufferedReader( file,encode );

		return out ;
	}

	/**
	 * フォーマットをセットします。
	 * UPDATE GExx SET CLMA = [CLMA] WHERE KEY1 = [KEY1] 形式より、
	 * [CLMA]、[KEY1] を順に、clmKeys 配列に格納します。
	 * また、入力フォーマットのは、[カラム名]の個所を ? に置き換えた
	 * 値を、返します。このSQL文は、そのまま PreparedStatement に渡します。
	 * 先のSQL文なら、UPDATE GExx SET CLMA = ? WHERE KEY1 = ? に、変換します。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) 新規追加
	 *
	 * @param	fmt  [カラム名] 形式のフォーマットデータ
	 * @return	 [カラム名]  を ? に置き換えた文字列
	 */
//	private String makeFormat( final String fmt ) {
//		StringBuilder buf = new StringBuilder( fmt.length() );
//
//		String fmt2 = fmt.replace( '[',']' );
//		CSVTokenizer token = new CSVTokenizer( fmt2,']',false );
//		int count = token.countTokens() / 2 ;
//		clmKeys = new String[ count ];
//
////		String format = null;
//		for( int i=0; i<count; i++ ) {
////			format = token.nextToken();
////			buf.append( format);
//			buf.append( token.nextToken() );	// format
//			clmKeys[i] = token.nextToken();
//			buf.append( "?" );		// [カラム] を ? に置き換え
//		}
////		format = token.nextToken();
////		buf.append( format );
//		buf.append( token.nextToken() );	// format
//
//		return buf.toString();
//	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】可変長ファイルを作成するときの項目区切り文字をセットします(初期値:タブ)。
	 *
	 * @og.tag 可変長ファイルを作成するときの項目区切り文字をセットします。
	 *
	 * @param   separator 項目区切り文字
	 */
	public void setSeparator( final String separator ) {
		this.separator = nval( getRequestParameter( separator ),this.separator );
	}

	/**
	 * 【TAG】読み取り元ディレクトリ名を指定します(初期値:システムパラメータのFILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリより、ファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダの下より、読み取ります。
	 *
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルURL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします(初期値:システムパラメータのFILE_FILENAME)。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * 初期値は、システムパラメータ の FILE_FILENAME 属性で指定のファイル名です。
	 *
	 * @param   filename ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】ファイルを作成するときのファイルエンコーディング名をセットします(初期値:システムパラメータのFILE_ENCODE)。
	 *
	 * @og.tag
	 * 初期値は、システムパラメータ の FILE_ENCODE 属性で、設定しています。
	 * Shift_JIS,MS932,Windows-31J,UTF-8,ISO-8859-1,UnicodeLittle･･･
	 *
	 * @param   enc ファイルエンコーディング名
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0040[　件登録しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 * ※ この属性には、リクエスト変数({&#064;XXXX})は使用できません。
	 *
	 * @param   id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		if( id != null ) { displayMsg = id; }
	}

	/**
	 * 【TAG】#NAME 属性の代わりとなるファイルのカラム名を CSV形式で指定します。
	 *
	 * @og.tag
	 * データファイルの先頭行に、#NAME 行があり、読み取るべきファイルの
	 * カラム名が記述されています。通常は、このカラム名を取り込んで、
	 * 各データ列のカラムを指定します。
	 * この属性は、ファイルに#NAME 行が存在しない(他システムからの入力ファイル等)
	 * 場合に、#NAME 属性の代わりに、カラム名を外部より指定します。
	 *
	 * @og.rev 3.8.5.1 (2006/05/08) getCSVParameter の使用を中止
	 *
	 * @param   clms ファイルのカラム名（カンマ区切り文字）
	 */
	public void setColumns( final String clms ) {
		columns = StringUtil.csv2Array( nval( getRequestParameter( clms ),null ),',' );
	}

	/**
	 * 【TAG】指定数毎にコミットを発行します(初期値:0 終了までコミットしません)。
	 *
	 * @og.tag
	 * 通常は、全ての処理が正常に終了するか、なにもしないか（トランザクション）
	 * を判断すべきで、途中でのコミットはしません。
	 * しかし、場合によって、件数が異常に多い場合や、再実行可能な場合は、
	 * 途中でコミットして、都度、処理できるものだけを処理してしまうという方法があります。
	 * また、ロールバックエリアの関係などで、データ量が多い場合に、処理時間が異常に
	 * 長くなる事があり、指定件数ごとのコミット機能を用意しています。
	 * 0 に設定すると、終了までコミットしません。初期値は、0 です。
	 *
	 * @param   cmtBat 指定数毎にコミットを発行(初期値：0)
	 */
	public void setCommitBatch( final String cmtBat ) {
		commitBatch = nval( getRequestParameter( cmtBat ),commitBatch );
	}

	/**
	 * 【TAG】カラムチェック(DBTypeチェック)を行うかどうかを設定します(初期値：false)
	 *
	 * @og.tag
	 * カラムの整合性チェックを行う場合、この属性を設定(true)します。
	 * 初期値は、行わない(false)です。
	 * チェックするカラムは、#NAME や columns で指定されたカラムではなく、
	 * BODY部のSQL文で指定されたカラム名( [カラム名] )です。これは、直接、SQL文中に
	 * 記述している値や、{&#064;XXXX}文字等は、チェック出来ない為です。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) 新規追加 取り込み時全チェック
	 *
	 * @param   flag チェックを行うかどうか（true:行う/false:行わない）
	 * @see     #setUseColumnAdjust( String )
	 */
	public void setUseColumnCheck( final String flag ) {
		useColumnCheck = nval( getRequestParameter( flag ),useColumnCheck );
	}

	/**
	 * 【TAG】カラム変換(DBType変換)を行うかどうかを設定します(初期値：false)
	 *
	 * @og.tag
	 * カラムの変換を行う場合、この属性を設定(true)します。
	 * 初期値は、行わない(false)です。
	 * 変換するカラムは、#NAME や columns で指定されたカラムではなく、
	 * BODY部のSQL文で指定されたカラム名[カラム名]です。これは、直接、SQL文中に
	 * 記述している値や、{&#064;XXXX}文字等は、変換出来ない為です。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) 新規追加 取り込み時変換
	 *
	 * @param   flag 変換を行うかどうか（true:行う/false:行わない）
	 * @see     #setUseColumnCheck( String )
	 */
	public void setUseColumnAdjust( final String flag ) {
		useColumnAdjust = nval( getRequestParameter( flag ),useColumnAdjust );
	}

	/**
	 * 【TAG】NULL チェックすべきカラム列をカンマ区切りで指定します。
	 *
	 * @og.tag nullCheck="AAA,BBB,CCC,DDD"
	 * <del>先に配列に分解してからリクエスト変数の値を取得するようにします。
	 * こうする事で、リクエストにカンマ区切りの値を設定できるようになります。</del>
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.8.0.2 (2005/06/30) 新規追加
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param   clms String
	 */
	public void setNullCheck( final String clms ) {
		nullCheck = StringUtil.csv2Array( getRequestParameter( clms ) );
		if( nullCheck.length == 0 ) { nullCheck = null; }
	}

	/**
	 * カラム変換、カラムチェックを行う内部クラス
	 *
	 * @og.rev 4.0.0 (2005/01/31) static クラス化、引数にResourceManager追加
	 * @og.group ファイル入力
	 *
	 * @version  4.0
	 * @author   Kazuhiko Hasegawa
	 * @since    JDK5.0,
	 */
	static class CheckColumnData {
		private final DBColumn[] dbClm    ;
		private final int[]      clmChkNo ;
		private final int        len      ;		// 長さ０の時は、なにもしない。
		private final ErrorMessage errMsg = new ErrorMessage( "Check Columns Error!" );

		/**
		 * コンストラクター
		 *
		 * @param clmNo int[]
		 * @param chkClm String[]
		 * @param resource ResourceManager
		 */
		CheckColumnData( final int[] clmNo, final String[] chkClm,final ResourceManager resource ) {
			if( clmNo  == null || clmNo.length  == 0 ||
				chkClm == null || chkClm.length == 0 ) { // return; }	// 何もしない

				dbClm    = null;
				clmChkNo = null;
				len      = 0;
			}
			else {
				clmChkNo = clmNo ;
				len = clmNo.length ;
				dbClm = new DBColumn[len];
				for( int i=0; i<len; i++ ) {
					dbClm[i] = resource.makeDBColumn( chkClm[i] );	// 4.0.0 (2005/01/31)
				}
			}
		}

		/**
		 * 引数のデータを DBColumn で正規化(valueSetメソッド経由)します。
		 *
		 * @param data String[]
		 * @return String[]
		 * @see org.opengion.hayabusa.db.DBColumn#valueSet( String )
		 */
		String[] adjustData( final String[] data ) {
			if( len == 0 ) { return data; }
			String[] ajstData = new String[len];
			for( int i=0; i<len; i++ ) {
				String val = data[clmChkNo[i]];
				ajstData[i] = dbClm[i].valueSet( val );
			}
			return ajstData ;
		}

		/**
		 * 引数のデータを DBColumn で正規化(valueSetメソッド経由)します。
		 *
		 * @param row int
		 * @param data String[]
		 * @return ErrorMessage
		 * @see org.opengion.hayabusa.db.DBColumn#valueSet( String )
		 */
		ErrorMessage checkData( final int row,final String[] data ) {
			for( int i=0; i<len; i++ ) {
				String val = data[clmChkNo[i]];
				errMsg.append( row+1,dbClm[i].valueCheck( val ) );
			}
			return errMsg ;
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "separator"		,separator		)
				.println( "fileURL"			,fileURL		)
				.println( "filename"		,filename		)
				.println( "encode"			,encode			)
				.println( "displayMsg"		,displayMsg		)
				.println( "columns"			,columns		)
				.println( "clmKeys"			,clmKeys		)
				.println( "sql"				,sql			)
				.println( "commitBatch"		,commitBatch	)
				.println( "useColumnCheck"	,useColumnCheck	)
				.println( "useColumnAdjust"	,useColumnAdjust)
				.println( "nullCheck"		,nullCheck		)
				.println( "dyStart"			,dyStart		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
