/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.html.TableFormatter;

/**
 * ヘッダ、フッタ、ボディを指定して作成する、自由レイアウトが可能な、カスタムテーブル表示クラスです。
 * このクラスは、ViewForm_HTMLFormatTable クラスの代替えとしても使用できます。
 * その場合は、thead のみ指定すれば、同じフォームが tbody にも適用されます。
 * これは、まさに、ViewForm_HTMLFormatTable と同じです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.rev 3.7.1.1 (2005/05/23) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_CustomData extends ViewForm_HTMLTable	{
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private TableFormatter 		headerFormat	= null;
	private TableFormatter[]	bodyFormats		= null;
	private TableFormatter		footerFormat	= null;
	private int					bodyFormatsCount = 0;

	private static final int BODYFORMAT_MAX_COUNT = 10;

	// 4.3.4.4 (2009/01/01)
//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public ViewForm_CustomData() {
//		super();
//	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) フォーマットが設定されていない場合のエラー追加・編集行のみを表示する属性(isSkipNoEdit)追加
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( headerFormat == null ) {
			String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		headerLine	 = null;		// 3.5.3.1 (2003/10/31) キャッシュクリア

		int lastNo = getLastNo( startNo, pageSize );

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		headerFormat.makeFormat( getDBTableModel() );	// 3.5.6.2 (2004/07/05) 移動

		if( bodyFormatsCount != 0 ) {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
			}
		}

		out.append( getHeader() );
		for( int row=startNo; row<lastNo; row++ ) {
//			if( isSkip( row ) ) { continue; }		// 3.5.3.1 (2003/10/31)
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			for( int i=0; i<bodyFormatsCount; i++ ) {
				TableFormatter bodyFormat = bodyFormats[i];
				if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)

				int cl = 0;
				for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
					String fmt = bodyFormat.getFormat(cl);
					int loc = bodyFormat.getLocation(cl);	// 3.5.5.0
					out.append( fmt );			// 3.5.0.0

					if( loc >= 0 ) {
						switch( bodyFormat.getType(cl) ) {
							case '#' : out.append( getColumnLabel(loc) );		break;
							case '$' : out.append( getRendererValue(row,loc) );	break;
							case '!' : out.append( getValue(row,loc) );			break;
							default  : out.append( getValueLabel(row,loc) );	break;
						}
					}
					else {
						out.append( bodyFormat.getSystemFormat(row,loc) );
					}
				}
				out.append( bodyFormat.getFormat(cl) );
			}
		}

		if( footerFormat != null ) {
			out.append( getTableFoot() );
		}

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		headerFormat			= null;
		bodyFormats				= null;
		footerFormat			= null;
		bodyFormatsCount		= 0;
	}

	/**
	 * DBTableModel から テーブルのヘッダータグ文字列を作成して返します。
	 *
	 * <del>これは、内部的にキャッシュしているため、状況が変更される都度に、
	 * キャッシュをクリアする必要があります。
	 *
	 * @return	テーブルのヘッダータグ文字列
	 */
	@Override
	protected String getHeader() {
		if( headerFormat == null ) { return ""; }	// 存在しないケース
		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		int cl = 0;
		for( ; cl < headerFormat.getLocationSize(); cl++ ) {
			buf.append( headerFormat.getFormat(cl) );
			int loc = headerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
		}
		buf.append( headerFormat.getFormat(cl) ).append( HybsSystem.CR );

		headerLine = buf.toString();
		return headerLine;
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getTableFoot() {
		footerFormat.makeFormat( getDBTableModel() );

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		int cl = 0;
		for( ; cl < footerFormat.getLocationSize(); cl++ ) {
			int loc = footerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
		}
		buf.append( footerFormat.getFormat(cl) ).append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		for( int i=0; i<list.size(); i++ ) {
			TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応

			switch( format.getFormatType() ) {
				case TYPE_HEAD : headerFormat = format; break;
				case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
				case TYPE_FOOT : footerFormat = format; break;
				default : String errMsg = "FormatterType の定義外の値が指定されました。";
						// 4.3.4.4 (2009/01/01)
						  throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}
}
