/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.io.BufferedReader;
import java.util.Map;

import org.opengion.fukurou.util.CSVTokenizer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.CodeData;

/**
 * 指定の区切り記号(初期値:タブ区切り)ファイルの読み取りクラスです。
 *
 * 名前，データの入力部のみオーバーライドすれば，各種入力フォーマットに合わせた
 * サブクラスを実現する事が可能です。
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public abstract class AbstractTableReader implements TableReader {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String	separator	= TAB_SEPARATOR;		// 項目区切り文字
	private ResourceManager resource = null;			// 4.0.0 (2005/01/31)
	private int		maxRowCount	= HybsSystem.sysInt( "DB_MAX_ROW_COUNT" ) ;

	protected DBTableModel	table		= null;
	protected DBColumn[]	dbColumn	= null;

	// 3.5.4.5 (2004/01/23) カラム名の外部指定を出来る様にする。
//	private String	  columns	= null;	 // 外部指定のカラム名
	protected String  columns	= null;	 // 外部指定のカラム名 ( 4.3.4.7 (2009/01/22) protectedに変更 )
	private String	  encode	= null;
	private boolean	  useNumber	= true;			// 3.7.0.5 (2005/04/11)

	private int		  skipRowCount	= 0;		// 5.1.6.0 (2010/05/01) データの読み飛ばし設定
	private boolean	  useRenderer	= false;	// 5.2.1.0 (2010/10/01)

	// 5.2.1.0 (2010/10/01) コードリソース毎のラベル逆引きマップ
	private Map<?,?>[] maps = null;				// 5.5.1.7 (2012/04/16) ワイルドカードを指定

	private boolean	  useDebug	= false;		// 5.5.7.2 (2012/10/09) デバッグ情報の出力するかどうか

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @see #isExcel()
	 */
	abstract public void readDBTable();

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @param   reader BufferedReaderオブジェクト
	 */
	abstract public void readDBTable( final BufferedReader reader );

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語(ロケール)に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  resource リソースマネージャー
	 */
	public void setResourceManager( final ResourceManager resource ) {
		this.resource = resource;
	}

	/**
	 * DBColumn オブジェクトをDBTable に設定します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) private を protected に変更。
	 * @og.rev 3.5.4.5 (2004/01/23) DBColumn 配列に値をセットします。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer対応(コードリソース毎のラベル逆引き)
	 *
	 * @param names カラム名配列
	 */
	protected void setTableDBColumn( final String[] names ) {
		dbColumn = new DBColumn[names.length] ;	 // 3.5.4.5 追加
		for( int i=0; i<names.length; i++ ) {
			DBColumn clm = resource.makeDBColumn( names[i] );
			table.setDBColumn( i,clm );
			dbColumn[i] = clm;		  // 3.5.4.5 追加
		}

		if( useRenderer ) {
			maps = new Map<?,?>[names.length];		// 5.5.1.7 (2012/04/16) ワイルドカードを指定
			for( int i=0; i<names.length; i++ ) {
				CodeData cd = dbColumn[i].getCodeData();
				if( cd != null ) { maps[i] = cd.makeLabelMap(); }
				else             { maps[i] = null; }
			}
		}
	}

	/**
	 * DBTableModelオブジェクトに、１行分のデータを追加します。
	 * これ自体は、メソッドの共通化による 拡張をしやすくするために用意しました。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param values １行分のデータ配列
	 */
	protected void setTableColumnValues( final String[] values ) {
		if( useRenderer ) {
			int clmSize = values.length;
			for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
				if( maps[clmNo] != null ) {
					String val = values[clmNo];
					if( val == null ) { val = ""; }
					else {
						String tmp = (String)maps[clmNo].get( val );
						if( tmp != null ) { values[clmNo] = tmp; }
						else {
							int sp = val.indexOf( ':' );
							if( sp >= 0 ) {
								values[clmNo] = val.substring( 0,sp );
							}
						}
					}
				}
			}
		}

		table.addColumnValues( values );
	}

	/**
	 * １行のデータを テーブルモデルにセットするように分割します。
	 *
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 * データの分割は、separator文字を使用します。
	 *
	 * @og.rev 3.3.3.1 (2003/07/18) ファイルリード/ライト時に後ろスペースの除去を行います。
	 * @og.rev 3.7.0.5 (2005/04/11) useNumber 属性を考慮します。
	 *
	 * @param	data	１行のデータ
	 * @param	clmSize	カラム数
	 *
	 * @return	分割された文字列配列
	 */
	protected String[] readData( final String data,final int clmSize ) {
		String[] rtnData = new String[ clmSize ];
		CSVTokenizer token = new CSVTokenizer( data,separator.charAt(0) );
		// 超イレギュラー処理 最初の separator 以前の文字は無視する。
		// 3.7.0.5 (2005/04/11)
		if( useNumber ) { token.nextToken(); }	  // 先頭は行番号のため無視する。

		int clmNo = 0;
		while( token.hasMoreTokens() ) {
			String val = StringUtil.csvOutQuote( token.nextToken() );
			if( val != null && val.startsWith( "'0" ) ) {
				rtnData[clmNo++] = StringUtil.rTrim( val.substring( 1 ) );
			}
			else {
				rtnData[clmNo++] = StringUtil.rTrim( val );
			}
			if( clmNo >= clmSize ) { break; }	// 3.7.0.5 (2005/04/11) 多い場合は、以降を無視する。
		}
		// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
		for( int i=clmNo; i<clmSize; i++ ) {
			rtnData[i] = "";
		}

		return rtnData;
	}

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return	テーブルモデル
	 */
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   sep 区切り文字
	 */
	public void setSeparator( final String sep ) {
		if( sep != null ) { this.separator = sep; }
	}

	/**
	 * データを書き込む場合の,区切り文字を返します。
	 *
	 * @return	区切り文字
	 */
	public String getSeparator() {
		return separator;
	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	public int getMaxRowCount() {
		return maxRowCount;
	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   maxRowCount 最大検索件数
	 */
	public void setMaxRowCount( final int maxRowCount ) {
		this.maxRowCount = maxRowCount;
	}

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 */
	public void setSheetName( final String sheetName ) {
		String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、カンマ区切りで、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	public void setSheetNos( final String sheetNos ) {
		String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか
	 */
	public boolean isExcel() {
		return false;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	public void setFilename( final String filename ) {
		String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 読み取り元ファイルのカラム列を、外部(タグ)より指定します。
	 * ファイルに記述された #NAME より優先して使用されます。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   clms 読み取り元ファイルのカラム列(カンマ区切り文字)
	 */
	public void setColumns( final String clms ) {
		columns = clms ;
	}

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	public void setEncode( final String enc ) {
		encode = enc;
	}

	/**
	 * 読み取り元ファイルのエンコード文字列を取得します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @return	ファイルのエンコード文字列
	 */
	protected String getEncode() {
		return encode;
	}

	/**
	 * 行番号情報を指定[true:使用している/false:していない]します(初期値:true)。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 *
	 * @param	useNumber	行番号情報  [true:使用する/false:使用しない]
	 */
	public void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * データの読み始めの初期値を取得します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @return	読み始めの初期値
	 */
	public int getSkipRowCount() {
		return skipRowCount ;
	}

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	count 読み始めの初期値
	 */
	public void setSkipRowCount( final int count ) {
		skipRowCount = count;
	}

	/**
	 * 読取処理でラベルをコードリソースに逆変換を行うかどうかを指定します。
	 *
	 * TableWriter_Renderer 系のクラスで出力した場合は、コードリソースがラベルで出力されます。
	 * そのファイルを読み取ると、当然、エラーになります。
	 * ここでは、コードリソースのカラムに対して、ラベルからコードを求める逆変換を行うことで、
	 * Renderer 系で出力したファイルを取り込むことができるようにします。
	 *
	 * ここでは、TableWriter 系と同様に、TableReader_Renderer 系のクラスを作るのではなく、
	 * 属性値のフラグで、制御します。
	 * 将来的には、TableWriter 系も廃止して、同様のフラグで制御するように変更する予定です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	useRenderer	コードリソースのラベル変換を行うかどうかを指定
	 */
	public void setUseRenderer( final boolean useRenderer ) {
		this.useRenderer = useRenderer;
	}

	/**
	 * 行番号情報を、使用している(true)/していない(false)を返します。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.0.0 (2007/07/20) メソッド名変更(getUseNumber() ⇒  isUseNumber())
	 *
	 * @return	行番号情報を、使用している(true)/していない(false)を指定
	 */
	protected boolean isUseNumber() {
		return useNumber ;
	}

	/**
	 * デバッグ情報を出力するかどうかを指定します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 * 通常は使用しませんので、設定を無視します。
	 * 初期値は、false:デバッグ情報を出力しない です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param	useDebug	デバッグ情報を出力するかどうかを指定
	 */
	public void setDebug( final boolean useDebug ) {
		this.useDebug = useDebug;
	}

	/**
	 * デバッグ情報を出力するかどうかを取得します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @return	デバッグ情報を出力するかどうか(true:する/false:しない)
	 */
	protected boolean isDebug() {
		return useDebug ;
	}
}
