/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceManager;

import java.io.BufferedReader;

/**
 * DBTableModel インターフェース のオブジェクトをReader を用いて入力する為の，共通インターフェースです。
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface TableReader {

	/**
	 * ヘッダー情報の入力時の区切り文字
	 */
	String TAB_SEPARATOR = "\t";		// 項目区切り文字

	/**
	 * DBTableModel から 各形式のデータを作成して,Reader より読み取ります。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 *
	 * @see #isExcel()
	 */
	void readDBTable() ;

	/**
	 * DBTableModel から 各形式のデータを作成して,Reader より読み取ります。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取ル要に変更します。
	 *
	 * @param   reader BufferedReaderオブジェクト
	 */
	void readDBTable( final BufferedReader reader ) ;

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語(ロケール)に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  resource リソースマネージャー
	 */
	void setResourceManager( final ResourceManager resource ) ;

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return  DBTableModelオブジェクト
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @param   separator 区切り文字
	 */
	void setSeparator( final String separator ) ;

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	int getMaxRowCount() ;

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @param   maxRowCount 最大検索件数
	 */
	void setMaxRowCount( final int maxRowCount ) ;

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 * @see		#setSheetNos( String ) 
	 */
	void setSheetName( final String sheetName ) ;

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、カンマ区切りで、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	void setSheetNos( final String sheetNos ) ;

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか
	 */
	boolean isExcel() ;

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	void setFilename( final String filename ) ;

	/**
	 * 読み取り元ファイルのカラム列を、外部(タグ)より指定します。
	 * ファイルに記述された #NAME より優先して使用されます。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   clms 読み取り元ファイルのカラム列(カンマ区切り文字)
	 */
	void setColumns( final String clms ) ;

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	void setEncode( final String enc ) ;

	/**
	 * 行番号情報を、使用している(true)/していない(false)を指定します。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 *
	 * @param	useNumber	行番号情報 [true:使用している/false:していない]
	 */
	void setUseNumber( final boolean useNumber ) ;

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	count 読み始めの初期値
	 */
	void setSkipRowCount( final int count ) ;

	/**
	 * 読取処理でラベルをコードリソースに逆変換を行うかどうかを指定します。
	 *
	 * TableWriter_Renderer 系のクラスで出力した場合は、コードリソースがラベルで出力されます。
	 * そのファイルを読み取ると、当然、エラーになります。
	 * ここでは、コードリソースのカラムに対して、ラベルからコードを求める逆変換を行うことで、
	 * Renderer 系で出力したファイルを取り込むことができるようにします。
	 *
	 * ここでは、TableWriter 系と同様に、TableReader_Renderer 系のクラスを作るのではなく、
	 * 属性値のフラグで、制御します。
	 * 将来的には、TableWriter 系も廃止して、同様のフラグで制御するように変更する予定です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	useRenderer	コードリソースのラベル逆変換を行うかどうかを指定
	 */
	void setUseRenderer( final boolean useRenderer ) ;

	/**
	 * デバッグ情報を出力するかどうかを指定します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 * 通常は使用しませんので、設定を無視します。
	 * 初期値は、false:デバッグ情報を出力しない です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param	useDebug	デバッグ情報を出力するかどうかを指定
	 */
	void setDebug( final boolean useDebug ) ;
}
