/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.daemon;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsTimerTask;
import org.opengion.fukurou.util.URLConnect;
import org.opengion.fukurou.util.XHTMLTag;

import java.io.IOException;
import java.util.Date;

/**
 * 【URLアクセス】
 * 指定したパラメータでURLに接続します。
 * このクラスは、HybsTimerTask を継承した タイマータスククラスです。
 * startDaemon() がタイマータスクによって、呼び出されます。
 *
 * 接続のためのパラメータは以下です
 * url			 : 接続先URL(必須)
 * proxyHost	 : プロキシのホスト名
 * proxyPort	 : プロキシのポート番号
 * useSystemUser : デフォルトのユーザ/パスワードを利用するか(初期値:true)
 * 					trueの場合はSYSTEM:*********を利用します。
 * authUserPass	 : ユーザとパスワードをUSER:PASSWORDの形で記述
 * keys			 : リクエストパラメータのキー(カンマ区切り)
 * vals			 : リクエストパラメータの値(カンマ区切り)
 * method		 : POSTかGETを指定(初期値:GET)
 * debug		 : 接続したページを受信して、ログに書き出します(初期値:false)
 *
 * 接続エラー時のログはファイル(SYS_LOG_URL)に出力されます。
 *
 * @og.rev 4.3.4.4 (2009/01/01) 新規作成
 * @og.group デーモン
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class Daemon_URLConnect extends HybsTimerTask {

	private static final String DEFAULT_USER = "SYSTEM:MANAGER" ;
	private static final int LOOP_COUNTER = 24; // カウンタを24回に設定

	private int loopCnt = 0;

	private boolean debug = false;
	private boolean	useSystemUser	= true;
	private String	method			= "GET";
	private String	urlStr			= null;
	private int		proxyPort		= -1;

	private URLConnect conn			= null;

	/**
	 * このタイマータスクによって初期化されるアクションです。
	 * パラメータを使用した初期化を行います。
	 *
	 */
	@Override
	public void initDaemon() {
		debug = StringUtil.nval( getValue( "DEBUG" ),debug ) ;
		useSystemUser		= StringUtil.nval( getValue( "useSystemUser" ), useSystemUser );
		method				= StringUtil.nval( getValue( "method" ), method );
		urlStr				= getValue( "url" );
		proxyPort			= StringUtil.nval( getValue( "proxyPort" ), proxyPort );
		String proxyHost	= getValue( "proxyHost" );
		String keys			= getValue( "keys" );
		String vals			= getValue( "vals" );
		String authUserPass	= getValue( "authUserPass" );

		if( useSystemUser ) { authUserPass = DEFAULT_USER; }

		String urlEnc = XHTMLTag.urlEncode( keys,vals );

		if( ! "POST".equals( method ) ) {
			urlStr = XHTMLTag.addUrlEncode( urlStr,urlEnc );
		}
		conn = new URLConnect( urlStr,authUserPass );

		if( proxyHost != null ) {
			conn.setProxy( proxyHost,proxyPort );
		}

		if( "POST".equals(method) && keys != null && vals != null ) {
			conn.setPostData( urlEnc );
		}
	}

	/**
	 * タイマータスクのデーモン処理の開始ポイントです。
	 *
	 */
	@Override
	protected void startDaemon() {
		if( loopCnt % LOOP_COUNTER == 0 ) {
			loopCnt = 1;
			System.out.println( toString() + " " + new Date()  + " " );
		}
		else {
			loopCnt++ ;
		}
		// URLへのconnect及びデータ取得実行
		try{
			conn.connect();

			if(debug){
				// System.out.println( conn.readData() );
				String debugMsg = "Daemon_URLConnect:url=[" + urlStr + "]" + HybsSystem.CR
								+ conn.readData();
				LogWriter.log( debugMsg );
			}
		}
		catch( IOException ex ) {
			System.out.println(ex);
			String errMsg = "Daemon_URLConnect:データ取得中にエラーが発生しました。" + HybsSystem.CR
						+ " url=[" + urlStr + "]" + HybsSystem.CR
						+ ex;
			LogWriter.log( errMsg );
		}
		finally {
			if( conn != null ) { conn.disconnect(); }
		}
	}

	/**
	 * このタイマータスクのcancel() メソッドをオーバーライドします。
	 * HybsTimerTaskManager#cancelTask( int ) を実行します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) super.cancel() のみ実行なら、オーバーライドの必要はない
	 *
	 * @see java.util.TimerTask#cancel()
	 */
//	public boolean cancel() {
//		return super.cancel();
//	}
}
