/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
// import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
// import java.text.SimpleDateFormat;
// import java.util.Locale;
import java.util.List;
import java.util.ArrayList;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelUtil;

/**
 * POI による、EXCELバイナリファイルを読み取る実装クラスです。
 *
 * ファイル名、シート名を指定して、データを読み取ることが可能です。
 * 第一カラムが # で始まる行は、コメント行なので、読み飛ばします。
 * カラム名の指定行で、カラム名が null の場合は、その列は読み飛ばします。
 *
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_Excel extends TableReader_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String  filename		= null;		// 3.5.4.3 (2004/01/05)
	private String  sheetName		= null;		// 3.5.4.2 (2003/12/15)
	private String  sheetNos		= null;		// 5.5.7.2 (2012/10/09)

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.6.0 (2010/05/01) columns 処理 追加
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCountの追加
	 * @og.rev 5.1.8.0 (2010/07/01) Exception をきちっと記述(InvalidFormatException)
	 * @og.rev 5.2.1.0 (2010/10/01) setTableColumnValues メソッドを経由して、テーブルにデータをセットする。
	 * @og.rev 5.5.1.2 (2012/04/06) HeaderData を try の上にだし、エラーメッセージを取得できるようにする。
	 * @og.rev 5.5.7.2 (2012/10/09) sheetNos 追加による複数シートのマージ読み取りサポート
	 *
	 * @see #isExcel()
	 */
	@Override
	public void readDBTable() {
		InputStream  in = null;
		HeaderData data = null;		// 5.5.1.2 (2012/04/06)
		try {
			boolean isDebug = isDebug();			// 5.5.7.2 (2012/10/09) デバッグ情報

			if( isDebug ) { System.out.println( " Filename=" + filename ) ; }

			in = new FileInputStream(filename);

//			POIFSFileSystem fs = new POIFSFileSystem(in);
//			HSSFWorkbook wb = new HSSFWorkbook(fs);
//			HSSFSheet sheet ;
			Workbook wb = WorkbookFactory.create(in);
//			Sheet sheet ;
			Sheet[] sheets ;									// 5.5.7.2 (2012/10/09) 配列に変更

			if( isDebug ) { wb = ExcelUtil.activeWorkbook( wb ); }		// デバッグモード時には、エクセルのアクティブセル領域のみにシュリンクを行う

			// 5.5.7.2 (2012/10/09) 複数シートのマージ読み取り。 sheetNos の指定が優先される。
			if( sheetNos != null && sheetNos.length() > 0 ) {
				String[] sheetList = StringUtil.csv2ArrayExt( sheetNos , wb.getNumberOfSheets()-1 );	// 最大シート番号は、シート数-1
				sheets = new Sheet[sheetList.length];
				for( int i=0; i<sheetList.length; i++ ) {
					sheets[i] = wb.getSheetAt( Integer.parseInt( sheetList[i] ) );
				}
			}
			else if( sheetName != null && sheetName.length() > 0 ) {
				Sheet sheet = wb.getSheet( sheetName );
				if( sheet == null ) {
					String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]" ;
					throw new HybsSystemException( errMsg );
				}
				sheets = new Sheet[] { sheet };
			}
			else {
				Sheet sheet = wb.getSheetAt(0);
				sheets = new Sheet[] { sheet };
			}

//			if( sheetName == null || sheetName.length() == 0 ) {
//				sheet = wb.getSheetAt(0);
//			}
//			else {
//				sheet = wb.getSheet( sheetName );
//				if( sheet == null ) {
//					String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]" ;
//					throw new HybsSystemException( errMsg );
//				}
//			}

			boolean  nameNoSet = true;
			table = DBTableModelUtil.newDBTable();

			int numberOfRows = 0;
//			HeaderData data = new HeaderData();
			data = new HeaderData();				// 5.5.1.2 (2012/04/06)

			// 5.1.6.0 (2010/05/01) columns 処理
			data.setUseNumber( isUseNumber() );
			if( data.setColumns( columns ) ) {
				nameNoSet = false;
				table.init( data.getColumnSize() );
				setTableDBColumn( data.getNames() ) ;
			}

			int skip = getSkipRowCount();					// 5.1.6.0 (2010/05/01)
			// 5.5.7.2 (2012/10/09) 複数シートのマージ読み取り。 
			for( int i=0; i<sheets.length; i++ ) {			// 5.5.7.2 (2012/10/09) シート配列を処理します。
				Sheet sheet = sheets[i] ;					// 5.5.7.2 (2012/10/09)
				int nFirstRow = sheet.getFirstRowNum();
				if( nFirstRow < skip ) { nFirstRow = skip; }	// 5.1.6.0 (2010/05/01)
				int nLastRow  = sheet.getLastRowNum();
				if( isDebug ) {		// 5.5.7.2 (2012/10/09) デバッグ情報
					System.out.println( " Debug: Row=" + numberOfRows + " : Sheet= " + sheet.getSheetName() + " , 開始=" + nFirstRow + " , 終了=" + nLastRow );
				}
				for( int nIndexRow = nFirstRow; nIndexRow <= nLastRow; nIndexRow++) {
	//				HSSFRow oRow = sheet.getRow(nIndexRow);
					Row oRow = sheet.getRow(nIndexRow);
					if( data.isSkip( oRow ) ) { continue; }
					if( nameNoSet ) {
						nameNoSet = false;
						table.init( data.getColumnSize() );
						setTableDBColumn( data.getNames() ) ;
					}

					if( numberOfRows < getMaxRowCount() ) {
						setTableColumnValues( data.row2Array( oRow ) );		// 5.2.1.0 (2010/10/01)
	//					table.addColumnValues( data.row2Array( oRow ) );
						numberOfRows ++ ;
					}
					else {
						table.setOverflow( true );
					}
				}

				// 最後まで、#NAME が見つから無かった場合
				if( nameNoSet ) {
					String errMsg = "最後まで、#NAME が見つかりませんでした。"
									+ HybsSystem.CR
									+ "ファイルが空か、もしくは損傷している可能性があります。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
		}
//		catch (Exception e) {
		catch ( IOException ex ) {
			String errMsg = "ファイル読込みエラー[" + filename + "]"  ;
			if( data != null ) { errMsg = errMsg + data.getLastCellMsg(); }		// 5.5.1.2 (2012/04/06)
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		// 5.1.8.0 (2010/07/01) Exception をきちっと記述
		catch (InvalidFormatException ex) {
			String errMsg = "ファイル形式エラー[" + filename + "]"  ;
			if( data != null ) { errMsg = errMsg + data.getLastCellMsg(); }		// 5.5.1.2 (2012/04/06)
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			Closer.ioClose( in );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取る要に変更します。
	 * @og.rev 4.0.0 (2006/09/31) UnsupportedOperationException を発行します。
	 *
	 * @param   reader 各形式のデータ(使用していません)
	 */
	@Override
	public void readDBTable( final BufferedReader reader ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 */
	@Override
	public void setSheetName( final String sheetName ) {
		this.sheetName = sheetName;
	}

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、カンマ区切りで、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	@Override
	public void setSheetNos( final String sheetNos ) {
		this.sheetNos = sheetNos;
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか(常にtrue)
	 */
	@Override
	public boolean isExcel() {
		return true;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	@Override
	public void setFilename( final String filename ) {
		this.filename = filename;
		if( filename == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new HybsSystemException( errMsg );
		}
	}
}

/**
 * EXCEL ネイティブのデータを処理する ローカルクラスです。
 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
 * 行情報(Row)から、カラムの配列の取得などを行います。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   儲
 * @since    JDK5.0,
 */
class HeaderData {
	private String[] names ;
//	private short[]  index;
	private int[]    index; // 4.3.4.0 (2008/12/01) POI3.2対応
	private int		 columnSize = 0;
	private boolean  nameNoSet = true;
	private boolean  useNumber = true;

	private String[] orgNames ;			// 5.5.1.2 (2012/04/06) オリジナルのカラム名
	private Cell     lastCell = null;	// 5.5.1.2 (2012/04/06) 最後に実行しているセルを保持(エラー時に使用する。)

	/**
	 * 行番号情報を、使用しているかどうか[true/false]を指定します(初期値:true)。
	 *
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	useNumber	行番号情報 [true:使用している/false:していない]
	 */
	void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * カラム名を外部から指定します。
	 * カラム名が、NULL でなければ、＃NAME より、こちらが優先されます。
	 * カラム名は、順番に、指定する必要があります。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	columns	EXCELのカラム列(CSV形式)
	 *
	 * @return true:処理実施/false:無処理
	 */
	boolean setColumns( final String columns ) {
		if( columns != null && columns.length() > 0 ) {
			names = StringUtil.csv2Array( columns );
			columnSize = names.length ;
			index = new int[columnSize];
			int adrs = (useNumber) ? 1:0 ;	// useNumber =true の場合は、１件目(No)は読み飛ばす。
			for( int i=0; i<columnSize; i++ ) { index[i] = adrs++; }
			nameNoSet = false;

			return true;
		}
		return false;
	}

	/**
	 * EXCEL ネイティブのデータを処理する ローカルクラスです。
	 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
	 * 行情報(Row)から、カラムの配列の取得などを行います。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 *
	 * @return true:コメント行/false:通常行
	 */
//	boolean isSkip( HSSFRow oRow ) {
	boolean isSkip( Row oRow ) {
		if( oRow == null ) { return true; }

//		short nFirstCell = oRow.getFirstCellNum();
		int nFirstCell = oRow.getFirstCellNum();
//		HSSFCell oCell = oRow.getCell(nFirstCell);
		Cell oCell = oRow.getCell(nFirstCell);
		String strText =  getValue( oCell );
		if( strText != null && strText.length() > 0 ) {
			if( nameNoSet ) {
				if( strText.equalsIgnoreCase( "#Name" ) ) {
					makeNames( oRow );
					nameNoSet = false;
					return true;
				}
				else if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。"
									+ HybsSystem.CR
									+ "可能性として、ファイルが、ネイティブExcelでない事が考えられます。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
			}
		}

		return nameNoSet ;
	}

	/**
	 * EXCEL ネイティブの行情報(Row)からカラム名情報を取得します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber(行番号情報を、使用している(true)/していない(false)を指定)
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber(行番号情報を、使用している(true)/していない(false)を指定)
	 * @og.rev 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 */
//	private void makeNames( final HSSFRow oRow ) {
	private void makeNames( final Row oRow ) {
		// 先頭カラムは、#NAME 属性行であるかどうかを、useNumber で判定しておく。
		short nFirstCell = (short)( (useNumber) ? 1:0 );
//		short nFirstCell = oRow.getFirstCellNum();
		short nLastCell  = oRow.getLastCellNum();

		orgNames = new String[nLastCell+1];		// 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得

		int maxCnt = nLastCell - nFirstCell;
		String[] names2 = new String[maxCnt];
//		short[]  index2 = new short[maxCnt];
		int[]  index2 = new int[maxCnt];

		// 先頭カラムは、#NAME 属性行である。++ で、一つ進めている。
//		for( short nIndexCell = ++nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
//		for( int nIndexCell = ++nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
		// 先頭カラムは、#NAME 属性行であるかどうかを、useNumber で判定しておく。
		for( int nIndexCell = nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
//			HSSFCell oCell = oRow.getCell(nIndexCell);
			Cell oCell = oRow.getCell(nIndexCell);
			String strText = getValue( oCell );

			orgNames[nIndexCell] = strText;		// 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得

			// #NAME 行が、ゼロ文字列の場合は、読み飛ばす。
			if( strText != null && strText.length() > 0 ) {
				names2[columnSize] = strText;
				index2[columnSize] = nIndexCell;
				columnSize++;
			}
		}

		// #NAME を使用しない場合：no欄が存在しないケース
		if( maxCnt == columnSize ) {
			names = names2;
			index = index2;
		}
		else {
			names = new String[columnSize];
//			index = new short[columnSize];
			index = new int[columnSize];
			System.arraycopy(names2, 0, names, 0, columnSize);
			System.arraycopy(index2, 0, index, 0, columnSize);
		}
	}

	/**
	 * カラム名情報を返します。
	 * ここでは、内部配列をそのまま返します。
	 *
	 * @return String[] カラム列配列情報
	 */
	String[] getNames() {
		return names;
	}

	/**
	 * カラムサイズを返します。
	 *
	 * @return	カラムサイズ
	 */
	int getColumnSize() {
		return columnSize;
	}

	/**
	 * カラム名情報を返します。
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 *
	 * @return String[] カラム列配列情報
	 */
//	String[] row2Array( final HSSFRow oRow ) {
	String[] row2Array( final Row oRow ) {
		if( nameNoSet ) {
			String errMsg = "#NAME が見つかる前にデータが見つかりました。";
			throw new HybsSystemException( errMsg );
		}

		String[] data = new String[columnSize];
		for( int i=0;i<columnSize; i++ ) {
//			HSSFCell oCell = oRow.getCell( index[i] );
			Cell oCell = oRow.getCell( index[i] );
			data[i] = getValue( oCell );
		}
		return data;
	}

	/**
	 * セルオブジェクト(Cell)から値を取り出します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 取り出し方法を少し修正
	 * @og.rev 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
	 *
	 * @param oCell Cell EXCELのセルオブジェクト
	 *
	 * @return	セルの値
	 */
//	private String getValue( final Cell oCell ) {
	private String getValue( final Cell oCell ) {
		lastCell = oCell;	// 5.5.1.2 (2012/04/06) 今から実行するセルを取得しておきます。

		if( oCell == null ) { return null; }

		String strText = "";
//		HSSFRichTextString richText;
		RichTextString richText;
		int nCellType = oCell.getCellType();
		switch(nCellType) {
//			case HSSFCell.CELL_TYPE_NUMERIC:
			case Cell.CELL_TYPE_NUMERIC:
					strText = getNumericTypeString( oCell );
					break;
//			case HSSFCell.CELL_TYPE_STRING:
			case Cell.CELL_TYPE_STRING:
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					break;
//			case HSSFCell.CELL_TYPE_FORMULA:
			case Cell.CELL_TYPE_FORMULA:
	// POI3.0		strText = oCell.getStringCellValue();
					// 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
					Workbook wb = oCell.getSheet().getWorkbook();
					CreationHelper crateHelper = wb.getCreationHelper();
					FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();

					try {
						strText = getValue(evaluator.evaluateInCell(oCell));
					}
					catch ( Throwable th ) {
//						String errMsg = "セルフォーマットが解析できません。[" + String.valueOf(oCell.getCellFormula()) + "]"
						String errMsg = "セルフォーマットが解析できません。[" + oCell.getCellFormula() + "]"
									+ getLastCellMsg();
						throw new HybsSystemException( errMsg,th );
					}
					break;

//					richText = oCell.getRichStringCellValue();
//					if( richText != null ) {
//						strText = richText.getString();
//					}
//					else {
//						strText = getNumericTypeString( oCell );
//					}
//					break;
//			case HSSFCell.CELL_TYPE_BOOLEAN:
			case Cell.CELL_TYPE_BOOLEAN:
					strText = String.valueOf(oCell.getBooleanCellValue());
					break;
//			case HSSFCell.CELL_TYPE_BLANK :
//			case HSSFCell.CELL_TYPE_ERROR:
			case Cell.CELL_TYPE_BLANK :
			case Cell.CELL_TYPE_ERROR:
					break;
			default :
				break;
		}
		return strText.trim();
	}

	/**
	 * セル値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param oCell Cell
	 *
	 * @return	数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 */
//	private String getNumericTypeString( final HSSFCell oCell ) {
	private String getNumericTypeString( final Cell oCell ) {
		final String strText ;

		double dd = oCell.getNumericCellValue() ;
//		if( HSSFDateUtil.isCellDateFormatted( oCell ) ) {
		if( DateUtil.isCellDateFormatted( oCell ) ) {
			strText = HybsDateUtil.getDate( DateUtil.getJavaDate( dd ).getTime() , "yyyyMMddHHmmss" );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用

// 			DateFormat dateFormat = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
// //			strText = dateFormat.format( HSSFDateUtil.getJavaDate( dd ) );
// 			strText = dateFormat.format( DateUtil.getJavaDate( dd ) );
		}
		else {
			NumberFormat numFormat = NumberFormat.getInstance();
			if( numFormat instanceof DecimalFormat ) {
				((DecimalFormat)numFormat).applyPattern( "#.####" );
			}
			strText = numFormat.format( dd );
		}
		return strText ;
	}

	/**
	 * 最後に実行しているセル情報を返します。
	 *
	 * エラー発生時に、どのセルでエラーが発生したかの情報を取得できるようにします。
	 *
	 * @og.rev 5.5.1.2 (2012/04/06) 新規追加
	 *
	 * @return	最後に実行しているセル情報の文字列
	 */
	String getLastCellMsg() {
		String lastMsg = null;

		if( lastCell != null ) {
			int rowNo = lastCell.getRowIndex();
			int celNo = lastCell.getColumnIndex();
			int no = lastCell.getColumnIndex();

			lastMsg = "Row=" + rowNo + ",Cel=" + celNo ;
			if( orgNames != null && orgNames.length < no ) {
				lastMsg = lastMsg + ", NAME=" + orgNames[no] ;
			}
		}
		return lastMsg;
	}
}
