/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringUtil;

import org.opengion.fukurou.mail.MailReceiveListener ;
import org.opengion.fukurou.mail.MailTX ;
import org.opengion.fukurou.mail.MailMessage ;
import org.opengion.fukurou.mail.MailAttachFiles ;

import java.util.Date;

/**
 * MailReceiveListener の実装クラスです。
 * MailRX クラスにセットすることで、メール１件ごとに receive( MailMessage ) メソッドが
 * 呼び出されます。
 * メール１件に、添付ファイルが複数登録されている場合の処理を行っています。
 * 添付ファイルごとに要求番号を採番して、要求番号.xls としてセーブし、帳票要求テーブルに
 * 書き込みます。
 *
 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ExcelInsertReceiveListener implements MailReceiveListener {

	// EXCEL取込時に使用するテンポラリフォルダ名。ファイル名は、要求番号.xls
	private final String EXCELIN_URL =
		HybsSystem.url2dir( StringUtil.nval(
								HybsSystem.sys( "EXCEL_IN_FILE_URL" ) ,
								HybsSystem.sys( "FILE_URL"   ) + "EXCELIN/" ) ) ;

	private GE50Access ge50 = null;

	/**
	 * メール受信処理で、１メール受信ごとに呼び出されます。
	 * 処理結果を、boolean で返します。
	 *
	 * @param message MailMessageオブジェクト
	 *
	 * @return	処理結果(正常:true / 異常:false)
	 */
	public boolean receive( final MailMessage message ) {
		System.out.println();
		System.out.println( "Receive " + new Date() );

		String errMsg ;
		boolean okFlag = false;

		// 毎回 オブジェクトを構築します。登録日付が初期化されます。
		ge50 = new GE50Access( "CXXXXX","M_RECEIVE","ExcelInsert" );

		try {
			String from		= message.getHeader( "From" );
			String subject	= message.getSubject();
			String content	= message.getContent();
			String msgid	= message.getMessageID();
			String comments = from + " " + subject + " " + msgid ;

			String systemId = getContentParam( content,"SYSTEM_ID" );
			if( systemId == null ) {
				errMsg = "メール本文に SYSTEM_ID=[xx] のキーワードが存在しません。" ;
				errorReport( message,errMsg ) ;
				return okFlag;
			}

			System.out.println( " From:" + from );
			String joken = getContentParam( content,"JOKEN" );
			if( joken == null ) { joken = "EXCELIN"; }

			ge50.setSystemId( systemId );
			ge50.setJoken   ( joken );
			ge50.setComments( comments );

			MailAttachFiles attFiles = new MailAttachFiles( message.getMessage() );
			String[] files = attFiles.getNames();
			if( files == null || files.length == 0 ) {
				errMsg = "メールに EXCEL 添付ファイルが存在しません。" ;
				errorReport( message,errMsg ) ;
				return okFlag;
			}

			// 添付ファイルの数だけ処理します。
			for( int i=0; i<files.length; i++ ) {
				String ykno    = ge50.makeYkno();		// 新たな要求番号が内部にもセットされる。
				String file    = ykno + ".xls" ;
				String attFile = files[i];

				// LISTID を取得します。 EXCELファイル名に(帳票ID)を入れます。
				int st = attFile.indexOf( '(' );
				int ed = attFile.indexOf( ')',st );
				if( st < 0 || ed < 0 ) {
					errMsg = "EXCEL 添付ファイルに(帳票ID)が存在しません。[" + attFile + "]" ;
					errorReport( message,errMsg ) ;
					return okFlag;
				}

				String listId  = attFile.substring( st+1,ed );
				if( listId.length() == 0 ) {
					errMsg = "EXCEL 添付ファイルの(帳票ID)を切り出せません。[" + attFile + "]" ;
					errorReport( message,errMsg ) ;
					return okFlag;
				}

				String fileDir = EXCELIN_URL + systemId + "/" + listId + "/" ;

				attFiles.saveFileName( fileDir,file,i );

				ge50.setListId( listId );
				ge50.setOutDir( fileDir );
				ge50.setOutFile( files[i] );

				ge50.insertGE50( GE50Access.FG_SET );

				System.out.println( attFile + " -> " + file );
			}
			System.out.println( " End." );
			okFlag = true;
		}
		catch( Throwable ex ) {
			errMsg = StringUtil.stringStackTrace( ex );
			errorReport( message,errMsg ) ;
			okFlag = false;
		}
		return okFlag;
	}

	/**
	 * メール処理中にエラーが発生した場合の処理を行います。
	 * 処理は、３種類あります。
	 * １．エラーメールを、EXCELIN_URL/ERROR_MAIL フォルダに、メッセージID.txt で保存します。
	 * ２．COMMON_MAIL_SERVER と ERROR_MAIL_TO_USERS が設定されている場合に、
	 *     そのあて先に、返信メールを送信します。
	 * ３．GE50(帳票要求テーブル)と、GE51(帳票エラーテーブル)にエラー状況を書き込みます。
	 *
	 * @param message MailMessageオブジェクト
	 * @param errorMessage エラーメッセージ
	 */
	private void errorReport( final MailMessage message,final String errorMessage ) {
		// エラーメールを 所定のディレクトリに出力します。
		message.saveSimpleMessage( EXCELIN_URL + "/ERROR_MAIL/" );

		String subject		= "EXCEL取込メール受信中にエラーが発生しました。";
		String org_from		= message.getHeader( "From" ) ;
		String ord_subject	= message.getSubject();
		String content		= message.getContent();
		String errMsg = subject + HybsSystem.CR
					+ " Subject=" + ord_subject + " From=" + org_from + HybsSystem.CR
					+ " MessageID=" + message.getMessageID() + HybsSystem.CR
					+ " Save ErrorMail=" + EXCELIN_URL + "/ERROR_MAIL/" + HybsSystem.CR
					+ errorMessage + HybsSystem.CR
					+ "==========================================================="
					+ HybsSystem.CR
					+ content ;

		if( ge50 != null ) {
			String systemId = ge50.getSystemId() ;
			if( systemId == null ) { ge50.setSystemId( "ERR" ); }

			// エラー時の要求番号は、新たに採番しなおします。
			ge50.makeYkno();

			ge50.insertGE50( GE50Access.FG_ERR2 );
			ge50.insertErrorGE56( errMsg );
		}

		// host と user のメール送信先が記述されている場合は、メールでも転送する。
		String   host = message.getHost();
		String   from = message.getUser();

		if( host != null && from != null ) {
			String[] to		= new String[] { org_from };
			String[] cc = StringUtil.csv2Array( HybsSystem.sys( "ERROR_MAIL_TO_USERS" ) );

			MailTX tx = new MailTX( host );
//			tx.setHost( host );
			tx.setFrom( from );
			tx.setTo( to );
			if( cc.length > 0 ) { tx.setCc( cc ); }
			tx.setSubject( subject );
			tx.setMessage( errMsg );
			tx.sendmail();
		}
	}

	/**
	 * メッセージ本文より、指定のキーに関連付けられている情報を切り出します。
	 * これは、指定の文字列=[設定値] というキーワードから、設定値を取り出します。
	 * 指定の文字列と=のと間に、スペースを入れないでください。
	 * 設定値の切り出しは、"指定の文字列=[" と "]" の間の文字列を切り出します。
	 * 内部にスペースが入っていても、問題ありません。ただし、] が入っている場合は、
	 * 正常に切り出すことは出来ませんし、エスケープ文字も用意していません。
	 *
	 * @param	content	コンテンツの文字列
	 * @param	key	情報を切り出す時のキー
	 *
	 * @return	設定値 (設定値が見つからない場合:null
	 */
	private String getContentParam( final String content,final String key ) {
		if( content == null || key == null ) { return null; }

		String newKey = key + "=[" ;

		// キーの存在チェックと場所を求める。
		int keyAd = content.indexOf( newKey );
		if( keyAd >= 0 ) {
			// [設定値] の終了場所を求める。(見つけた位置＋文字列数)
			int st = keyAd + newKey.length() ;
			int ed = content.indexOf( ']' , st );
			if( ed >= 0 ) {
				return content.substring( st,ed );
			}
		}
		return null;
	}
}
