/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

// import java.text.DateFormat;
// import java.text.SimpleDateFormat;
// import java.text.ParseException;
// import java.util.Date;
// import java.util.Calendar;
// import java.util.Locale;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsDateUtil;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 文字列の厳密な日付属性(年/月/日)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * YYYYMMDD に対応している必要があります。
 * これは、YMD に対して、日付の整合性チェックを厳密に行います。
 * valueAdd( String value )による日付の加算時にも、正式な日付で加算されます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・YYYYMMDD 日付フォーマットでの変換
 *
 * @og.group データ属性
 * @og.rev 4.2.0.1 (2008/03/27) 新規作成
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_DATE extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.2.0.1 (2008/03/27)" ;

	/**
	 * NATIVEの型の識別コードを返します。
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	@Override
	public NativeType getNativeType() {
		return NativeType.CALENDAR;
	}

	/**
	 * 半角スペースで固定長(半角換算の数)に変換した文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		if( value != null && value.length() >= sizeX ) {
			return value.substring( 0, sizeX );
		}

		String errMsg = "指定文字が不正です。value=[" + value + "] length=[" + sizeX + "]";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、YYYYMMDD 形式のデータに、＋１ するので、日を＋１します。
	 * 
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、日を加算します。よって、年月日フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.getDatePlus( value , 1 );

// 		DateFormat formatter = new SimpleDateFormat( "yyyyMMdd",Locale.JAPAN );
// 		Date date = null;
// 
// 		try {
// 			date = formatter.parse( value );
// 		}
// 		catch( ParseException ex ) {
// 			// 入力したデータが不正な場合
// 			return value ;
// 		}
// 
// 		Calendar cal = Calendar.getInstance();
// 		cal.setTime( date );
// 		cal.add( Calendar.DATE, 1 );		// 日にちを１日加算します。
// 
// 		return formatter.format( cal.getTime() );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * このメソッドでは、sizeX で指定された
	 * YYYYMMDD または、YYYYMMDDHHMMSS 形式の日付データに
	 * フォーマットを統一します。
	 * 上記フォーマット化において、sizeX で指定した桁数に強制的に合わされます。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param   value 一般に編集データとして登録されたデータ
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.parseDate( value , 8 );		// 不要な記号等を削除し、桁数をそろえる。

// 		String rtn = ( value == null ) ? "" : value.trim();
// 
// 		if( rtn.length() == 6 ) { rtn = "20" + rtn ; }
// 		else if( rtn.length() > 8 ) { rtn = rtn.substring(0,8) ; }
// 
// 		return rtn;
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
//	public ErrorMessage valueCheck( final String key ,final String value ,
//									final int sizeX ,final int sizeY ,final String param ) {
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		// 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
		if( isStrict ) {
			if( len != value.length() ) {
				// 文字列の長さが指定の長さと異なります。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
			}
		}
		else {
			if( len < value.length() ) {
				// 文字列の長さが指定の長さよりも長いです。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,
										String.valueOf( value.length() ),String.valueOf( len ) );
			}
		}

		StringBuilder val = new StringBuilder();
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt( i );
			if( ('0' > ch || '9' < ch) ) {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				val.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 日付の厳密なチェック
		if( ! HybsDateUtil.isStrict( value ) ) {
			// 入力したデータが不正です。key={0} value={1} 形式={2}
			msg.addMessage( 0,ErrorMessage.NG,"ERR0030", key,value,"yyyyMMdd" );
		}

// 		// 日付の厳密なチェック
// 		DateFormat formatter = new SimpleDateFormat( "yyyyMMdd",Locale.JAPAN );
// 		formatter.setLenient( false );		// 日付/時刻解析を厳密に行う(false=厳密)
// 
// 		try {
// 			formatter.parse( value );
// 		}
// 		catch( ParseException ex ) {
// 			// 入力したデータが不正です。key={0} value={1} 形式={2}
// 			msg.addMessage( 0,ErrorMessage.NG,"ERR0030", key,value,"yyyyMMdd" );
// 		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		String check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
