/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

// import java.text.DateFormat;
// import java.text.SimpleDateFormat;
// import java.util.Locale;
import java.util.Calendar;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 文字列の日付属性(年/月/日)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * yyyyMMdd に対応している必要があります。
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )による
 * 日付の加算時には正式な日付で加算されます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YMD extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * NATIVEの型の識別コードを返します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	@Override
	public NativeType getNativeType() {
		return NativeType.CALENDAR;
	}

	/**
	 * 半角スペースで固定長(半角換算の数)に変換した文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		if( value != null && value.length() >= sizeX ) {
			return value.substring( 0, sizeX );
		}

		String errMsg = "指定文字が不正です。value=[" + value + "] length=[" + sizeX + "]";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、yyyyMMdd 形式のデータに、＋１ するので、日を＋１します。
	 * 
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、日を加算します。よって、年月日フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 日の加算ロジックの誤り修正。(roll ⇒ add)
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.getDatePlus( value , 1 );

// 		Calendar cal = Calendar.getInstance();
// 
// 		int year  = Integer.parseInt( value.substring( 0,4 ) );
// 		int month = Integer.parseInt( value.substring( 4,6 ) ) - 1;
// 		int date  = Integer.parseInt( value.substring( 6,8 ) );
// 
// 		cal.set( year,month,date );
// 		cal.add( Calendar.DATE, 1 );		// 日にちを１日加算します。
// 
// 		DateFormat formatter = new SimpleDateFormat( "yyyyMMdd",Locale.JAPAN );
// 		return formatter.format( cal.getTime() );
	}

	/**
	 * String引数の文字列に、第２引数に指定の文字列(数字、日付等)を加算して返します。
	 *
	 * ここでは、yyyyMMdd 形式のデータに、日付を加算します。
	 *
	 * この YMD は、引数に、日付単位を指定できます。単位は、yyyyMMdd 形式の
	 * １文字を指定します。大文字、小文字も識別します。value="5M" とすれば、５か月、value="5d"
	 * とすれば、５日 追加します。
	 * 指定しない場合は、日を加算します。
	 * 
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、日を加算します。よって、年月日フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 * @og.rev 5.6.1.0 (2013/02/01) 加算する引数に、日付単位('y','M','d')を指定可能にします。
	 *
	 * @param   value  String引数
	 * @param   add    加算する日付文字列(単位として、y,M,d を指定できます)
	 *
	 * @return  引数の文字列に、日付を加算します
	 */
	@Override
	public String valueAdd( final String value,final String add ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.getDatePlus( value,add,Calendar.DATE,"yyyyMMdd" );

//		int addSu = 1;	// 初期値(add が null や Empty の場合は、＋１となる)
//		if( add != null && !add.isEmpty() ) {
//			addSu = Integer.parseInt( add );
//		}

//		return HybsDateUtil.getDatePlus( value , addSu );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * このメソッドでは、sizeX と無関係に ８ケタの yyyyMMdd 形式の
	 * 日付データにフォーマットに統一します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.parseDate( value , 8 );		// 不要な記号等を削除し、桁数をそろえる。

// 		String rtn = ( value == null ) ? "" : value.trim();
// 
// 		if( rtn.length() == 6 ) { rtn = "20" + rtn ; }
// 		else if( rtn.length() > 8 ) { rtn = rtn.substring(0,8) ; }
// 		else if( rtn.startsWith( "0000" ) ) { rtn = "00000101"; }
// 		else if( rtn.startsWith( "9999" ) ) { rtn = "99991231"; }
// 
// 		return rtn;
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 * @og.rev 5.6.0.3 (2012/01/24) 日付チェックの整合性まで含めたチェックを入れる。
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
//	public ErrorMessage valueCheck( final String key ,final String value ,
//									final int sizeX ,final int sizeY ,final String param ) {
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		// 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
		if( isStrict ) {
			if( len != value.length() ) {
				// 文字列の長さが指定の長さと異なります。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, 
										String.valueOf( value.length() ), String.valueOf( len ) );
			}
		}
		else {
			if( len < value.length() ) {
				// 文字列の長さが指定の長さよりも長いです。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,
										String.valueOf( value.length() ),String.valueOf( len ) );
			}
		}

//		StringBuilder val = new StringBuilder();
//		boolean isError = false;
//		for( int i=0; i<value.length(); i++ ) {
//			char ch = value.charAt( i );
//			if( ('0' > ch || '9' < ch) ) {
//				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
//				isError = true;
//			}
//			else {
//				val.append( ch );
//			}
//		}
//		if( isError ) {
//			// 指定の文字以外の文字が使われています。
//			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
//		}

		// 5.6.0.3 (2012/01/24) 文字の範囲チェック
		String check = DBTypeCheckUtil.rangeCheck( value, '0', '9' );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 5.6.0.3 (2012/01/24) 日付文字の整合性チェック
		// "0000","9999" で始まる日付は未条件許可。
		check = DBTypeCheckUtil.ymdFormatCheck( value );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
