/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import org.opengion.fukurou.util.LogWriter;

import java.util.Set;
import java.util.HashSet;
import java.io.IOException;
import java.lang.reflect.Field;

import com.sun.javadoc.RootDoc;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.FieldDoc;
import com.sun.javadoc.Doc;
import com.sun.javadoc.ConstructorDoc;
import com.sun.javadoc.ExecutableMemberDoc;
import com.sun.javadoc.Type;
import com.sun.javadoc.Parameter;
import com.sun.javadoc.Tag;
import com.sun.javadoc.SourcePosition;
import com.sun.javadoc.AnnotationDesc;

/**
 * ソースコメントから、タグ情報を取り出す Doclet クラスです。
 * クラスファイルの仕様を表現する為、og.formSample , og.rev , og.group ,
 * version , author , since の各タグコメントより値を抽出します。
 * また、各クラスの継承関係、インターフェース、メソッドなども抽出します。
 * これらの抽出結果をDB化し、EXCELファイルに帳票出力する事で、クラスファイルの
 * ソースから仕様書を逆作成します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DocletSpecific {
	private static final String  SELECT_PACKAGE	= "org.opengion" ;
	private static final boolean NOT_PRIVATE	= false ;
	private static final String  ENCODE			= "UTF-8";

	private static final String	OG_FOR_SMPL		= "og.formSample";
	private static final String	OG_REV			= "og.rev";
	private static final String	OG_GROUP		= "og.group";
	private static final String	DOC_VERSION		= "version";
	private static final String	DOC_AUTHOR		= "author";
	private static final String	DOC_SINCE		= "since";

	private static final String	DOC_PARAM		= "param";		// 5.1.9.0 (2010/08/01) チェック用
	private static final String	DOC_RETURN		= "return";		// 5.1.9.0 (2010/08/01) チェック用

	private static final String	CONSTRUCTOR		= "コンストラクタ" ;
	private static final String	METHOD			= "メソッド" ;
	private static final Set<String> methodSet	= new HashSet<String>();

	private static       int	debugLevel		= 0;	// 0:なし  1:最小チェック  2:日本語化   3:体裁

	/**
	 * すべてが staticメソッドなので、コンストラクタを呼び出さなくしておきます。
	 *
	 */
	private DocletSpecific() {}

	/**
	 * Doclet のエントリポイントメソッドです。
	 *
	 * @og.rev 5.5.4.1 (2012/07/06) Tag出力時の CR → BR 変換を行わない様にする。
	 *
	 * @param	root	エントリポイントのRootDocオブジェクト
	 *
	 * @return 正常実行時 true
	 */
	public static boolean start( final RootDoc root ) {
		String version	= DocletUtil.getOption( "-version" , root.options() );
		String file		= DocletUtil.getOption( "-outfile" , root.options() );
		String dbgLvl	= DocletUtil.getOption( "-debugLevel" , root.options() );		// 5.5.4.1 (2012/07/06) パラメータ引数
		if( dbgLvl != null ) { debugLevel = Integer.parseInt( dbgLvl ); }

		DocletTagWriter writer = null;
		try {
//			writer = new DocletTagWriter( file,ENCODE,true );
			writer = new DocletTagWriter( file,ENCODE );		// 5.5.4.1 (2012/07/06)

			writer.printTag( "<?xml version=\"1.0\" encoding=\"", ENCODE, "\" ?>" );
			writer.printTag( "<javadoc>" );
			writer.printTag( "  <version>",version,"</version>" );
			writer.printTag( "  <description></description>" );
			writeContents( root.classes(),writer );
			writer.printTag( "</javadoc>" );
		}
		catch( IOException ex ) {
			LogWriter.log( ex );
		}
		finally {
			if( writer != null ) { writer.close(); }
		}
		return true;
	}

	/**
	 * ClassDoc 配列よりコンテンツを作成します。
	 *
	 * @og.rev 5.5.4.1 (2012/07/06) コメントは文字列でなく、Tag配列として処理させる。
	 * @og.rev 5.6.6.0 (2013/07/05) VERSION staticフィールドと、@og.rev コメントの比較チェック
	 *
	 * @param classes	ClassDoc配列
	 * @param writer	Tagを書き出すWriterオブジェクト
	 */
	private static void writeContents( final ClassDoc[] classes,final DocletTagWriter writer ) {
		for(int i=0; i< classes.length; i++) {
			ClassDoc classDoc	= classes[i] ;
			String className	= classDoc.name();
			String fullName		= classDoc.qualifiedName() ;
			String modifiers	= (classDoc.modifiers()
								+ ( classDoc.isClass() ? " class" : "" ) ).trim();

			Type superType = classDoc.superclassType();
			String superClass = ( superType == null ) ? "" : superType.qualifiedTypeName();

			Type[] interfaceTypes = classDoc.interfaceTypes();
			StringBuilder buf = new StringBuilder( 200 );
			for( int j=0; j<interfaceTypes.length; j++ ) {
				buf.append( interfaceTypes[j].qualifiedTypeName() ).append( "," );
			}
			if( interfaceTypes.length > 0 ) { buf.deleteCharAt( buf.length()-1 ); }
			String intFase = buf.toString();

			Tag[] desc = classDoc.firstSentenceTags();
//			String cmnt = DocletUtil.htmlFilter( classDoc.commentText() );			// 5.5.4.1 (2012/07/06)
			Tag[] cmnt = classDoc.inlineTags();									// 5.5.4.1 (2012/07/06)
			Tag[] smplTags	= classDoc.tags(OG_FOR_SMPL);
			Tag[] revTags	= classDoc.tags(OG_REV);
			Tag[] createVer	= classDoc.tags(DOC_VERSION);
			Tag[] author	= classDoc.tags(DOC_AUTHOR);
			Tag[] since		= classDoc.tags(DOC_SINCE);
			Tag[] grpTags	= classDoc.tags(OG_GROUP);

			writer.printTag( "<classDoc>" );
			writer.printTag( "  <fullName>"		,fullName		,"</fullName>"		);
			writer.printTag( "  <modifiers>"	,modifiers		,"</modifiers>"		);
			writer.printTag( "  <className>"	,className		,"</className>"		);
			writer.printTag( "  <superClass>"	,superClass		,"</superClass>"	);
			writer.printTag( "  <interface>"	,intFase		,"</interface>"		);
			writer.printTag( "  <createVer>"	,createVer		,"</createVer>"		);
			writer.printTag( "  <author>"		,author			,"</author>"		);
			writer.printTag( "  <since>"		,since			,"</since>"			);
			writer.printTag( "  <description>"	,desc			,"</description>"	);
			writer.printTag( "  <contents>"		,cmnt			,"</contents>"		);
			writer.printTag( "  <classGroup>"	);
			writer.printCSVTag(		grpTags		);
			writer.printTag(   "</classGroup>"	);
			writer.printTag( "  <formSample>"	,smplTags		,"</formSample>"	);
			writer.printTag( "  <history>"		,revTags		,"</history>"		);

			// 5.1.9.0 (2010/08/01) ソースチェック用(コメントや概要が無い場合。スーパークラスは省く)
//			if( ( cmnt.length() == 0 || desc.length == 0 ) && superClass.length() == 0 ) {
			if( debugLevel >= 2 && ( cmnt.length == 0 || desc.length == 0 ) && superClass.length() == 0 ) {
				System.err.println( "警告2:コメントC=\t" + classDoc.position() );
			}

//			methodSet.clear();		// 5.5.4.1 (2012/07/06) メソッドの重複処理判定は、クラス名も含めて行うので、clear() しない。
			int extendFlag = 0;		// 0:オリジナル 1:org.opengion関連Extend 2:Java関連Extend
	//		while( fullName.startsWith( SELECT_PACKAGE ) ) {

			// 5.6.6.0 (2013/07/05) VERSION staticフィールドと、@og.rev コメントの比較チェック
			// while 以下で、fullName と classDoc を順番に上にさかのぼっているので、先にチェックします。
			checkTag2( fullName,classDoc );

			while( true ) {
//				ConstructorDoc[] cnstrctrs = classDoc.constructors();
				ConstructorDoc[] cnstrctrs = classDoc.constructors( false );	// 5.1.9.0 (2010/08/01) チェック用
				for(int j=0; j < cnstrctrs.length; j++) {
					if( isAction( cnstrctrs[j],extendFlag ) ) {
						if( extendFlag < 2 ) { checkTag( cnstrctrs[j] ); }		// 5.5.4.1 (2012/07/06)  チェックを分離
						menberTag( cnstrctrs[j],CONSTRUCTOR,writer,extendFlag );
					}
				}

//				MethodDoc[] methods = classDoc.methods();
				MethodDoc[] methods = classDoc.methods( false );	// 5.1.9.0 (2010/08/01) チェック用
				for(int j=0; j < methods.length; j++) {
					if( isAction( methods[j],extendFlag ) ) {
						if( extendFlag < 2 ) { checkTag( methods[j] ); }		// 5.5.4.1 (2012/07/06)  チェックを分離
						menberTag( methods[j],METHOD,writer,extendFlag );
					}
				}

				// 対象クラス(オリジナル)から、上に上がっていく。
				Type type = classDoc.superclassType();
				if( type == null ) { break; }
				classDoc  = type.asClassDoc() ;
				fullName = classDoc.qualifiedName();
				// java.lang.Object クラスは対象が多いため、処理しません。
				if( "java.lang.Object".equals( fullName ) || classDoc.isEnum() ) {
					break;
				}
				else if( fullName.startsWith( SELECT_PACKAGE ) ) {
					extendFlag = 1;
				}
				else {
					extendFlag = 2;
				}
			}

			writer.printTag( "  </classDoc>" );
		}
	}

	/**
	 * メンバークラスのXML化を行うかどうか[true/false]を判定します。
	 *
	 * 以下の条件に合致する場合は、処理を行いません。(false を返します。)
	 *
	 * １．同一クラスを処理中にEXTENDで継承元をさかのぼる場合、すでに同じシグネチャのメソッドが
	 *     存在している。
	 * ２．NOT_PRIVATE が true の時の private メソッド
	 * ３．extendFlag が 0以上(1,2)の時の private メソッド
	 * ４．メソッド名におかしな記号(&lt;など)が含まれている場合
	 *
	 * @og.rev  5.5.4.1 (2012/07/06) メソッドの重複処理判定は、クラス名も含めて行う
	 *
	 * @param	menber ExecutableMemberDocオブジェクト
	 * @param	extendFlag	0:オリジナル 1:org.opengion関連Extend 2:Java関連Extend
	 *
	 * @return	XML化を行うかどうか[true/false]
	 */
	private static boolean isAction( final ExecutableMemberDoc menber,final int extendFlag ) {
		String menberName = menber.name() ;
//		String signature  = menberName + menber.signature();
//		boolean rtn = 	( ! methodSet.add( signature ) )
		boolean rtn = 	( ! methodSet.add( menber.toString() ) )	// 5.5.4.1 (2012/07/06) メソッドの重複処理判定は、クラス名も含めて行う
					||	( NOT_PRIVATE    && menber.isPrivate() )
					||	( extendFlag > 0 && menber.isPrivate() )
					||	( menberName.charAt(0) == '<' ) ;

		return ! rtn ;
	}

	// 5.1.9.0 (2010/08/01) ソースチェック用(半角文字＋空白文字のみ)
	private static java.util.regex.Pattern PTN = java.util.regex.Pattern.compile("[\\w\\s]+");

	/**
	 * param,return 等の整合性をチェックします。
	 *
	 * @og.rev 5.5.4.1 (2012/07/06) 新規作成。
	 *
	 * @param menber ExecutableMemberDocオブジェクト
	 */
	private static void checkTag( final ExecutableMemberDoc menber ) {

		// 親が Enum クラスの場合、処理しません。
		Type prntType = menber.containingClass().superclassType();
		String prntClass = ( prntType == null ) ? "" : prntType.qualifiedTypeName();
		if( "java.lang.Enum".equals( prntClass ) ) { return; }

		SourcePosition posi = menber.position();
		String modifiers = null;

		if( menber instanceof MethodDoc ) {
			// メソッドの処理(コンストラクターを省く)
			Type	rtnType	= ((MethodDoc)menber).returnType();
			String	typNm	= rtnType.typeName();

			StringBuilder modifyBuf = new StringBuilder( 200 );
			modifyBuf.append( menber.modifiers() ).append( " " ).append( typNm );
			if( rtnType.dimension() != null ) { modifyBuf.append( rtnType.dimension() ); }
			modifiers = modifyBuf.toString();

			String wormMsg = "=\t" + posi + "\t" + modifiers ;

			// 5.1.9.0 (2010/08/01) ソースチェック用(@return との整合性チェック)
			Tag[] docReturn	= menber.tags(DOC_RETURN);		// 5.1.9.0 (2010/08/01) チェック用
			if( docReturn.length > 0 ) {
				String data = (docReturn[0].text()).trim();			// 5.5.4.1 (2012/07/06) trim でスペース等の削除
				wormMsg = wormMsg + "\t" + data ;

				// 5.5.4.1 (2012/07/06) ソースチェック用(@return と引数の個数が異なる場合)
				if( debugLevel >= 1 && "void".equals( typNm ) ) {
					System.err.println( "警告1:RTNコメント不要" + wormMsg );
				}
				// ｢@return 解説｣ の形式で、解説に日本語がなければ、警告
//				if( debugLevel >= 2 && PTN.matcher( data ).matches() && data.indexOf( ' ' ) < 0 && data.indexOf( '\t' ) < 0 ) {
				if( debugLevel >= 2 && PTN.matcher( data ).matches() ) {
					System.err.println( "警告2:RTN未解説" + wormMsg );
				}
				// ｢@return	String｣ の形式の場合は警告
				if( debugLevel >= 2 && data.equals( typNm ) ) {
					System.err.println( "警告2:RTN一致" + wormMsg );
				}
				// ｢@return	String[]｣ など、配列や、<String>などが含まれる場合は警告
				if( debugLevel >= 2 && ( data.indexOf( "[]" ) >= 0 || data.indexOf( '<' ) >= 0 ) ) {
					System.err.println( "警告2:RTN配列" + wormMsg );
				}
				// ｢@return	String 解説｣ の場合は警告(後ろにスペースか、タブがある場合)
				if( debugLevel >= 3 && (data.indexOf( typNm + " " ) >= 0 || data.indexOf( typNm + "\t" ) >= 0 ) ) {
					System.err.println( "警告3:RTNタイプ" + wormMsg );
				}
				// ｢@return	xxxx 解説｣ の場合で、最初のスペースまでが、すべて英数字のみの場合は警告
				int adrs1 = data.indexOf( ' ' );
				if( debugLevel >= 3 && adrs1 > 0 ) {
					boolean flag = true;
					for( int j=0; j<adrs1; j++ ) {
						char ch = data.charAt( j );
						if( ( ch < '0' || ch > '9' ) && ( ch < 'a' || ch > 'z' ) && ( ch < 'A' || ch > 'Z' )  && ch != '[' && ch != ']' ) {
							flag = false;	// 英数字でない記号が現れた場合
							break;
						}
					}
					if( flag ) {	// すべてが英数字の場合は、
						System.err.println( "警告3:RTN値" + wormMsg );
					}
				}
			}
			else {	// Tag上には、@return 記述が存在しない。
				// 5.5.4.1 (2012/07/06) ソースチェック用(@return と引数の個数が異なる場合)
				if( debugLevel >= 1 && !"void".equals( typNm ) ) {
					System.err.println( "警告1:RTNコメントなし" + wormMsg );
				}
			}

			// オーバーライドチェック：アノテーションの記述漏れ
			// その逆は、コンパイラが警告してくれる。
			MethodDoc mdoc= ((MethodDoc)menber).overriddenMethod();
			if( debugLevel >= 3 && mdoc != null ) {
				AnnotationDesc[] annotations = menber.annotations();
				// 本来は、Override の有無を調べるべきだが、Deprecated と SuppressWarnings の付いている
				// 旧のメソッドに、いちいちOverrideを付けないので、何もなければと条件を緩めます。
				if( annotations.length == 0 ) {
					System.err.println( "警告3:@Overrideなし" + wormMsg );
				}
			}
		}

		Parameter[] prm = menber.parameters();

		// 5.1.9.0 (2010/08/01) ソースチェック用(@param と引数の個数が異なる場合)
		Tag[] docParam	= menber.tags(DOC_PARAM);		// 5.1.9.0 (2010/08/01) チェック用
		if( debugLevel >= 1 && docParam.length != prm.length ) {
			System.err.println( "警告1:PRM個数違い=\t" + posi );
		}

		for( int k=0; k<prm.length; k++ ) {
			String typNm = prm[k].type().typeName();
			String prmNm = prm[k].name();

			// 5.1.9.0 (2010/08/01) ソースチェック用(@param と引数の個数が異なる場合)
			if( docParam.length > k ) {
				String data  = (docParam[k].text()).trim();			// 5.5.4.1 (2012/07/06) trim でスペース等の削除
				String data2 = data.replaceAll( prmNm,"" ).trim();
				String data3 = data2.replaceAll( typNm,"" ).replaceAll( "\\[\\]","" ).trim();
				String wormMsg = "=\t" + posi + "\t" + data ;

				// ｢@param	aaa	解説｣形式で、aaa(引数名)がない場合
				if( debugLevel >= 1 && data.indexOf( prmNm ) < 0 ) {
					System.err.println( "警告1:PRM引数名" + wormMsg );
				}
				// 引数の文字列の長さが、１文字の場合
				if( debugLevel >= 2 && prmNm.length() == 1 ) {
					System.err.println( "警告2:PRM短い" + wormMsg );
				}
				// ｢@param	aaa	解説｣形式で、解説に日本語がない、または、解説がなければ、警告
//				if( debugLevel >= 2 && PTN.matcher( data ).matches() && data.indexOf( ' ' ) < 0 && data.indexOf( '\t' ) < 0 ) {
				if( debugLevel >= 2 && ( PTN.matcher( data2 ).matches() || data3.length() == 0 ) ) {
					System.err.println( "警告2:PRM未解説" + wormMsg );
				}
				// ｢@param	aaa	String[]｣など、配列や、<String>などが含まれる場合は警告
				if( debugLevel >= 2 && ( data.indexOf( "[]" ) >= 0 || data.indexOf( '<' ) >= 0 ) ) {
					System.err.println( "警告2:PRM配列" + wormMsg );
				}
				// ｢@param	aaa	解説｣形式で、String が有って、その後ろにスペースか、タブがあれば警告
				// data2 を使うのは、パラメータ名(xxxMap)にタイプ名(Map)が含まれているケースの対応
				if( debugLevel >= 3 && (data2.indexOf( typNm + " " ) >= 0 || data2.indexOf( typNm + "\t" ) >= 0 ) ) {
					System.err.println( "警告3:PRMタイプ" + wormMsg );
				}
				// ｢@param	aaa	解説｣形式で、解説がない場合
//				if( debugLevel >= 3 && data3.length() == 0 ) {
//					System.err.println( "警告3:PRM解説なし" + wormMsg );
//				}
			}
		}

		Tag[]	desc	= menber.firstSentenceTags();
		Tag[]	cmnt	= menber.inlineTags();									// 5.5.4.1 (2012/07/06)
//		String	extClass = ( extendFlag == 0 ) ? "" : menber.containingClass().qualifiedName() ;

		// 5.1.9.0 (2010/08/01) ソースチェック用
		if( ( cmnt.length == 0 || desc.length == 0 ) 		// コメントや概要が無い
//				&& extClass.length() == 0					// 拡張クラスが存在しない
				&& ( menber instanceof MethodDoc )			// メソッドに限定
				&& !menber.isSynthetic()					// コンパイラによって合成されていない
				&& !menber.isNative()						// ネイティブメソッドでない
				&& debugLevel >= 2 ) {						// debugLevel が 2 以上

			// さらに、親が Enum クラス以外
//			Type prntType = menber.containingClass().superclassType();
//			String prntClass = ( prntType == null ) ? "" : prntType.qualifiedTypeName();
//			if( debugLevel >= 2 && !"java.lang.Enum".equals( prntClass ) ) {
				System.err.println( "警告2:コメントM=" + "\t" + posi + "\t" + menber.name() );
//			}
		}
	}

	/**
	 * VERSION staticフィールドと、@og.rev コメントの比較チェックを行います。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 新規作成
	 *
	 * @param fullName オリジナルのクラス名
	 * @param classDoc ClassDocオブジェクト
	 */
	private static void checkTag2( final String fullName, final ClassDoc classDoc ) {
		String cnstVar = getFieldVERSION( fullName ) ;				// VERSION 文字列    例:5.6.6.0 (2013/07/05)
		String seriUID = getSerialVersionUID( fullName ) ;			// serialVersionUID  例:566020130705L

		// VERSION 文字列 か、serialVersionUID のどちらかがあれば処理します。
		if( cnstVar != null || seriUID != null ) {
			SourcePosition posi = null;

			String maxRev = ( cnstVar != null ) ? cnstVar : "4.0.0.0 (2005/01/31)" ;		// Ver4 の最も古い版
			int    lenVar = maxRev.length();					// 比較時に使用する長さ
			boolean isChange = false;							// max が入れ替わったら、true

			// 本体、コンストラクタ、フィールド、メソッド内から、最大の @og.rev の値を取得します。
			Doc[][] docs = new Doc[4][] ;

			docs[0] = new Doc[] { classDoc } ;
			docs[1] = classDoc.constructors( false ) ;
			docs[2] = classDoc.fields( false ) ;
			docs[3] = classDoc.methods( false ) ;

			for( int i=0; i<docs.length; i++ ) {
				for( int j=0; j < docs[i].length; j++ ) {
					Doc doc = docs[i][j];

					Tag[] revTags = doc.tags(OG_REV);
					for( int k=0 ; k<revTags.length; k++ ) {
						String rev = revTags[k].text();

						if( rev.length() < lenVar ) {
							System.err.println( "警告4:og.revが短い=" + "\t" + rev + "\t" + doc.position() );
							continue;
						}

						rev = rev.substring( 0,lenVar );
						if( maxRev.compareTo( rev ) < 0 ) {			// revTags の og.rev が大きい場合
							maxRev = rev ;
							posi   = doc.position();				// 最後に入れ替わった位置 = 最大のrevの位置
							isChange = true;
						}
					}
				}
			}

			// VERSION 文字列 の定義があり、かつ、max の入れ替えが発生した場合のみ、警告4:VERSIONが古い
			if( cnstVar != null && isChange ) {
				System.err.println( "警告4:VERSIONが古い=" + "\t" + cnstVar + " ⇒ " + maxRev + "\t" + posi );
			}

			// serialVersionUID の定義がある。
			if( seriUID != null ) {
				StringBuilder buf = new StringBuilder();
				// maxRev は、最大の Revか、初期のVERSION文字列 例:5.6.6.0 (2013/07/05)
				for( int i=0; i<maxRev.length(); i++ ) {	// 
					char ch = maxRev.charAt( i );
					if( ch >= '0' && ch <= '9' ) { buf.append( ch ); }	// 数字だけ取り出す。 例:566020130705
				}
				if( !seriUID.equals( buf.toString() ) ) {
					// 以下の処理は、serialVersionUID のソースの位置(posi)を取り出すためだけに使用しています。
					FieldDoc[] fileds = classDoc.fields( false );
					for( int i=0; i<fileds.length; i++ ) {
						FieldDoc filed = fileds[i];
						// private static final long serialVersionUID で宣言されているので。
						if( filed.isPrivate() && filed.isStatic() && filed.isFinal() ) {
							String nm = filed.qualifiedName();
							if( nm.endsWith( "serialVersionUID" ) ) {
					//			String val = filed.constantValueExpression();	// これは、serialVersionUID の設定値の取得サンプル
								posi = filed.position();
							}
						}
					}
					System.err.println( "警告4:serialVersionUIDが古い=" + "\t" + seriUID + " ⇒ " + buf.toString() + "L\t" + posi );
				}
			}
		}
	}

	/**
	 * メンバークラス(コンストラクタ、メソッド)をXML化します。
	 *
	 * @og.rev 5.5.4.1 (2012/07/06) コメントは文字列でなく、Tag配列として処理させる。
	 *
	 * @param	menber		ExecutableMemberDocオブジェクト
	 * @param	menberType	メンバータイプ(コンストラクタ、メソッド)
	 * @param	writer		Tagを書き出すWriterオブジェクト
	 * @param	extendFlag	0:オリジナル 1::org.opengion関連Extend 2:Java関連Extend
	 */
	private static void menberTag(	final ExecutableMemberDoc menber,
									final String menberType,
									final DocletTagWriter writer,
									final int extendFlag ) {

		final String modifiers ;
		if( menber instanceof MethodDoc ) {
			// メソッドの処理
			Type rtnType = ((MethodDoc)menber).returnType();
			StringBuilder modifyBuf = new StringBuilder( 200 );
			modifyBuf.append( menber.modifiers() );
	//		modifyBuf.append( " " ).append( rtnType.qualifiedTypeName() );
			modifyBuf.append( " " ).append( rtnType.typeName() );
			if( rtnType.dimension() != null ) { modifyBuf.append( rtnType.dimension() ); }

			modifiers = modifyBuf.toString();
		}
		else {
			// コンストラクター処理
			modifiers  = menber.modifiers();
		}

		String menberName = menber.name();

		StringBuilder sigBuf = new StringBuilder( 200 );
		sigBuf.append( menberName ).append( "(" ) ;
		Parameter[] prm = menber.parameters();

		for( int k=0; k<prm.length; k++ ) {
//			sigBuf.append( prm[k].toString() ).append( "," );
			Type ptyp = prm[k].type();
			String prmNm =prm[k].name();

			sigBuf.append( ptyp.typeName() ).append( ptyp.dimension() ).append( " " )
					.append( prmNm ).append( "," );
		}

		if( prm.length > 0 ) { sigBuf.deleteCharAt( sigBuf.length()-1 ); }
		sigBuf.append( ")" );
		String signature = sigBuf.toString();

		Tag[]	desc	= menber.firstSentenceTags();
//		String	cmnt	= DocletUtil.htmlFilter( menber.commentText() );		// 5.5.4.1 (2012/07/06)
		Tag[]	cmnt	= menber.inlineTags();									// 5.5.4.1 (2012/07/06)
		Tag[]	tags	= menber.tags();
		Tag[]	revTags = menber.tags(OG_REV);
		String	extend  = String.valueOf( extendFlag );
		String	extClass = ( extendFlag == 0 ) ? "" : menber.containingClass().qualifiedName() ;

		String	position = String.valueOf( menber.position().line() );

		writer.printTag( "  <menber>" );
		writer.printTag( "    <type>"		,menberType	,"</type>"			);
		writer.printTag( "    <name>"		,menberName	,"</name>"			);
		writer.printTag( "    <modifiers>"	,modifiers	,"</modifiers>"		);
		writer.printTag( "    <signature>"	,signature	,"</signature>"		);
		writer.printTag( "    <position>"	,position	,"</position>"		);
		writer.printTag( "    <extendClass>",extClass	,"</extendClass>"	);
		writer.printTag( "    <extendFlag>"	,extend		,"</extendFlag>"	);
		writer.printTag( "    <description>",desc		,"</description>"	);
		writer.printTag( "    <contents>"	,cmnt		,"</contents>"		);
		writer.printTag( "    <tagText>" );
		writer.printTagsInfo(   tags );
		writer.printTag( "    </tagText>" );
		writer.printTag( "    <history>"	,revTags	,"</history>" );
		writer.printTag( "  </menber>");
	}

	/**
	 * 指定のオブジェクトの  VERSION と、serialVersionUID staticフィールドの値を取得します。
	 *
	 * 結果は、文字列配列にして返します。
	 * どちらもなければ、null, どちらかあれば、文字列配列にして、一つ目は、VERSION。２つめは、serialVersionUID を
	 * 文字列にした値を返します。
	 *
	 * このメソッドのオリジナルは、org.opengion.hayabusa.servlet.HybsAdmin の private 内部クラス ClassInfo にあります。
	 * 汎用性がない為、ソースのコピー＆ペーストで持ってきています。（若干、修正もしています）
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 新規作成
	 *
	 * @param	clsName	指定のクラスを表す名称
	 * @return	VERSION文字列( staticフィールドの値 ) と、serialVersionUIDを文字列にした値を含む配列
	 */
	private static String getFieldVERSION( final String clsName ) {
		String rtn ;
		try {
			Class<?> cls = Class.forName( clsName ) ;
			Field fld = cls.getDeclaredField( "VERSION" ) ;
			// privateフィールドの取得には、accessibleフラグを trueにする必要があります。
			fld.setAccessible( true );
			rtn = (String)fld.get( null );
		}
		catch( Throwable ex ) {
			rtn = null;
		}
		return rtn ;
	}

	/**
	 * 指定のオブジェクトの  VERSION と、serialVersionUID staticフィールドの値を取得します。
	 *
	 * 結果は、文字列配列にして返します。
	 * どちらもなければ、null, どちらかあれば、文字列配列にして、一つ目は、VERSION。２つめは、serialVersionUID を
	 * 文字列にした値を返します。
	 *
	 * このメソッドのオリジナルは、org.opengion.hayabusa.servlet.HybsAdmin の private 内部クラス ClassInfo にあります。
	 * 汎用性がない為、ソースのコピー＆ペーストで持ってきています。（若干、修正もしています）
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 新規作成
	 *
	 * @param	clsName	指定のクラスを表す名称
	 * @return	VERSION文字列( staticフィールドの値 ) と、serialVersionUIDを文字列にした値を含む配列
	 */
	private static String getSerialVersionUID( final String clsName ) {
		String rtn ;
		try {
			Class<?> cls = Class.forName( clsName );
			Field fld = cls.getDeclaredField( "serialVersionUID" ) ;
			// privateフィールドの取得には、accessibleフラグを trueにする必要があります。
			fld.setAccessible( true );
			rtn = String.valueOf( (Long)fld.get( null ) );
		}
		catch( Throwable ex ) {
			rtn = null;
		}
		return rtn ;
	}

	/**
	 * カスタムオプションを使用するドックレットの必須メソッド optionLength(String) です。
	 *
	 * ドックレットに認識させる各カスタムオプションに、 optionLength がその
	 * オプションを構成する要素 (トークン) の数を返さなければなりません。
	 * このカスタムオプションでは、 -tag オプションそのものと
	 * その値の 2 つの要素で構成されるので、作成するドックレットの
	 * optionLengthメソッドは、 -tag オプションに対して 2 を返さなくては
	 * なりません。また、認識できないオプションに対しては、0 を返します。
	 *
	 * @param	option	オプション文字列
	 *
	 * @return	要素(トークン) の数
	 */
	public static int optionLength( final String option ) {
		if(option.equalsIgnoreCase("-version")) {
			return 2;
		}
		else if(option.equalsIgnoreCase("-outfile")) {
			return 2;
		}
		else if(option.equalsIgnoreCase("-debugLevel")) {
			return 2;
		}
		return 0;
	}
}
