/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;

/**
 * Queryオブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  Queryオブジェクト の識別ID を元に、QueryFactory.newInstance( String id )
 * メソッドで，Queryオブジェクトを取得します。
 * <del>QueryFactory.close( Query query ) メソッドで，内部的に QueryFactory に
 * オブジェクトを戻す事によって,Queryオブジェクトのプーリングを行なっています。</del>
 *
 * 実装とマッピングの関係から,識別ID は、システムパラメータ で 定義します
 * <del>大前提として、ユーザー共通で使用することを考えており,ユーザー個別にプール
 * する必要があるならば, HttpSession オブジェクトに登録すべきです。</del>
 *
 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ()ではなく、オブジェクトを直接生成します。
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class QueryFactory {
	// 3.1.0.0 (2003/03/20) Hashtable を使用している箇所で、非同期でも構わない箇所を、HashMap に置換え。
	/** newInstance() 時のデフォルトクラス {@value} */
	public static final String DEFAULT  = "JDBC" ;

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private QueryFactory() {
	}

	/**
	 * 標準的な Queryオブジェクト(JDBCQuery)を取得します。
	 * 過去に使用された Queryオブジェクト はプールから取得されます。
	 * ただし、内部変数はすべてクリアされますので、 一旦取り出した
	 * オブジェクトを保持したい場合は，各アプリケーション側で保持して下さい。
	 *
	 * @return  Queryオブジェクト
	 */
	public static Query newInstance() {
		return newInstance( DEFAULT );
	}

	/**
	 * 識別id に応じた Queryオブジェクトを取得します。
	 * 過去に使用された Queryオブジェクト はプールから取得されます。
	 * ただし、内部変数はすべてクリアされますので、 一旦取り出した
	 * オブジェクトを保持したい場合は，各アプリケーション側で保持して下さい。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ廃止。直接生成します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、Query. から、Query_ に変更します。
	 * @og.rev 5.3.7.0 (2011/07/01) ゼロ文字列もDefaultを適用
	 *
	 * @param   id Queryインターフェースを実装したサブクラスの識別id
	 *
	 * @return  Queryオブジェクト
	 */
	public static Query newInstance( final String id ) {
//		String type = ( id == null ) ? DEFAULT : id ;
		String type = ( id == null || id.length() == 0 ) ? DEFAULT : id ;
		return (Query)HybsSystem.newInstance( HybsSystem.sys( "Query_" + type ) );
	}

	/**
	 * Queryオブジェクトをプールに戻します。
	 * newInstance でオブジェクトを取り出す方法によっては、close() する必要をなくす
	 * ことができますが、現状はこのメソッドでオブジェクトをプールに戻してください。
	 * オブジェクトを複数個貸し出していた場合,close() で戻すとすでに同じキーの
	 * 別のオブジェクトが存在するが,その場合は,先のオブジェクトは破棄されます。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) メソッド名がまぎらわしい為、変更します。
	 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ廃止。
	 * @og.rev 4.0.0.0 (2005/01/31) Queryの、close() 処理を呼び出しておきます。
	 *
	 * @param   query Queryオブジェクト
	 */
	public static void close( final Query query ) {
		if( query != null ) { query.close(); }	// 4.0.0 (2005/01/31)
	}

	/**
	 * Queryオブジェクトをプールからすべて削除します。
	 * システム全体を初期化するときや、動作が不安定になったときに行います。
	 * プールの方法自体が,一種のキャッシュ的な使いかたしかしていない為,
	 * 実行中でも、いつでもプールを初期化できます。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ廃止。メソッドも廃止します。
	 */
	public static void clear() {
		// ここでは処理を行いません。
	}
}
