/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.xml.HybsXMLSave ;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.sql.Connection;

import java.io.File;
import java.io.BufferedReader;
import java.util.Map;
import java.util.HashMap;

/**
 * 指定の拡張XDK形式ファイルを直接データベースに登録するデータ入力タグです。
 *
 * このクラスは、オラクル XDKの oracle.xml.sql.dml.OracleXMLSave クラスと
 * ほぼ同様の目的で使用できる org.opengion.fukurou.xml.HybsXMLSave のラッパークラスです。
 * 拡張XDK形式のXMLファイルを読み込み、データベースに INSERT します。
 * 拡張XDK形式の元となる オラクル XDK(Oracle XML Developer's Kit)については、以下の
 * リンクを参照願います。
 * <a href="http://otn.oracle.co.jp/software/tech/xml/xdk/index.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * このタグでは、keys,vals を登録することにより、MLファイルに存在しないカラムを
 * 追加したり、XMLファイルの情報を書き換えることが可能になります。
 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
 * 登録するなどです。
 *
 * 拡張XDK形式とは、ROW 以外に、SQL処理用タグ(EXEC_SQL)を持つ XML ファイルです。
 * また、登録するテーブル(table)を ROWSETタグの属性情報として付与することができます。
 * (大文字小文字に注意)
 * これは、オラクルXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLデータをデータベースに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、イベント毎に実行される為、その配置順は重要です。
 * このタグは、複数記述することも出来ますが、BODY部には、１つのSQL文のみ記述します。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 *   &lt;ROWSET tableName="XX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理(データクリア等)を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;MERGE_SQL&gt;                   このSQL文で UPDATEして、結果が０件ならINSERTを行います。
 *           update GEXX set AA=[AA] , BB=[BB] where CC=[CC]
 *       &lt;/MERGE_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;カラム1&gt;値1&lt;/カラム1&gt;
 *             ･･･
 *           &lt;カラムn&gt;値n&lt;/カラムn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @og.formSample
 * ●形式：&lt;og:directXMLSave filename="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:directXMLSave
 *       dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *       fileURL            【TAG】読み取り元ディレクトリ名を指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】ファイルを作成するときのファイル名をセットします (初期値:FILE_FILENAME[=file.xls])
 *       displayMsg         【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0040[　件登録しました])
 *       keys               【TAG】XMLファイルを読み取った後で指定するキーをCSV形式で複数指定します
 *       vals               【TAG】XMLファイルを読み取った後で指定する値をCSV形式で複数指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:directXMLSave
 *         dbid         = "ORCL"                接続データベースID(初期値:DEFAULT)
 *         fileURL      = "{&#064;USER.ID}"     読み取り元ディレクトリ名
 *         filename     = "{&#064;filename}"    読み取り元ファイル名
 *         displayMsg   = "MSG0040"             登録完了後のメッセージ
 *     /&gt;
 *
 * @og.group ファイル入力
 * @og.rev 4.0.0.0 (2007/03/08) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectXMLSaveTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.2.6 (2012/05/25)" ;

	private static final long serialVersionUID = 552620120525L ;

	private static final String ENCODE = "UTF-8";
	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid		= "DEFAULT";
	private String	dbid		= null;
	private String	fileURL		= HybsSystem.sys( "FILE_URL" );		// 4.0.0 (2005/01/31)
	private String	filename	= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
	private String	displayMsg	= "MSG0040";	// 　件登録しました。
	private String[]	keys	= null;
	private String[]	vals	= null;
	private long	dyStart		= 0;	// 実行時間測定用のDIV要素を出力します。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();		// 時間測定用
		return( SKIP_BODY );	// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。例外経路で null 値を利用することが保証されています。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		BufferedReader reader   = null;
		final int insCnt ;
		final int updCnt ;
		final int delCnt ;
		boolean errFlag = true;
//		Connection  conn = null;
		Transaction tran = null;	// 5.1.9.0 (2010/08/01) Transaction 対応
		try {
			// 5.1.9.0 (2010/08/01) Transaction 対応
			TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
//				tran = new TransactionReal( dbid,getApplicationInfo() );
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}
//			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );	// 3.8.7.0 (2006/12/15)

			Connection conn = tran.getConnection( dbid );		// 5.1.9.0 (2010/08/01) Transaction 対応
			reader = getBufferedReader();
			HybsXMLSave save = new HybsXMLSave( conn );
			if( keys != null ) { save.setAfterMap( getAfterMap() ); }

			save.insertXML( reader );
			insCnt = save.getInsertCount();
			updCnt = save.getUpdateCount();
			delCnt = save.getDeleteCount();
//			Closer.commit( conn );
			tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
			errFlag = false;		// エラーではない
		}
		catch( Throwable ex ) {
//			Closer.rollback( conn );
			if( tran != null ) {		// 5.5.2.6 (2012/05/25) findbugs対応
				tran.rollback();		// 5.1.9.0 (2010/08/01) Transaction 対応
			}
			throw new HybsSystemException( ex );
		}
		finally {
			if( tran != null ) {		// 5.5.2.6 (2012/05/25) findbugs対応
				tran.close( errFlag );
			}
//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
//			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//			Closer.connClose( conn );
			Closer.ioClose( reader );
		}

		// 実行件数の表示
		// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
		if( displayMsg != null && displayMsg.length() > 0 ) {
			StringBuilder buf = new StringBuilder();
			buf.append( "INS:"    ).append( insCnt );
			buf.append( " / UPD:" ).append( updCnt );
			buf.append( " / DEL:" ).append( delCnt );
//			buf.append( getResource().getMessage( displayMsg ) );
			buf.append( getResource().getLabel( displayMsg ) );
			buf.append( HybsSystem.BR );

			jspPrint( buf.toString() );
		}

		// 時間測定用の DIV 要素を出力
		long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( insCnt,dyTime,fileURL + filename ); }

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 */
	@Override
	protected void release2() {
		super.release2();
//		dbid		= "DEFAULT";
		dbid		= null;
		fileURL		= HybsSystem.sys( "FILE_URL" );		// 4.0.0 (2005/01/31)
		filename	= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
		displayMsg	= "MSG0040";	// 　件登録しました。
		keys		= null;
		vals		= null;
	}

	/**
	 * BufferedReader を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した BufferedReader を作成します。
	 *
	 * @return	ファイル入力用BufferedReaderオブジェクト
	 */
	private BufferedReader getBufferedReader() {
		if( filename == null ) {
			String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}
		String directory = HybsSystem.url2dir( fileURL );
		File file = new File( StringUtil.urlAppend( directory,filename ) );

		BufferedReader out = FileUtil.getBufferedReader( file,ENCODE );

		return out ;
	}

	/**
	 * BufferedReader を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した BufferedReader を作成します。
	 *
	 * @return	ファイル入力用BufferedReader
	 */
	private Map<String,String> getAfterMap() {
		Map<String,String> map = new HashMap<String,String>();

		for( int i=0; i<keys.length; i++ ) {
			map.put( keys[i],vals[i] );
		}
		return map ;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】読み取り元ディレクトリ名を指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリより、ファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダの下より、読み取ります。
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします
	 *		(初期値:FILE_FILENAME[={@og.value org.opengion.hayabusa.common.SystemData#FILE_FILENAME}])。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * (初期値:システム定数のFILE_FILENAME[={@og.value org.opengion.hayabusa.common.SystemData#FILE_FILENAME}])。
	 *
	 * @param   filename ファイル名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_FILENAME
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0040[　件登録しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 * ※ この属性には、リクエスト変数({&#064;XXXX})は使用できません。
	 *
	 * @param   id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		if( id != null ) { displayMsg = id; }
	}

	/**
	 * 【TAG】XMLファイルを読み取った後で指定するキーをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLファイルを読み取った後で、データを変更できます。
	 * 変更するカラム名(キー)をCSV形式で指定します。
	 * XMLファイルにキーが存在していた場合は、vals で指定の値に書き換えます。
	 * キーが存在していない場合は、ここで指定するキーと値が、データとして
	 * 追加されます。
	 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key リンク先に渡すキー
	 * @see		#setVals( String )
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】XMLファイルを読み取った後で指定する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLファイルを読み取った後で、データを変更できます。
	 * 変更する値をCSV形式で指定します。
	 * XMLファイルにキーが存在していた場合は、vals で指定の値に書き換えます。
	 * キーが存在していない場合は、ここで指定するキーと値が、データとして
	 * 追加されます。
	 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val keys属性に対応する値
	 * @see		#setKeys( String )
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "fileURL"			,fileURL		)
				.println( "filename"		,filename		)
				.println( "displayMsg"		,displayMsg		)
				.println( "dyStart"			,dyStart		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
