/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.report2.QueueManager_DIRECT;

/**
 * 検索結果の DBTableModelオブジェクトをレポート形式に変換するタグです。
 *
 * データ(DBTableModel)と、コントローラ(DBTableReport クラス)を与えて、
 * 外部からコントロールすることで、各種形式で データ(DBTableModel)を表示させることが
 * 可能です。
 * このタグを使用するには、OpenOffice.orgのモジュールがインストールされてている必要があります。
 * また、出力するために帳票システム関連のデータ設定やマスタ設定は一切必要ありません。
 *
 * @og.formSample
 * ●形式：&lt;og:report fileURL="[･･･]" listId="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:report2
 *       fileURL            【TAG】雛型のHTMLファイルの保存してある ディレクトリを指定します
 *       listId           ○【TAG】帳票IDを指定します。(必須)
 *       outFileURL         【TAG】出力HTMLファイルの保存してあるディレクトリを指定します
 *       outFilename      ○【TAG】ファイルを作成するときの出力ファイル名をセットします。(必須)
 *       headerKeys         【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します
 *       headerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       footerKeys         【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します
 *       footerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       pageEndCut         【TAG】ボディー部(繰り返し部)がなくなったときに、それ以降を表示するかどうか[true/false]を指定します(初期値:true)
 *       useLocalResource   【TAG】各システムのリソース(ローカルリソース)を使用するか[true/false]を指定します(初期値:true)
 *       fgrun              【TAG】出力方法を指定します(初期値:P(PDF出力))
 *       printerName        【TAG】プリンター名を指定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ReportTableTag2 extends CommonTagSupport {
	private static final String VERSION = "4.3.3.4 (2008/11/01)" ;

	private static final long serialVersionUID = 433420081101L ;

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String		fileURL			= BASE_URL;	// 雛形ファイルURL
	private String		listId			= null;		// 帳票ID
	private String		outFileURL		= BASE_URL;	// 出力ファイルURL
	private String		outFilename		= null;		// 出力ファイル名
	private String[]	headerKeys		= null;		// 固定部の{@KEY} の KEY 部分を指定する。カンマで複数指定できる。
	private String[]	headerVals		= null;		// 固定部のKEY に対応する値を指定する。 {@KEY} に置き換わる。
	private String[]	footerKeys		= null;		// 繰り返し部の終了後に表示する key 部分を指定する。カンマで複数指定できる。
	private String[]	footerVals		= null;		// 繰り返し部の終了後に表示する key に対する値を指定する。
	private boolean		pageEndCut		= true;		// ページエンドカットをするか

	private boolean		useLocalResource= true;		// ローカルリソースを使用するか
	private String		fgrun			= "P";		// PDF出力
	private String		printerName		= null;		// プリンタ名

	private String		tableId			= HybsSystem.TBL_MDL_KEY ;

	private transient DBTableModel body		= null;
	private transient DBTableModel header	= null;
	private transient DBTableModel footer	= null;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		final int rtnCode;

		body = (DBTableModel)getObject( tableId );
		if( body == null || body.getRowCount() == 0 ) {
			rtnCode = SKIP_PAGE ; // ページの残りの処理を行わない。
		}
		else {
			exec();
			rtnCode = EVAL_PAGE ;
		}

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL			= BASE_URL;
		listId			= null;
		outFileURL		= BASE_URL;
		outFilename		= null;
		headerKeys		= null;
		headerVals		= null;
		footerKeys		= null;
		footerVals		= null;
		pageEndCut		= true;
		useLocalResource= true;
		fgrun			= "P";
		printerName		= null;
		tableId			= HybsSystem.TBL_MDL_KEY ;
		body			= null;
		header			= null;
		footer			= null;
	}

	/**
	 * 帳票処理を行います。
	 *
	 * @og.rev 4.3.3.4 (2008/11/01) ヘッダー、フッター値が設定されていない場合にNullPointerExceptionが出るバグを修正
	 * @og.rev 4.3.3.4 (2008/11/01) 雛形のパス及び、出力先のパスを実ディレクトリのパスに変換
	 *
	 */
	private void exec()  {
		QueueManager_DIRECT manager = new QueueManager_DIRECT();
		manager.setListId( listId );
		manager.setLang( getLanguage() );
		manager.setOutputName( HybsSystem.url2dir( outFileURL ) + outFilename );
		manager.setOutputType( fgrun );
		manager.setTemplateName( HybsSystem.url2dir( fileURL ) + listId );
		manager.setPrinterName( printerName );
		manager.setFglocal( useLocalResource );
		manager.setFgcut( pageEndCut );

		manager.setBody( body );

		// 4.3.3.4 (2008/11/01)
		if( headerVals != null && headerVals.length > 0 ) {
			String[][] hvals = new String[headerVals.length][1];
			hvals[0] = headerVals;
			header = DBTableModelUtil.makeDBTable( headerKeys, hvals, getResource() );
			manager.setHeader( header );
		}

		// 4.3.3.4 (2008/11/01)
		if( footerVals != null && footerVals.length > 0 ) {
			String[][] fvals = new String[footerVals.length][1];
			fvals[0] = footerVals;
			footer = DBTableModelUtil.makeDBTable( footerKeys, fvals, getResource() );
			manager.setFooter( footer );
		}

		manager.create();
		manager.waitExec();
	}

	/**
	 * 【TAG】雛型のHTMLファイルの保存してある ディレクトリを指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこを操作します。
	 *
	 * @param	url 雛型のHTMLファイルのディレクトリ
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】帳票IDを指定します。
	 *
	 * @og.tag
	 * 帳票IDを指定します。
	 *
	 * @param	listId	帳票ID
	 */
	public void setListId( final String listId ) {
		this.listId = nval( getRequestParameter( listId ), this.listId );
	}

	/**
	 * 【TAG】出力HTMLファイルの保存してあるディレクトリを指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリにファイルを出力します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこに出力します。
	 *
	 * @param	url 出力HTMLファイルのディレクトリ
	 */
	public void setOutFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			outFileURL = StringUtil.urlAppend( outFileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときの出力ファイル名をセットします。
	 *
	 * @og.tag
	 * ファイルを作成するときの出力ファイル名をセットします。
	 * 紙に印字する場合などファイルに出力しない場合は不要です。
	 *
	 * @param   filename 出力ファイル名
	 */
	public void setOutFilename( final String filename ) {
		this.outFilename = nval( getRequestParameter( filename ),this.outFilename );
	}

	/**
	 * 【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   hKeys 固定部の key
	 */
	public void setHeaderKeys( final String hKeys ) {
		headerKeys = getCSVParameter( hKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   hVals 固定部の値
	 */
	public void setHeaderVals( final String hVals ) {
		headerVals = getCSVParameter( hVals );
	}

	/**
	 * 【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   ftKeys 繰り返し部の終了後に表示する key
	 */
	public void setFooterKeys( final String ftKeys ) {
		footerKeys = getCSVParameter( ftKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   ftVals 繰り返し部の終了後に表示する値
	 */
	public void setFooterVals( final String ftVals ) {
		footerVals = getCSVParameter( ftVals );
	}

	/**
	 * 【TAG】ボディー部(繰り返し部)がなくなったときに、それ以降を表示するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true では、それ以降を出力しません。
	 * 初期値は "true" (なくなった時点で、出力しない。)です。
	 *
	 * @param   peCut 繰り返し部の終了後に継続処理するかどうか (true:処理しない/false:処理する)
	 */
	public void setPageEndCut( final String peCut ) {
		pageEndCut = nval( getRequestParameter( peCut ),pageEndCut );
	}

	/**
	 * 【TAG】各システムのリソース(ローカルリソース)を使用するか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true の場合、各システムに登録されたリソース情報を使用して帳票データが変換されます。
	 * false の場合は、帳票デーモンが起動しているシステム(通常は'GE')のリソースが適用されます。
	 * 初期値は "true" (ローカルリソースを使用する)です。
	 *
	 * @param   fgl  ローカルリソースを使用するか(true:処理しない/false:処理する)
	 */
	public void setUseLocalResource( final String fgl ) {
		useLocalResource = nval( getRequestParameter( fgl ),useLocalResource );
	}

	/**
	 * 【TAG】出力方法を指定します(初期値:P(PDF出力))。
	 *
	 * @og.tag
	 * 出力方法のコードは、FGRUNのコードリソースと同じものが指定できます。
	 * 初期値は "P" (PDF出力)です。
	 *
	 * @param   flg 出力方法
	 */
	public void setFgrun( final String flg ) {
		fgrun= nval( getRequestParameter( flg ),fgrun );
	}

	/**
	 * 【TAG】プリンター名を指定します。
	 *
	 * @og.tag
	 * プリンター名を指定します。このプリンター名は帳票サーバー上でのプリンタ名です。
	 * ファイル出力等、紙に印刷しない場合は不要です。
	 *
	 * @param   ptnm プリンター名
	 */
	public void setPrinterName( final String ptnm ) {
		printerName = nval( getRequestParameter( ptnm ),printerName );
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，(内部ポインタを書き換えた)DBTableModelオブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param   tableId sessionに登録する時の ID
	 */
	public void setTableId( final String tableId ) {
		this.tableId = nval( getRequestParameter( tableId ), this.tableId );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "report2" ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"			,fileURL			)
				.println( "listId"			,listId				)
				.println( "outFileURL"		,outFileURL			)
				.println( "outFilename"		,outFilename		)
				.println( "headerKeys"		,headerKeys			)
				.println( "headerVals"		,headerVals			)
				.println( "footerKeys"		,footerKeys			)
				.println( "footerVals"		,footerVals			)
				.println( "pageEndCut"		,pageEndCut			)
				.println( "useLocalResource",useLocalResource	)
				.println( "fgrun"			,fgrun				)
				.println( "printerName"		,printerName		)
				.println( "tableId"			,tableId			)
				.println( "BASE_URL"		,BASE_URL			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
