/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_SEQRESET は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブルモデルのシーケンス項目の値を再セットしています。
 * シーケンスの値は、通常10ずつ増加しますが、BREAK_CLMが指定されており、
 * かつ、その項目の値がブレイクした場合は、100増加させます。
 * また、CLEAR_CLMが指定されている場合、そのカラムの値がキーブレイクした場合は、
 * シーケンス値を初期化(10)にします。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       SEQ_CLM   : シーケンス項目
 *       BREAK_CLM : キーブレイク項目      (任意)
 *       CLEAR_CLM : シーケンス初期化項目
 *       INCREMENT : シーケンスの増分      (初期値：10)
 *       START_NO  : シーケンスの初期値    (初期値：0)
 *  }
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="SEQRESET" keys="SEQ_CLM,BREAK_CLM" vals="SEQNO,DAIBNRUI" /&gt;
 *
 *      ② &lt;og:tableFilter classId="SEQRESET" &gt;
 *               {
 *                   SEQ_CLM   : SEQNO ;
 *                   BREAK_CLM : DAIBNRUI ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_SEQRESET extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.0 (2013/07/05)" ;

	// 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	// keysMap は、AbstractTableFilter で、protected static final 定義しておきます。
	static {
		keysMap.put( "SEQ_CLM"		, "シーケンス項目"					);
		keysMap.put( "BREAK_CLM"	, "キーブレイク項目"				);
		keysMap.put( "CLEAR_CLM"	, "シーケンス初期化項目"			);
		keysMap.put( "INCREMENT"	, "シーケンスの増分(初期値:10)"		);
		keysMap.put( "START_NO"		, "シーケンスの初期値(初期値:0)"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 *
	 * @return	処理後のテーブルモデル
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		int seqNo   =  table.getColumnNo( getValue( "SEQ_CLM"   ),false );
		int breakNo =  table.getColumnNo( getValue( "BREAK_CLM" ),false );
		int clearNo =  table.getColumnNo( getValue( "CLEAR_CLM" ),false );
		String increment =  getValue( "INCREMENT" );
		String start =  getValue( "START_NO" );

		if( seqNo < 0 ){ return table; }

		int seq = StringUtil.nval( start, 0 );
		int inc = StringUtil.nval( increment, 10 );
		String[] data = null;
		String preKey = "";
		String cleKey = "";
		int[] rowNo = getParameterRows();					// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		for( int row = 0; row < rowNo.length; row ++ ) {
			data = table.getValues( rowNo[row] );

			if( clearNo >= 0 && cleKey != null && cleKey.length() > 0 && !cleKey.equals( data[clearNo] ) ) {
				seq = StringUtil.nval( start, 0 );
			}

			if( breakNo >= 0 && preKey != null && preKey.length() > 0 && !preKey.equals( data[breakNo] ) ) {
				seq += inc * 10;
			}
			else {
				seq += inc;
			}

			data[seqNo] = String.valueOf( seq );

			if( breakNo >= 0 ) {
				preKey = data[breakNo];
			}

			if( clearNo >= 0 ) {
				cleKey = data[clearNo];
			}
		}
		return table;
	}
}
