/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.HashMap;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.text.ParseException;

/**
 * HybsDateUtil.java は、共通的に使用される Date,Calender関連メソッドを集約した、staticメソッドのみで構成されるクラスです。
 *
 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
 *
 * @og.group ユーティリティ
 *
 * @version  5.5
 * @author	 Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public final class HybsDateUtil {

	/** システム依存の改行記号をセットします。	*/
	private static final String CR = System.getProperty("line.separator");

	/** 各種フォーマットを簡易的に表した文字列 */
	private static final Map<String,String>	DATE_FORMAT = new HashMap<String,String>();
	static {
		DATE_FORMAT.put( "YMD"		,"yyyyMMdd"				);
		DATE_FORMAT.put( "Y2MD"		,"yyMMdd"				);
		DATE_FORMAT.put( "YM"		,"yyyyMM"				);
		DATE_FORMAT.put( "MD"		,"MMdd"					);	// 5.5.5.2 (2012/08/18)
		DATE_FORMAT.put( "HMS"		,"HHmmss"				);
		DATE_FORMAT.put( "YMDHMS"	,"yyyyMMddHHmmss"		);
		DATE_FORMAT.put( "EEE"		,"EEE"					);
		DATE_FORMAT.put( "YMDF"		,"yyyy/MM/dd"			);
		DATE_FORMAT.put( "Y2MDF"	,"yy/MM/dd" 			);
		DATE_FORMAT.put( "YMF"		,"yyyy/MM"				);
		DATE_FORMAT.put( "HMSF"		,"HH:mm:ss" 			);
		DATE_FORMAT.put( "YMDHMSF"	,"yyyy/MM/dd HH:mm:ss"	);
		DATE_FORMAT.put( "MDF"		,"MM/dd"				); // 5.5.0.2 (2012/03/09) 和暦
		DATE_FORMAT.put( "MDEF"		,"MM/dd(EEE)"			); // 5.5.0.2 (2012/03/09) 和暦
		DATE_FORMAT.put( "MD2F"		,"MM月dd日"				); // 								5.5.5.2 (2012/08/18) 和暦
		DATE_FORMAT.put( "GYMDF"	,"GGGGyyyy年MM月dd日"	); // 5.5.0.2 (2012/03/09) 和暦
		DATE_FORMAT.put( "G2YMDF"	,"Gyyyy/MM/dd"			); // 5.5.0.2 (2012/03/09) 和暦
		DATE_FORMAT.put( "GYMF"		,"GGGGyyyy年MM月"		); // 5.5.0.2 (2012/03/09) 和暦
		DATE_FORMAT.put( "GYF"		,"GGGGyyyy" 			); // 5.5.0.2 (2012/03/09) 和暦
	}

	/**
	 *	デフォルトコンストラクターをprivateにして、
	 *	オブジェクトの生成をさせないようにする。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 */
	private HybsDateUtil() {}

	/**
	 * 現在日付、時刻を指定のフォーマットで文字列に変換して返します。
	 * 出力フォーマットは、"yyyy/MM/dd HH:mm:ss" 固定です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @return	現在日付、時刻 ( 例 2012/09/05 18:10:24 )
	 */
	public static final String getDate() {
		DateFormat formatter = new SimpleDateFormat( "yyyy/MM/dd HH:mm:ss",Locale.JAPAN );
		return formatter.format(new Date());
	}

	/**
	 * 現在時刻を指定のフォーマットで文字列に変換して返します。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 * 変換時のロケーションは、Locale.JAPAN です。
	 * 現在時刻は、new Date() で求めます。
	 *
	 * @param	form フォーム文字列 ( 例 "yyyy/MM/dd HH:mm:ss.SSS" )
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @return	現在日付、時刻
	 * @see		java.text.SimpleDateFormat
	 */
	public static final String getDate( final String form ) {
		DateFormat formatter = new SimpleDateFormat( form,Locale.JAPAN );
		return formatter.format( new Date() );
	}

	/**
	 * 指定時刻を指定のフォーマットで文字列に変換して返します。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 * 変換時のロケーションは、Locale.JAPAN です。
	 * 指定時刻は、new Date( time ) で求めます。
	 *
	 * @param	time 指定のカレントタイムのロング値
	 * @param	form フォーム文字列 ( 例 "yyyy/MM/dd HH:mm:ss.SSS" )
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @return	現在日付、時刻( 例 2001/04/17 15:48:22 )
	 */
	public static final String getDate( final long time,final String form ) {
		DateFormat formatter = new SimpleDateFormat( form,Locale.JAPAN );
		return formatter.format( new Date( time ) );
	}

	/**
	 * 指定の文字列から、以下の文字を削除した文字列を返します。
	 * '/' , '-' , ' ' , ':' の数字以外の文字を含むフォーマットされた
	 * 日付文字列を、日付データだけに変換する場合に利用することを想定しています。
	 * よって、マイナス記号や、小数点、コンマなども削除されます。
	 * このメソッドでは、日付としての整合性や桁チェックは行いません。
	 * 
	 * 引数が、null の場合は、ゼロ文字列に、変換します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 * @og.rev 5.5.8.3 (2012/11/17) 数字のみ返す仕様だったが、対象以外の文字入力はそのまま返すよう変更
	 *
	 * @param	value 任意の文字列（例：2001/04/17 15:48:22）
	 *
	 * @return	数字だけで構成される文字列（例：20010417154822）(nullはゼロ文字列を返します)
	 */
	public static final String parseNumber( final String value ) {
		if( value == null ) { return ""; }

		StringBuilder buf = new StringBuilder();
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt(i);
//			if( ch >= '0' && ch <= '9' ) { buf.append( ch ); }
			if( ch == '/' || ch == '-' || ch == ' ' || ch == ':'){} // 5.5.8.3 (2012/11/17) 何もしない
			else {
				buf.append( ch );
			}
		}

		return buf.toString();
	}

	/**
	 * 指定の文字列から、yyyy-mm-dd hh:mm:ss[.f...] 形式の文字列を作成します。
	 * これは、java.sql.Timestamp オブジェクトを文字列から作成するに当たり、
	 * Timestamp の文字列形式にしなければならないためです。
	 * 桁数は、８桁 または、１４桁以外の場合は、変換エラーとします。
	 *
	 * @og.rev 5.5.8.5 (2012/11/27) 新規作成
	 *
	 * @param	value 任意の文字列（例：20010417 or 20010417154822）
	 *
	 * @return	Timestampの文字列形式（例：2001-04-17 00:00:00 or 2001-04-17 15:48:22）
	 */
	public static final String parseTimestamp( final String value ) {
		if( value == null || ( value.length() != 8 && value.length() != 14 ) ) {
			String errMsg = "日付文字列に、不正な値が指定されました。８桁 または、１４桁で指定してください。"
						+ " value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}

		StringBuilder buf = new StringBuilder();
		buf.append( value.substring( 0,4 ) ).append( "-" );
		buf.append( value.substring( 4,6 ) ).append( "-" );
		buf.append( value.substring( 6,8 ) ).append( " " );
		if( value.length() == 8 ) {
			buf.append( "00:00:00" );
		}
		else {
			buf.append( value.substring( 8,10  ) ).append( ":" );
			buf.append( value.substring( 10,12 ) ).append( ":" );
			buf.append( value.substring( 12,14 ) );
		}

		return buf.toString();
	}


	/**
	 * 日付文字列の桁数の整合性を取ります。
	 * これは、内部で、parseNumber(String) 処理により、不要なフォーマット記号を削除します。
	 * ここでは、基本的には、６文字（yyyyMM）、８文字（yyyyMMdd）、１４文字（yyyyMMddHHmmss）
	 * の日付文字列を作成することを想定していますが、指定の桁数以外は、エラーになります。
	 * 
	 * 引数が、null         ⇒ 桁数に無関係に、空文字列を返す。
	 * 引数の桁数が一致     ⇒ その値を返す。
	 * 引数の桁数が不一致   ⇒ エラー
	 * ただし、引数の最大長は、１４ケタに制限しています。
	 * 
	 * このメソッドでは、日付として成立しているかどうか（99999999など）は判定していません。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) メソッドの内容を移す。
	 *
	 * @param	value   任意の日付け文字列
	 * @param	size    変換したい桁数
	 *
	 * @return	数字だけで構成される文字列（例：20010417154822）(nullはゼロ文字列を返します)
	 */
	public static final String parseDate( final String value , final int size ) {
		return parseDate( value , size , size );		// 最小と最大を同じ値にする。
	}

	/**
	 * 日付文字列の桁数の整合性を取ります。
	 * これは、内部で、parseNumber(String) 処理により、不要なフォーマット記号を削除します。
	 * ここでは、基本的には、６文字（yyyyMM）、８文字（yyyyMMdd）、１４文字（yyyyMMddHHmmss）
	 * の日付文字列を作成することを想定していますが、それ以外の桁数でも下記のルールに従って
	 * 処理されます。
	 * 
	 * 引数が、null         ⇒ 桁数に無関係に、空文字列を返す。
	 * 引数の桁数が範囲内   ⇒ 以下の処理を実行する。
	 * 引数の桁数を同じ     ⇒ そのまま返す。
	 * 引数の桁数より大きい ⇒ 余をカットして、引数の最大長にそろえる。
	 * 引数の桁数に足りない ⇒ "20000101000000" の文字列の部分文字列を結合させて、引数の最大長にそろえる。
	 * ただし、引数の最大長は、１４ケタに制限しています。
	 * 
	 * このメソッドでは、日付として成立しているかどうか（99999999など）は判定していません。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 * @og.rev 5.6.1.1 (2013/02/08) 桁数チェック導入。６桁以下だとエラーにする。
	 * @og.rev 5.6.6.0 (2013/07/05) 桁数チェックの最小-最大指定
	 *
	 * @param	value   任意の日付け文字列
	 * @param	minSize 変換したい桁数の最小値
	 * @param	maxSize 変換したい桁数の最大値
	 *
	 * @return	数字だけで構成される文字列（例：20010417154822）(nullはゼロ文字列を返します)
	 */
//	public static final String parseDate( final String value , final int size ) {
	public static final String parseDate( final String value , final int minSize , final int maxSize ) {
		if( value == null ) { return ""; }

		String rtn = parseNumber( value );

		// 引数の最大長は、１４ケタに制限しています。
		if( maxSize > 14 ) {
			String errMsg = "日付登録に許可できる最大桁数は、１４ケタです。"
						+ " maxSize=[" + maxSize + "]" ;
			throw new RuntimeException( errMsg );
		}

//		int maxSize = ( size > 14 ) ? 14 : size ;		// 先に最大サイズを求めておく
		int len     = rtn.length() ;

		if(      len == maxSize ) { rtn = value; }
//		else if( len >  maxSize ) { rtn = rtn.substring( 0,maxSize ); }
		// 5.6.1.1 (2013/02/08) 桁数チェック導入。６桁以下だとエラーにする。
//		else if( len < 6 ) {
		// 5.6.6.0 (2013/07/05) 桁数チェックの最小-最大指定で、範囲外はエラー
		else if( len < minSize || len > maxSize ) {
			String errMsg = "日付文字列に、不正な値が指定されました。最小["
						+ minSize + "] から、最大[" + maxSize + "]の範囲で指定してください。"
						+ " value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}
		else {
			rtn = rtn + "20000101000000".substring( len,maxSize ) ;	// 中間文字列を加える。
		}

		return rtn ;
	}

	/**
	 * 日付文字列の厳密な整合性チェックを行います。
	 * ここで指定できるのは、８文字（yyyyMMdd）、１４文字（yyyyMMddHHmmss）のどちらかの
	 * 数字だけの日付文字列であり、それが、日付として正しいかどうかのチェックを行います。
	 * 正しければ、true を、間違っていれば、false を返します。
	 * ここでは、20120230（２月３０日）などの日付や、20120101235960 なども false になります。
	 * 引数が、null および、空文字列の場合も、false を返しますので、避けたい場合は、事前に
	 * 判定しておいてください。
	 *
	 * 内部処理としては、DateFormat で、setLenient( false ) を設定することで、
	 * 日付/時刻解析を厳密に解析するにして、ParseException が発生しないかどうか判定しています。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param	value  数字だけで構成される日付け文字列
	 *
	 * @return	true:日付として正しい場合/false:日付として間違っている場合
	 */
	public static final boolean isStrict( final String value ) {
		if( value == null || ( value.length() != 8 && value.length() != 14 ) ) { return false; }

		// 日付の厳密なチェック
		String form = (value.length() == 8) ? "yyyyMMdd" : "yyyyMMddHHmmss" ;
		DateFormat formatter = new SimpleDateFormat( form,Locale.JAPAN );
		formatter.setLenient( false );		// 日付/時刻解析を厳密に行う(false=厳密)

		boolean flag ;
		try {
			formatter.parse( value );
			flag = true;
		}
		catch( ParseException ex ) {
			flag = false;
		}

		return flag;
	}

	/**
	 * 日付関係の情報を簡易的に処理します。
	 *
	 * 第一引数 "XXXX" は、日付処理を行うフォーマットの予約語になっています。
	 * ・YMD　　：８文字の日付データ(yyyyMMdd)を扱います。
	 * ・Y2MD　 ：６文字の日付データ(yyMMdd)を扱います。
	 * ・YM　　 ：６文字の日付データ(yyyyMM)を扱います。
	 * ・MD　　 ：４文字の日付データ(MMdd)を扱います。(5.5.5.2 追加)
	 * ・HMS　　：６文字の時刻データ(HHmmss)を扱います。
	 * ・YMDHMS ：１４文字の日付データ(yyyyMMddHHmmss)を扱います。
	 * ・EEE　　：曜日をデフォルトロケール(EEE)で表示します。
	 *
	 * F付きは、フォーマットされた日付を返します。
	 * ・YMDF　 ：１０文字の日付表現(yyyy/MM/dd)を扱います。
	 * ・Y2MDF　：８文字の日付表現(yy/MM/dd)を扱います。
	 * ・YMF　　：７文字の日付表現(yyyy/MM)を扱います。
	 * ・HMSF　 ：８文字の時刻表現(HH:mm:ss)を扱います。
	 * ・YMDHMSF：１９文字の日付表現(yyyy/MM/dd HH:mm:ss)を扱います。
	 * ・MDF　　：５文字の月日表現(MM/dd)を扱います。
	 * ・MDEF　 ：５文字＋曜日の月日表現(MM/dd(EEE))を扱います。
	 * ・MD2F　 ：和暦の月日表現(MM月dd日)を扱います。(5.5.5.2 追加)
	 * ・GYMDF　：和暦の年月日表現(GGGGyyyy年MM月dd日)を扱います。
	 * ・G2YMDF ：和暦の日付表現(Gyyyy/MM/dd)を扱います。
	 * ・GYMF　 ：和暦の年月表現(GGGGyyyy年MM月)を扱います。
	 * ・GYF　　：和暦の年表現(GGGGyyyy)を扱います。
	 *
	 * なお、上記以外のフォーマットを指定する場合は、XXXX部分に直接記述できます。(5.5.5.2 追加)
	 * ただし、基本的には、自由フォーマットは、エラーチェックがない為、使わないでください。
	 *
	 * 第二引数 AA は、基準となる日付を、yyyyMMdd形式で指定します。nullの場合は、現在時刻を使用します。
	 * 指定できる日付は、yyyyMMdd形式を推奨しますが、８文字でない場合は、'-' や '/' を削除の上、
	 * 前半８文字のみ使用します。たとえば、"2012/09/05 16:52:36" のようなフォーマットデータが渡された
	 * 場合、"20120905 16:52:36" に変換後、前半８文字の "20120905" が使われます。
	 * よって、フォーマットされている場合でも処理をおこないますが、月、日は、頭ゼロ埋めの２文字で
	 * 渡してください。
	 *
	 * 第三引数 BB は、日付についての加減算処理を行うためのコマンドを指定します。
	 * nullの場合は、なにも加減算処理を行いません。
	 * ・SD ：当月の最初の日付にセットします。(当月１日)
	 * ・ED ：当月の最後の日付にセットします。(当月月末)
	 * ・SW ：日付処理の週初め(月曜日)にセットします。日付は当日より前に移動します。
	 * ・EW ：日付処理の週末(日曜日)にセットします。日付は当日より後ろに移動します。
	 * ・H1 ～ HXXX ：時を指定の分だけ進めます。H1なら１時間後、H24 なら２４時間後(5.5.5.6 (2012/08/31) 追加)
	 * ・D1 ～ DXXX ：日を指定の分だけ進めます。D1なら翌日、D200 なら200日後
	 * ・M1 ～ MXXX ：月を指定の分だけ進めます。M1なら翌月、M6 なら半年後
	 * ・BSD ：先月の最初の日付にセットします。(先月１日)(5.5.5.2 追加)
	 * ・BED ：先月の最後の日付にセットします。(先月月末)(5.5.5.2 追加)
	 * ・ASD ：翌月の最初の日付にセットします。(翌月１日)(5.5.5.2 追加)
	 * ・AED ：翌月の最後の日付にセットします。(翌月月末)(5.5.5.2 追加)
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 * @og.rev 5.6.1.1 (2013/02/08) prmB処理を、calendarCalc メソッドへ移動
	 *
	 * @param   key		フォーマットの予約語
	 * @param   prmA	基準となる日付(nullの場合は、処理時刻)
	 * @param   prmB	処理コマンド
	 *
	 * @return   メッセージ情報
	 */
	public static final String getDateFormat( final String key ,final String prmA ,final String prmB ) {

		// prmA が null の場合は、そのまま、現在時刻が使われます。
		Calendar now = getCalendar( prmA );

		// 5.6.1.1 (2013/02/08) getDateFormat( String ,String ,String ) から分離。
		calendarCalc( now,prmB );

//		if( prmB != null ) {
//			if( "SD".equals( prmB ) ) { now.set( Calendar.DATE,1 ); }
//			else if( "ED".equals( prmB ) ) { now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) ); }
//			// 5.5.5.2 (2012/08/18) BSD,BED,ASD,AED 追加
//			else if( "BSD".equals( prmB ) ) {				// (先月１日)
//				now.roll( Calendar.MONTH,false ); now.set( Calendar.DATE,1 );
//			}
//			else if( "BED".equals( prmB ) ) {				// (先月月末)
//				now.roll( Calendar.MONTH,false ); now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) );
//			}
//			else if( "ASD".equals( prmB ) ) {				// (翌月１日)
//				now.roll( Calendar.MONTH,true );  now.set( Calendar.DATE,1 );
//			}
//			else if( "AED".equals( prmB ) ) {				// (翌月月末)
//				now.roll( Calendar.MONTH,true );  now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) );
//			}
//			else if( "SW".equals( prmB ) ) {
//				// 日付型文字列入力データの開始日を月曜日にセットします。
//				// SUNDAY=1 , MONDAY=2 になります。月曜日との差だけ、前に戻します。
//				// 指定日が日曜日の場合は、月曜日まで戻します。
//				int shu = now.get( Calendar.DAY_OF_WEEK ) - Calendar.MONDAY ;
//
//				if(      shu > 0 ) { now.add( Calendar.DATE, -shu ); }
//				else if( shu < 0 ) { now.add( Calendar.DATE, -6 );   }	// 日曜日の処理
//
//			}
//			else if( "EW".equals( prmB ) ) {
//				// 日付型文字列入力データの終了日を日曜日にセットします。
//				// SUNDAY=1 , MONDAY=2 になります。日曜日になるように、先に進めます。
//				int shu = now.get( Calendar.DAY_OF_WEEK ) ;
//
//				if( shu != Calendar.SUNDAY ) { now.add( Calendar.DATE, 8-shu ); }
//			}
//			// 5.5.5.6 (2012/08/31) H1 ～ HXXX 追加。時間の加算を指定できる。
//			else if( prmB.startsWith( "H" ) && prmB.length() > 1 ) {
//				int hour = Integer.parseInt( prmB.substring( 1 ) );
//				now.add( Calendar.HOUR_OF_DAY, hour );
//			}
//			else if( prmB.startsWith( "D" ) && prmB.length() > 1 ) {
//				int day = Integer.parseInt( prmB.substring( 1 ) );
//				now.add( Calendar.DATE, day );
//			}
//			else if( prmB.startsWith( "M" ) && prmB.length() > 1 ) {
//				int month = Integer.parseInt( prmB.substring( 1 ) );
//				now.add( Calendar.MONTH , month );
//			}
//			else {
//				String errMsg = "日付変数パラメータに、不正な値が指定されました。以下の中から指定しなおしてください。"
//							+ "指定可能：[SD,ED,BSD,BED,ASD,AED,SW,EW,D1～DXXX,M1～MXXX]"
//							+ " prmB=[" + prmB + "]" ;
//				throw new RuntimeException( errMsg );
//			}
//		}

		String format = DATE_FORMAT.get( key );
		if( format == null ) {
	//		// DATE_FORMAT に存在しないフォーマットを指定しても、エラーにしません。ただし、後処理でフォーマットエラーになる可能性は残ります。
	//		String errMsg = "日付変数キーに、不正な値が指定されました。以下の中から指定しなおしてください。"
	//					+ CR
	//					+ "指定可能：" + Arrays.toString( DATE_FORMAT.keySet().toArray( new String[DATE_FORMAT.size()] ) )	// 5.5.5.2 (2012/08/18)
	//					+ CR
	//					+ " key=[" + key + "]" ;
	//		// 5.5.5.2 (2012/08/18) 自由フォーマットを設定可能とします。
	//		System.err.println( "CommonTagSupport {@DATE.XXXX AA BB} 警告" );
	//		System.err.println( errMsg );
			format = key;	// 5.5.5.2 (2012/08/18) 自由フォーマット指定
		}

		//5.5.0.2 先頭Gの場合は和暦なのでformatterのLocaleを変更する
		DateFormat formatter = null;
		if( key.indexOf('G') == 0 ){
			formatter = new SimpleDateFormat( format, new Locale("ja","JP","JP"));
		}
		else{
			formatter = new SimpleDateFormat( format,Locale.JAPAN );
		}

		return formatter.format( now.getTime() );
	}

	/**
	 * 開始前設定値、または 終了後設定値の文字列から、オプション文字列を合成します。
	 * 基準となる日付に計算した結果を反映させます。
	 *
	 * prmB は、日付についての加減算処理を行うためのコマンドを指定します。
	 * ・SD ：当月の最初の日付にセットします。(当月１日)
	 * ・ED ：当月の最後の日付にセットします。(当月月末)
	 * ・SW ：日付処理の週初め(月曜日)にセットします。日付は当日より前に移動します。
	 * ・EW ：日付処理の週末(日曜日)にセットします。日付は当日より後ろに移動します。
	 * ・D1 ～ DXXX ：日を指定の分だけ進めます。D1なら翌日、D200 なら200日後
	 * ・M1 ～ MXXX ：月を指定の分だけ進めます。M1なら翌月、M6 なら半年後
	 * ・BSD ：先月の最初の日付にセットします。(先月１日)(5.5.5.2 追加)
	 * ・BED ：先月の最後の日付にセットします。(先月月末)(5.5.5.2 追加)
	 * ・ASD ：翌月の最初の日付にセットします。(翌月１日)(5.5.5.2 追加)
	 * ・AED ：翌月の最後の日付にセットします。(翌月月末)(5.5.5.2 追加)
	 * ・数字：日を指定の分だけ進めます。D1 ～ DXXX の簡略系
	 *
	 * @og.rev 5.6.1.1 (2013/02/08) getDateFormat( String ,String ,String ) から分離。
	 *
	 * @param   now     基準となる日付（Calendarオブジェクト）
	 * @param   prmB	処理コマンド
	 *
	 */
	public static final void calendarCalc( final Calendar now,final String prmB ) {

		if( prmB != null ) {
			if( "SD".equals( prmB ) ) { now.set( Calendar.DATE,1 ); }
			else if( "ED".equals( prmB ) ) { now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) ); }
			else if( "BSD".equals( prmB ) ) {				// (先月１日)
				now.roll( Calendar.MONTH,false ); now.set( Calendar.DATE,1 );
			}
			else if( "BED".equals( prmB ) ) {				// (先月月末)
				now.roll( Calendar.MONTH,false ); now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) );
			}
			else if( "ASD".equals( prmB ) ) {				// (翌月１日)
				now.roll( Calendar.MONTH,true );  now.set( Calendar.DATE,1 );
			}
			else if( "AED".equals( prmB ) ) {				// (翌月月末)
				now.roll( Calendar.MONTH,true );  now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) );
			}
			else if( "SW".equals( prmB ) ) {
				// 日付型文字列入力データの開始日を月曜日にセットします。
				// SUNDAY=1 , MONDAY=2 になります。月曜日との差だけ、前に戻します。
				// 指定日が日曜日の場合は、月曜日まで戻します。
				int shu = now.get( Calendar.DAY_OF_WEEK ) - Calendar.MONDAY ;

				if(      shu > 0 ) { now.add( Calendar.DATE, -shu ); }
				else if( shu < 0 ) { now.add( Calendar.DATE, -6 );   }	// 日曜日の処理

			}
			else if( "EW".equals( prmB ) ) {
				// 日付型文字列入力データの終了日を日曜日にセットします。
				// SUNDAY=1 , MONDAY=2 になります。日曜日になるように、先に進めます。
				int shu = now.get( Calendar.DAY_OF_WEEK ) ;

				if( shu != Calendar.SUNDAY ) { now.add( Calendar.DATE, 8-shu ); }
			}
			else if( prmB.startsWith( "D" ) && prmB.length() > 1 ) {
				int day = Integer.parseInt( prmB.substring( 1 ) );
				now.add( Calendar.DATE, day );
			}
			else if( prmB.startsWith( "M" ) && prmB.length() > 1 ) {
				int month = Integer.parseInt( prmB.substring( 1 ) );
				now.add( Calendar.MONTH , month );
			}
			else {
				// 上記のパターン以外は、数字（日付）なので、変換できなければ、フォーマットエラー
				try {
					int day = Integer.parseInt( prmB );
					now.add( Calendar.DATE, day );
				}
				catch( NumberFormatException ex ) {
					String errMsg = "日付変数パラメータに、不正な値が指定されました。以下の中から指定しなおしてください。"
								+ "指定可能：[SD,ED,BSD,BED,ASD,AED,SW,EW,D1～DXXX,M1～MXXX]"
								+ " prmB=[" + prmB + "]" ;
					throw new RuntimeException( errMsg,ex );
				}
			}
		}
	}

	/**
	 * 指定の引数の日付け文字列より、カレンダオブジェクトを作成します。
	 * 引数は、数字以外の文字を削除した状態に変換後、処理に回します。
	 * 不要な文字を削除した状態で、８文字以上になるように指定してください。
	 * 例外的に、６文字の場合は、yyyyMM01 とみなして、"01" 文字列を付与します。
	 * 引数に null を指定すると、現在時刻のカレンダを返します。
	 * それ以外のデータで、８ケタ以下の場合は、RuntimeException が発生します。
	 * ８ケタ以上１４ケタ未満の場合は、８ケタ分を、年月日に分離したカレンダ
	 * オブジェクトを作成します。１４ケタ以上で初めて、時分秒を含むカレンダ
	 * を作成します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 * @og.rev 5.5.8.2 (2012/11/09) value の判定に、null と ゼロ文字列を判定する。
	 *
	 * @param value 日付け文字列
	 *
	 * @return	カレンダオブジェクト(引数がnullの場合は、現在時刻)
	 */
	public static final Calendar getCalendar( final String value ) {
		Calendar cal = Calendar.getInstance();

//		if( value == null ) { return cal; }
		if( value == null || value.isEmpty() ) { return cal; }		// 5.5.8.2 (2012/11/09) null と ゼロ文字列を判定する。

		// 日付表記に不要な文字を削除します。
		String dateStr = parseNumber( value ) ;

		if( dateStr.length() == 6 ) { dateStr = dateStr + "01"; }	// yyyyMM01 形式に無理やり合わせる。
		else if( dateStr.length() < 8 ) {
			String errMsg = "日付指定パラメータに、不正な値が指定されました。value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}

		cal.clear();	// 日付文字列が存在するので、カレンダをリセット

		int year   = Integer.parseInt( dateStr.substring( 0,4 ) );
		int month  = Integer.parseInt( dateStr.substring( 4,6 ) ) - 1;
		int date   = Integer.parseInt( dateStr.substring( 6,8 ) );

		int hour=0, minute=0, second=0;
		if( dateStr.length() >= 14 ) {
			hour   = Integer.parseInt( dateStr.substring( 8,10 ) );
			minute = Integer.parseInt( dateStr.substring( 10,12 ) );
			second = Integer.parseInt( dateStr.substring( 12,14 ) );
		}

		cal.set( year,month,date,hour,minute,second );

		return cal;
	}

	/**
	 * 指定の引数の日付け文字列(yyyyMMdd)より、日付を加算して返します。
	 * マイナスを与えると、減算します。
	 * 日付以上の精度の文字列を渡しても、日付のみの計算となります。
	 * 結果は、引数の日付フォーマットとは全く別で、yyyyMMdd の８文字形式になります。
	 * 引数に null を渡すと、実行時の日付をベースとして処理します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param baseDate 日付け文字列(yyyyMMdd)
	 * @param plus     加算する日数(過去にするにはマイナス値を指定する)
	 *
	 * @return	結果の日付(yyyyMMdd)
	 */
	public static final String getDatePlus( final String baseDate,final int plus ) {
		Calendar cal = getCalendar( baseDate );
		cal.add( Calendar.DATE,plus );

		return getDate( cal.getTimeInMillis() , "yyyyMMdd" );
	}

	/**
	 * 現在の月に、指定の月数をプラスした日付文字列を返します。
	 * 日付文字列のフォーマットは、"yyyyMM" です。
	 * 指定する月数にマイナスを指定すると、減算できます。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param baseDate 日付け文字列(yyyyMM)
	 * @param plus     加算する月数(過去にするにはマイナス値を指定する)
	 *
	 * @return	指定の月数をプラスした日付文字列(yyyyMM)
	 */
	public static final String getMonthPlus( final String baseDate,final int plus ) {
		Calendar cal = getCalendar( baseDate );
		cal.set( Calendar.DATE, 1 );		// 当月の 1 日に設定
		cal.add( Calendar.MONTH , plus );

		return getDate( cal.getTimeInMillis() , "yyyyMM" );
	}

	/**
	 * 指定の引数の日付け文字列(yyyyMMdd、yyyyMMddHHmmss)に、日付を加算して返します。
	 * マイナスを与えると、減算します。
	 *
	 * 指定する日付には、単位を付与することが可能です。
	 * 単位は、yyyyMMddHHmmss 形式の１文字を指定します。大文字、小文字も識別します。
	 * plus="5M" とすれば、５か月、plus="5d"  とすれば、５日 追加します。
	 * plus に単位を付けない場合は、tani に指定の単位を使います。
	 * plus そのものが、null か、isEmpty の場合は、加算は、1 になります。
	 *
	 * baseDate 文字列を日付文字列に変換後、Calendar で計算し、結果を、format 形式に変換します。
	 * 引数に null を渡すと、実行時の日付をベースとして処理します。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
	 *
	 * @param baseDate 日付け文字列(yyyyMMdd、yyyyMMddHHmmss 形式の日付文字列)
	 * @param plus     加算する日数(日付単位を含む。単位は、y,M,d,H,m,s の文字で、大文字小文字の区別があります)
	 * @param defTani  日付単位が未指定の場合の初期単位('y','M','d','H','m','s' のどれか)
	 * @param format   返す日付文字列のフォーマット(yyyyMMdd、yyyyMMddHHmmss)
	 *
	 * @return	結果の日付(yyyyMMdd)
	 * @throws	NumberFormatException 加算する日数の単位が('y','M','d','H','m','s')以外の場合。
	 */
	public static final String getDatePlus( final String baseDate,final String plus,final int defTani,final String format ) {

		int addSu = 1;				// 初期値(plus が null や Empty の場合は、＋１となる)
		int tani  = defTani;

		if( plus != null && !plus.isEmpty() ) {
			boolean flag = true;	// 日付単位を持っているかどうか。持っている場合は、true
			char ch = plus.charAt( plus.length()-1 );		// 最後の一文字を取得(単位か、数字本体)
			switch( ch ) {
				case 'y' : tani = Calendar.YEAR;		break ;
				case 'M' : tani = Calendar.MONTH;		break ;
				case 'd' : tani = Calendar.DATE;		break ;
				case 'H' : tani = Calendar.HOUR_OF_DAY;	break ;
				case 'm' : tani = Calendar.MINUTE;		break ;
				case 's' : tani = Calendar.SECOND;		break ;
				default	 : flag = false;	break ;		// 日付単位を持っていない。
			}
			if( flag ) {
				addSu = Integer.parseInt( plus.substring( 0,plus.length()-1 ) );	// 日付単位 あり
			}
			else {
				addSu = Integer.parseInt( plus ) ;									// 日付単位 なし
			}
		}

		Calendar cal = getCalendar( baseDate );
		cal.add( tani,addSu );

		return getDate( cal.getTimeInMillis() , format );
	}
}
