/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

// import java.text.DateFormat;
// import java.text.SimpleDateFormat;
// import java.util.Date;
// import java.util.Locale;

/**
 * Logを書き込む為の LogWriter を呼び出す 簡易クラスです。
 *
 * Log の書き込み日時や，キー(識別ID)などを出力します。
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class LogSender {

	/** Log出力レベル FATAL  {@value} */
	public static final int FATAL        = 4;
	/** Log出力レベル ERROR  {@value} */
	public static final int ERROR        = 3;
	/** Log出力レベル WARNING  {@value} */
	public static final int WARNING      = 2;
	/** Log出力レベル INFORMATION  {@value} */
	public static final int INFORMATION  = 1;
	/** Log出力レベル DEBUG  {@value} */
	public static final int DEBUG        = 0;

	private static final int LOG_LEVEL		= ERROR;		// 初期値

	private final String userId;
	private final int    logLevel;
	private String guiId = null;
	private String msgId = null;
	private StringBuilder buf = null;

	/** システム依存の改行記号をセットします。	*/
	private static final String CR = System.getProperty("line.separator");

	/**
	 * デフォルトコンストラクター
	 *
	 */
	public LogSender() {
		this( null,LOG_LEVEL );
	}

	/**
	 * ユーザーIDを与えてのコンストラクター
	 * ユーザーIDは，Logの識別IDとして使用します。
	 *
	 * @param   userId  ユーザーID
	 */
	public LogSender( final String userId ) {
		this( userId,LOG_LEVEL );
	}

	/**
	 * ユーザーIDと、ログレベルを与えてのコンストラクター
	 * ユーザーIDは，Logの識別IDとして使用します。
	 * ログレベルは，0 ～ 4 の数字で、大きくなるほど、出力条件はきつくなります。
	 * 0:DEBUG, 1:INFORMATION, 2:WARNING, 3:ERROR, 4:FATAL
	 * 初期値は、3:ERROR です。
	 *
	 * @param   userId  ユーザーID
	 * @param   logLbl  ログレベル(0 ～ 4 の数字)
	 */
	public LogSender( final String userId,final int logLbl ) {
		this.userId = userId;
		logLevel = logLbl;
//		buf		 = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		buf		 = new StringBuilder( 200 ); // 4.1.0.1 (2008/01/23)

		if( logLevel < DEBUG || logLevel > FATAL ) {
			String errMsg = "ログレベルの指定レベルが、間違っています。"
						+ "指定レベル=[" + logLbl + "]"
						+ "  指定可能レベル="
						+ DEBUG + " ～ " + FATAL ;
//			throw new HybsSystemException( errMsg );
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 画面IDを設定します。
	 * ログ集計時の参考情報に使います。
	 *
	 * @param	guiId	画面ID
	 */
	public void setGuiId( final String guiId ) {
		this.guiId = guiId;
	}

	/**
	 * メッセージIDを設定します。
	 * ログ集計時の参考情報に使います。
	 *
	 * @param	msgId	メッセージID
	 */
	public void setMsgId( final String msgId ) {
		this.msgId = msgId;
	}

	/**
	 * Logレベルを取得します。
	 * ログレベルは, FATAL,ERROR,WARNING,INFORMATION,DEBUG から選べます。
	 *
	 * @return   Logレベル
	 */
	public int getLogLevel() {
		return logLevel;
	}

	/**
	 * LogをLogWriter に渡します。
	 * DEBUG ログレベルと，あらかじめ設定されているLogレベルを
	 * 比較して, ログレベルが小さい場合は,出力しません。
	 *
	 * @param   message メッセージ
	 */
	public void debug( final String message ) {
		if( DEBUG < logLevel ) { return; }
		log( "DEBUG",message ) ;
	}

	/**
	 * LogをLogWriter に渡します。
	 * INFORMATION ログレベルと，あらかじめ設定されているLogレベルを
	 * 比較して, ログレベルが小さい場合は,出力しません。
	 *
	 * @param   message メッセージ
	 */
	public void info( final String message ) {
		if( INFORMATION < logLevel ) { return; }
		log( "INFO",message ) ;
	}

	/**
	 * LogをLogWriter に渡します。
	 * WARNING ログレベルと，あらかじめ設定されているLogレベルを
	 * 比較して, ログレベルが小さい場合は,出力しません。
	 *
	 * @param   message メッセージ
	 */
	public void warn( final String message ) {
		if( WARNING < logLevel ) { return; }
		log( "WARNING",message ) ;
	}

	/**
	 * LogをLogWriter に渡します。
	 * ERROR ログレベルと，あらかじめ設定されているLogレベルを
	 * 比較して, ログレベルが小さい場合は,出力しません。
	 *
	 * @param   message メッセージ
	 */
	public void error( final String message ) {
		if( ERROR < logLevel ) { return; }
		log( "ERROR",message ) ;
	}

	/**
	 * LogをLogWriter に渡します。
	 * FATAL ログレベルと，あらかじめ設定されているLogレベルを
	 * 比較して, ログレベルが小さい場合は,出力しません。
	 *
	 * @param   message メッセージ
	 */
	public void fatal( final String message ) {
		if( FATAL < logLevel ) { return; }
		log( "FATAL",message ) ;
	}

	/**
	 * LogをLogWriter に渡します。
	 * ただし，引数でセットしたLoglevelと，あらかじめ設定されているLogレベルを
	 * 比較して, 引数のLoglebelの方が小さい場合は,出力しません。
	 * ログレベルは, FATAL,ERROR,WARNING,INFORMATION,DEBUG から選べます。
	 *
	 * @og.rev 3.5.5.4 (2004/04/15) ログにエラーレベルを記述します。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param   level   ログレベル(FATAL,ERROR,WARNING,INFORMATION,DEBUG)
	 * @param   message メッセージ
	 */
	private void log( final String level ,final String message ) {
		buf.append( "[" );
		buf.append( level );
//		buf.append( " : Time=" ).append( HybsSystem.getDate() );
//		DateFormat formatter = new SimpleDateFormat( "yyyy/MM/dd HH:mm:ss.SSS",Locale.JAPAN );
//		buf.append( " : Time=" ).append( formatter.format( new Date() ) );  // 4.1.0.1 (2008/01/23)
		buf.append( " : Time=" ).append( HybsDateUtil.getDate( "yyyy/MM/dd HH:mm:ss.SSS" ) );  // 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		buf.append( " , User=" ).append( userId );
		buf.append( " , Gui="  ).append( guiId );
		buf.append( " , Msg="  ).append( msgId );
		buf.append( "]");
		buf.append( CR );
		buf.append( message );
	}

	/**
	 * LogWriter を明示的にclose() します。
	 * LogWriter は，内部的に キャッシュを持っており,初めてアクセスされた場合に
	 * すでにオブジェクトが存在していれば,それを使い,無ければ 新たに追加モードで
	 * PrintWriter を作成します。
	 * よって，明示的にclose()する必要性は ほとんど有りません。
	 *
	 */
	public void flush() {
		LogWriter.log( buf.toString() );
	}

	/**
	 * LogWriter を明示的にclose() します。
	 * LogWriter は，内部的に キャッシュを持っており,初めてアクセスされた場合に
	 * すでにオブジェクトが存在していれば,それを使い,無ければ 新たに追加モードで
	 * PrintWriter を作成します。
	 * よって，明示的にclose()する必要性は ほとんど有りません。
	 *
	 */
	public void close() {
		LogWriter.close();
	}
}
