/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.HybsDateUtil;

import org.opengion.fukurou.security.HybsCryptography ;		// 5.7.2.1 (2014/01/17)

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;
// import java.util.Date;
// import java.util.Locale;
// import java.text.DateFormat;
// import java.text.SimpleDateFormat;

/**
 * FileLineModel は、LineModel を継承した ファイルリスト専用の
 * LineModel の実装クラスです。
 *
 * FileLineModel オブジェクトには、ファイル属性(Level,File,Length,Modify)
 * が設定されます。
 *
 * データの１行分を FileLineModel に割り当てます。
 * カラム番号は、0 から始まります。カラム名よりカラム番号を求める場合に、
 * 存在しない場合は、-1 を返します。
 * カラム番号が -1 の場合は、処理を行いません。
 *
 * 注意：このクラスは、同期処理されていません。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileLineModel extends LineModel {
	// 5.7.2.1 (2014/01/17) MD5 項目追加
//	private static final String[] KEYS = new String[] { "Level","File","Length","Modify","LineCnt","Biko" };
	private static final String[] KEYS = new String[] { "Level","File","Length","Modify","LineCnt","Biko","MD5" };

	private static final int LEVEL    = 0;
	private static final int FILE     = 1;
	private static final int LENGTH   = 2;
	private static final int MODIFY   = 3;
	private static final int LINECNT  = 4;
	private static final int BIKO     = 5;
	private static final int MD5      = 6;		// 5.7.2.1 (2014/01/17)

//	private final DateFormat formatter = new SimpleDateFormat( "yyyy/MM/dd HH:mm:ss",Locale.JAPAN );

	private final boolean useLineCnt ;
	private final boolean useMD5 ;				// 5.7.2.1 (2014/01/17) MD5 項目追加

	/**
	 * コンストラクターです。
	 * ラインカウントとMD5計算の初期値は、false です。
	 *
	 * @og.rev 5.7.2.1 (2014/01/17) MD5対応
	 *
	 */
	public FileLineModel() {
//		this( false );
		this( false,false );					// 5.7.2.1 (2014/01/17)
//		super();
//		init( KEYS );
	}

	/**
	 * ラインカウントの有無を指定した、コンストラクターです。
	 * MD5計算の初期値は、false です。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 行数カウントの使用有無
	 * @og.rev 5.7.2.1 (2014/01/17) MD5対応
	 *
	 * @param	isLineCnt	行数カウントの使用有無
	 */
	public FileLineModel( final boolean isLineCnt ) {
		this( isLineCnt,false );				// 5.7.2.1 (2014/01/17)

		// 4.3.4.4 (2009/01/01)
//		super();
//		useLineCnt = isLineCnt;
//		init( KEYS );
	}

	/**
	 * ラインカウントの有無と、MD5計算の有無を指定した、コンストラクターです。
	 *
	 * @og.rev 5.7.2.1 (2014/01/17) 新規追加(MD5対応)
	 *
	 * @param	isLineCnt	行数カウントの使用有無
	 * @param	isMD5		ファイルのMD5の使用有無
	 */
	public FileLineModel( final boolean isLineCnt,final boolean isMD5 ) {
		// 4.3.4.4 (2009/01/01)
//		super();
		useLineCnt = isLineCnt;
		useMD5     = isMD5;				// 5.7.2.1 (2014/01/17)
		init( KEYS );
	}

	/**
	 * LineModel を元に、FileLineModel を構築します。
	 * これは、一旦ファイル等にセーブされた FileLineModel 形式を
	 * 元に戻す簡易コンストラクタです。
	 *
	 * @og.rev 4.2.3.0 (2008/05/26) 新規追加
	 * @og.rev 5.7.2.1 (2014/01/17) MD5の設定処理追加
	 *
	 * @param	model	元のLineModel
	 */
	public FileLineModel( final LineModel model ) {
		// 4.3.4.4 (2009/01/01)
//		super();
		init( model.getNames() );

		Object[] obj = model.getValues();

		setValue( LEVEL   ,Integer.valueOf( (String)obj[LEVEL] ) );
		setValue( FILE    ,new File((String)obj[FILE]) );
		setValue( LENGTH  ,Long.valueOf( (String)obj[LENGTH] ) );
		setValue( MODIFY  ,(String)obj[MODIFY] );

		String cnt = (String)obj[LINECNT] ;
//		useLineCnt = ( cnt != null && cnt.length() > 0 && ! "null".equals( cnt ) );
		useLineCnt = ( cnt != null && cnt.length() > 0 && ! "null".equalsIgnoreCase( cnt ) );
		if( useLineCnt ) { setValue( LINECNT ,cnt ); }

		setValue( BIKO  ,(String)obj[BIKO] );

		// 5.7.2.1 (2014/01/17) 
		String md5Data = (String)obj[MD5] ;
		useMD5 = ( md5Data != null && md5Data.length() > 0 && ! "null".equalsIgnoreCase( md5Data ) );
		if( useMD5 ) { setValue( MD5 ,md5Data ); }
	}

	/**
	 * File属性値をセットします。
	 * LEVEL,FILE,LENGTH,MODIFY,LINECNT,MD5 の各属性を設定します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 行数カウントの使用有無
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 5.7.2.1 (2014/01/17) MD5計算処理の追加
	 *
	 * @param	level	ファイルのディレクトリ階層
	 * @param	file	ファイルオブジェクト
	 */
	public void setFileVals( final int level, final File file ) {
		setValue( LEVEL  ,Integer.valueOf( level ) );
		setValue( FILE   ,file );
		setValue( LENGTH ,Long.valueOf( file.length() ) );
//		setValue( MODIFY ,formatter.format( new Date( file.lastModified() ) ) );
		setValue( MODIFY ,HybsDateUtil.getDate( file.lastModified(),"yyyy/MM/dd HH:mm:ss" ) );		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用する
		if( useLineCnt ) {
			setValue( LINECNT ,getLineCnt( file ) );
		}

		// 5.7.2.1 (2014/01/17) MD5計算がtrue で、かつ、ファイルの場合、MD5 計算を行います。
		if( useMD5 && file.isFile() ) {
			setValue( MD5 ,HybsCryptography.getMD5( file ) );
		}
	}

	/**
	 * File属性値をセットします。
	 *
	 * @param	file	ファイルオブジェクト
	 */
	public void setFile( final File file ) {
		setValue( FILE,file );
	}

	/**
	 * 備考情報属性値をセットします。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 行数カウントの使用有無
	 *
	 * @param	biko	備考情報
	 */
	public void setBiko( final String biko ) {
		setValue( BIKO,biko );
	}

	/**
	 * レベル File属性値を取得します。
	 *
	 * @return	ファイルのディレクトリ階層
	 */
	public int getLebel() {
		return ((Integer)getValue( LEVEL )).intValue();
	}

	/**
	 * ファイルを取得します。
	 *
	 * @return	ファイル
	 */
	public File getFile() {
		return (File)getValue( FILE );
	}

	/**
	 * ファイルサイズ File属性値を取得します。
	 *
	 * @return	ファイルサイズ
	 */
	public long getLength() {
		return ((Long)getValue( LENGTH )).longValue();
	}

	/**
	 * 更新日時 File属性値を取得します。
	 *
	 * @return	更新日時(yyyy/MM/dd HH:mm:ss)
	 */
	public String getModify() {
		return (String)getValue( MODIFY );
	}

	/**
	 * MD5 File属性値を取得します。
	 * ただし、useMD5 が true でないと値は返しません。
	 *
	 * @og.rev 5.7.2.1 (2014/01/17) 新規追加(MD5対応)
	 *
	 * @return	MD5の値
	 */
	public String getMD5() {
		return (String)getValue( MD5 );
	}

	/**
	 * 行数を取得します。
	 *
	 * @param	file	行数を数えるファイルオブジェクト
	 *
	 * @return  行数
	 */
	private String getLineCnt( final File file ) {
		int cnt = 0;

		BufferedReader reader = FileUtil.getBufferedReader( file,"JISAutoDetect" );

		try {
			if( ! file.isDirectory() ) {
//				String line ;	// findbugs で、意味の無い代入チェックがかかりますが、OKです。
//				while((line = reader.readLine()) != null) {
//					cnt++;
//				}
				while( reader.readLine() != null) {
					cnt++;
				}
			}
		}
		catch( IOException ex ) {
			String errMsg = "ファイルカウント中に例外が発生しました。[" + file + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader ) ;
		}

		return String.valueOf( cnt );
	}
}
