/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.LogWriter;

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.io.File;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Process_FileCopy は、上流から受け取った FileLineModel を処理する、
 * ChainProcess インターフェースの実装クラスです。
 *
 * 上流から受け取った FileLineModel の ファイルから、inPath の共通パス
 * 以下のファイルを、outPath の共通パス以下にコピーします。
 * コピーの種類は、バイナリか、テキストで、テキストの場合は、エンコード
 * 変換も行うことが可能です。
 * inPath と outPath が同じ、または、outPath が未設定の場合は、入力と出力が
 * 同じですので、自分自身のエンコード変換処理を行うことになります。
 *
 * コピーされるファイルのファイル名は、入力ファイル名と同一です。保存される
 * フォルダが異なります。(同一にすることも可能です。)
 *
 * 上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト
 * である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを
 * 使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し
 * できれば、使用可能です。
 *
 * 引数文字列中に空白を含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_FileCopy -inPath=入力共通パス -inEncode=Windows-31J -outPath=出力共通パス -outEncode=UTF-8
 *
 *     -inPath=入力共通パス         ：上流で検索されたファイルパスの共通部分
 *   [ -inEncode=入力エンコード   ] ：入力ファイルのエンコードタイプ
 *   [ -outPath=出力共通パス      ] ：出力するファイルパスの共通部分
 *   [ -outEncode=出力エンコード  ] ：出力ファイルのエンコードタイプ
 *   [ -binary=[false/true]       ] ：trueは、バイナリファイルのコピー(初期値:false)
 *   [ -changeCrLf=[false/true]   ] ：trueは、バイナリファイルのコピー時にCR+LFに変換します(初期値:false)
 *   [ -keepTimeStamp=[false/true]] ：trueは、コピー元のファイルのタイムスタンプで作成します(初期値:false)
 *   [ -display=[false/true]      ] ：trueは、コピー状況を表示します(初期値:false)
 *   [ -debug=false|true          ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_FileCopy extends AbstractProcess implements ChainProcess {
	private File	tempFile	= null;

	private String	inPath			= null;
	private String	inEncode		= null;
	private String	outPath			= null;
	private String	outEncode		= null;
	private boolean	binary			= false;
	private boolean	changeCrLf		= false;	// 4.2.2.0 (2008/05/10)
	private boolean	keepTimeStamp	= false;	// 5.1.5.0 (2010/04/01)
	private boolean	display			= false;
	private boolean	debug			= false;	// 5.7.3.0 (2014/02/07) デバッグ情報

	private int		inPathLen	= 0;
	private boolean	isEquals	= false;
	private int		inCount		= 0;

	private static final Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "inPath",	"コピー元のファイル基準パス" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "inEncode",			"コピー元のファイルのエンコードタイプ" );
		usableProparty.put( "outPath",			"コピー先のファイル基準パス" );
		usableProparty.put( "outEncode",		"コピー先のファイルのエンコードタイプ" );
		usableProparty.put( "binary",			"trueは、バイナリファイルをコピーします(初期値:false)" );
		usableProparty.put( "changeCrLf",		"trueは、バイナリファイルのコピー時にCR+LFに変換します(初期値:false)" );		// 4.2.2.0 (2008/05/10)
		usableProparty.put( "keepTimeStamp",	"trueは、コピー元のファイルのタイムスタンプで作成します(初期値:false)" );	// 5.1.5.0 (2010/04/01)
		usableProparty.put( "display",			"trueは、コピー状況を表示します(初期値:false)" );
		usableProparty.put( "debug",			"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
													CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_FileCopy() {
		super( "org.opengion.fukurou.process.Process_FileCopy",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) changeCrLf 属性対応
	 * @og.rev 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		inPath			= arg.getProparty("inPath" );
		outPath			= arg.getProparty("outPath" );
		inEncode		= arg.getProparty("inEncode" ,System.getProperty("file.encoding"));
		outEncode		= arg.getProparty("outEncode",System.getProperty("file.encoding"));
		binary			= arg.getProparty("binary" ,binary);
		changeCrLf		= arg.getProparty("changeCrLf" ,changeCrLf);		// 4.2.2.0 (2008/05/10)
		keepTimeStamp	= arg.getProparty("keepTimeStamp" ,keepTimeStamp);	// 5.1.5.0 (2010/04/01)
		display			= arg.getProparty("display",display);
		debug			= arg.getProparty("debug",debug);				// 5.7.3.0 (2014/02/07) デバッグ情報
//		if( debug ) { println( arg.toString() ); }			// 5.7.3.0 (2014/02/07) デバッグ情報

		inPathLen = inPath.length();

		// 入力と出力が同じか？
		isEquals  = outPath == null || inPath.equalsIgnoreCase( outPath );

		if( binary ) {
			// 4.2.2.0 (2008/05/10) 判定ミスの修正
			if( ! inEncode.equalsIgnoreCase( outEncode ) ) {
				String errMsg = "バイナリコピー時には、入出力のエンコードは同じ必要があります。" + CR
							+ " inEncode=[" + inEncode + "] , outEncode=[" + outEncode + "]" ;
				throw new RuntimeException( errMsg );
			}
			if( isEquals ) {
				String errMsg = "入出力が同じファイルのバイナリコピーはできません。" + CR
							+ " inPath=[" + inPath + "] , outPath=[" + outPath + "]" ;
				throw new RuntimeException( errMsg );
			}
		}

		// 入力と出力が同じ場合は、中間ファイルを作成します。
		if( isEquals ) {
			try {
				tempFile = File.createTempFile( "X", ".tmp", new File( outPath ) );
				tempFile.deleteOnExit();
			}
			catch( IOException ex ) {
				String errMsg = "中間ファイル作成でエラーが発生しました。" + CR
							+ " outPath=[" + outPath + "]" ;
				throw new RuntimeException( errMsg,ex );
			}
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		tempFile  = null;
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.2.2.0 (2008/05/10) changeCrLf 属性対応
	 * @og.rev 4.2.3.0 (2008/05/26) LineModel が FileLineModel でない場合の処理
	 * @og.rev 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
	 * @og.rev 5.1.6.0 (2010/05/01) changeCrLf 属性が、.FileUtil#changeCrLfcopy メソッドへの移動に伴う対応
	 * @og.rev 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		inCount++ ;
		final FileLineModel fileData ;
		if( data instanceof FileLineModel ) {
			fileData = (FileLineModel)data ;
		}
		else {
			// LineModel が FileLineModel でない場合、オブジェクトを作成します。
			fileData = new FileLineModel( data );
//			String errMsg = "データが FileLineModel オブジェクトではありません。" + CR ;
//			throw new RuntimeException( errMsg );
		}

		if( debug ) { println( "Before:" + data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更

		File inFile = fileData.getFile() ;
		if( ! inFile.isFile() ) {
			if( display ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
			return data;
		}

		// ファイル名を作成します。
		// ファイル名は、引数ファイル名 から、inPath を引き、outPath を加えます。
		File outFile = new File( outPath, inFile.getAbsolutePath().substring( inPathLen ) );
		fileData.setFile( outFile );

//		if( display ) { println( inFile + " => " + outFile ); }

		// 入出力が異なる場合
		if( !isEquals ) {
			tempFile = outFile;
			File parent = outFile.getParentFile();
			if( parent != null && ! parent.exists() && !parent.mkdirs() ) {
				String errMsg = "所定のフォルダが作成できませんでした。[" + parent + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg );
			}
		}

		if( binary ) {
//			FileUtil.copy( inFile,tempFile );
//			FileUtil.copy( inFile,tempFile,changeCrLf );	// 4.2.2.0 (2008/05/10)
			// 5.1.6.0 (2010/05/01) changeCrLfcopy 対応
			if( changeCrLf ) { FileUtil.changeCrLfcopy( inFile,tempFile ); }
			else             { FileUtil.copy( inFile,tempFile,keepTimeStamp ); }
		}
		else {
			BufferedReader reader = FileUtil.getBufferedReader( inFile ,inEncode  );
			PrintWriter    writer = FileUtil.getPrintWriter( tempFile  ,outEncode );

			try {
				String line1;
				while((line1 = reader.readLine()) != null) {
					writer.println( line1 );
				}
			}
			catch( IOException ex ) {
				String errMsg = "ファイルコピー中に例外が発生しました。[" + data.getRowNo() + "]件目" + CR
							+ " inFile=[" + inFile + "] , tempFile=[" + tempFile + "]" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg,ex );
			}
			finally {
				Closer.ioClose( reader ) ;
				Closer.ioClose( writer ) ;
			}
		}

		if( isEquals ) {
			if( !outFile.delete() ) {
				String errMsg = "所定のファイルを削除できませんでした。[" + outFile + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg );
			}

			if( !tempFile.renameTo( outFile ) ) {
				String errMsg = "所定のファイルをリネームできませんでした。[" + tempFile + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg );
			}
		}

	 	// 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
		if( keepTimeStamp ) {
			if( !outFile.setLastModified( inFile.lastModified() ) ) {
				String errMsg = "lastModified 時間の設定が、できませんでした。[" + outFile + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg );
			}
		}

		if( display ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
		return data ;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Copy Count : " + inCount   + CR
				+ TAB + "inPath     : " + inPath    + CR
				+ TAB + "inEncode   : " + inEncode  + CR
				+ TAB + "outPath    : " + outPath   + CR
				+ TAB + "outEncode  : " + outEncode + CR
				+ TAB + "binary     : " + binary ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_FileCopy は、上流から受け取った FileLineModelを処理する、"				).append( CR );
		buf.append( "ChainProcess インターフェースの実装クラスです。"								).append( CR );
		buf.append( CR );
		buf.append( "上流から受け取った FileLineModel の ファイルから、inPath の共通パス" 			).append( CR );
		buf.append( "以下のファイルを、outPath の共通パス以下にコピーします。" 						).append( CR );
		buf.append( "コピーの種類は、バイナリか、テキストで、テキストの場合は、エンコード" 			).append( CR );
		buf.append( "変換も行うことが可能です。" 													).append( CR );
		buf.append( "inPath と outPath が同じ、または、outPath が未設定の場合は、入力と出力が" 		).append( CR );
		buf.append( "同じですので、自分自身のエンコード変換処理を行うことになります。" 				).append( CR );
		buf.append( CR );
		buf.append( "コピーされるファイルのファイル名は、入力ファイル名と同一です。保存される" 		).append( CR );
		buf.append( "フォルダが異なります。(同一にすることも可能です。)" 							).append( CR );
		buf.append( CR );
		buf.append( "上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト" 	).append( CR );
		buf.append( "である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを" 		).append( CR );
		buf.append( "使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し" 	).append( CR );
		buf.append( "できれば、使用可能です。" 														).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。" 	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に" 		).append( CR );
		buf.append( "繋げてください。" 																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_FileCopy().usage() );
	}
}
