/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.opengion.fukurou.util.Options;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * フォームの入力欄などで入力候補となるデータリストを定義するHTML拡張タグです。
 * HTML5 から、新たに追加された要素です。
 *
 * データリスト内の選択肢は、optionタグ、queryOptionタグによって指定します。
 * データリスト の id 属性は、フォームの list 属性と同じキーを指定する事で関連付けします。
 *
 * @og.formSample
 * ●形式：&lt;og:datalist id="…" /&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:datalist
 *       id               ○【TAG】入力候補を表示するフォームの list 属性に設定する id (必須)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:datalist&gt;
 *
 * ●使用例
 * &lt;og:input type="text" name="yourarea" autocomplete="on" list="tokyo" /&gt;
 *
 *  &lt;og:datalist id="tokyo" &gt;
 *      &lt;og:option value="渋谷" /&gt;
 *      &lt;og:option value="新宿" /&gt;
 *      &lt;og:option value="池袋" /&gt;
 *  &lt;/og:datalist&gt;&lt;
 * </pre>
 *
 * @og.group 【HTML5】選択データ制御
 * @og.rev 5.7.1.0 (2013/12/06) 新規追加
 *
 * @version  6.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DatalistTag extends CommonTagSupport implements OptionAncestorIF {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.1.0 (2013/12/06)" ;

	private static final long serialVersionUID = 571020131206L ;

	private transient Options option	= new Options();

	private String id	= null;		// フォームと関連付けるid

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		if( useTag() ) {
			return( EVAL_BODY_INCLUDE );	// Body インクルード( extends TagSupport 時)
		}
		return ( SKIP_BODY );				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			// display:none は、datalist の optionのBODY部が、HTML5 以外では表示されてしまうため。
			rtn.append("<div style=\"display:none;\">" );
			rtn.append("<datalist id=\"").append( id ).append( "\" >" );
			rtn.append( option.getOption() );
			rtn.append( "</datalist>" );
			rtn.append( "</div>" );

			jspPrint( rtn.toString() );
		}
		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		option	= new Options();
		id		= null;
	}

	/**
	 * データリストの選択項目を追加します。
	 *
	 * datalist タグのBODY要素の OptionTag よりアクセスされます。
	 *
	 * @param	opt 	 オプションタグ文字列
	 */
	public void addOption( final String opt ) {
		option.add( opt );
	}

	/**
	 * 【HTML】要素に対して固有の名前(id)をつける場合に設定します。
	 *
	 * @og.tag
	 * データリスト の id 属性は、フォームの list 属性と同じキーを指定する事で関連付けします。
	 *
	 * @param   id 固有の名前
	 */
	@Override
	public void setId( final String id ) {
		this.id = nval( getRequestParameter( id ), null );
	}

	/**
	 * 値を外部から取り出します。
	 *
	 * OptionTag で、value を取り出して、内部の値と同じ場合は、選択状態にします。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 *
	 * @return	内部に設定された値
	 */
	public String getValue() {
		// ここでは、何もしません。
		return null;
	}

	/**
	 * 複数選択可能時に全選択を設定するかどうかを返します。
	 *
	 * これは、上位入れ子のタグの OptionTag で、multipleAll を取り出して、
	 * true であれば、全選択に設定します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規作成
	 *
	 * @return	全選択:true / 通常:false
	 */
	public boolean isMultipleAll() {
		// ここでは、何もしません。
		return false;
	}

	/**
	 * パラメーター変換({&#064;XXXX}の置き換えをしない状態のパラメーターをセットします。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成(動的プルダウン実装見直し)
	 * @og.rev 5.5.4.0 (2012/07/02) 予約語対応
	 *
	 * @param   param パラメーター
	 */
	public void setRawParam( final String param ) {
		// ここでは、何もしません。
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		option = new Options();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "id"			,id				)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
