/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.HybsFileFilter;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * fileOption でファイルを検索する場合の条件を指定するタグです。
 *
 * @og.formSample
 * ●形式：&lt;og:fileWhere name="…" startsWith="…" endsWith="…" … /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:fileWhere
 *       startsWith         【TAG】指定された接頭辞で始まる場合、スルー(選択)されます
 *       endsWith           【TAG】指定された接頭辞で終わる場合、スルー(選択)されます
 *       instr              【TAG】指定された文字列がファイル名に含まれる場合、スルー(選択)されます
 *       equals             【TAG】ファイル名が一致する場合、スルー(選択)されます
 *       matches            【TAG】ファイル名が、指定された正規表現と一致する場合、スルー(選択)されます
 *       unMatches          【TAG】ファイル名が、指定された正規表現と一致しない場合、スルー(選択)されます
 *       lastModified       【TAG】指定のタイムスタンプ以後に変更されている場合、スルー(選択)されます
 *       isLarger           【TAG】指定の大きさより大きいファイルの場合、スルー(選択)されます
 *       isSmaller          【TAG】指定の大きさより小さいファイルの場合、スルー(選択)されます
 *       useDir             【TAG】フィルタによる対象の絞込みをディレクトリにも適用するかどうかを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:fileOption select="NAME" from="./jsp" &gt;
 *         &lt;og:fileWhere
 *             startsWith = "File"          :指定された接頭辞で始まる場合、選択される
 *             endsWith   = ".java"         :指定された接頭辞で終わる場合、選択される
 *             instr      = "Tag"           :指定された文字列がファイル名に含まれる場合、選択される
 *             fileEquals = "FileWhereTag.java" :ファイル名が一致する場合、選択される(大文字小文字は区別しない)
 *             matches    = "File*Tag"      :ファイル名が、指定された正規表現と一致する場合、選択される(大文字小文字は区別しない)
 *             unMatches  = "File*Tag"      :ファイル名が、指定された正規表現と一致しない場合、選択される(大文字小文字は区別しない)
 *             lastModified = "20050101"    :指定のタイムスタンプ以後に変更された場合、選択される
 *                                           日付けの指定は、YYYYMMDD 形式,TODAY , YESTERDAY が使用できます。
 *             isLarger   = "1024"          :指定の大きさ(バイト単位)より大きいファイルの場合、選択される
 *             isSmaller  = "1024"          :指定の大きさ(バイト単位)より小さいファイルの場合、選択される
 *             useDir     = "false"         :フィルタによる対象の絞込みをディレクトリにも適用するかどうか
 *         /&gt;
 *     &lt;/og:fileOption&gt;
 *
 *       ・検索条件が入力された時
 *         条件を満足して全てのファイルを列挙します。
 *
 *       ・検索条件が入力されなかった時
 *         該当ディレクトリ内の全てのファイルを列挙します。
 *
 * @og.rev 2.1.1.0 (2002/11/11) 新規作成
 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジックの全面見直し
 * @og.group その他入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileWhereTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.0 (2013/07/05)" ;

	private static final long serialVersionUID = 566020130705L ;

	private String	startsWith		= null;		// 指定された接頭辞で始まる場合、選択される
	private String	endsWith		= null;		// 指定された接頭辞で終わる場合、選択される
	private String	instr			= null;		// 指定された文字列がファイル名に含まれる場合、選択される
	private String	fileEquals		= null; 	// ファイル名が一致する場合、選択される(大文字小文字は区別しない)
	private String	matches			= null;		// ファイル名が、指定された正規表現と一致する場合、選択される(大文字小文字は区別しない)
	private String	unMatches		= null;		// ファイル名が、指定された正規表現と一致しない場合、選択される(大文字小文字は区別しない)
	private String	lastModified	= null;		// 指定のタイムスタンプ以後に変更された場合、選択される
	private int		isLarger		= -1;		// 指定の大きさ(バイト単位)より大きいファイルの場合、選択される
	private int		isSmaller		= -1;		// 指定の大きさ(バイト単位)より小さいファイルの場合、選択される
	private boolean	useDir			= false;	// フィルタによる対象の絞込みをディレクトリにも適用するかどうか 5.3.9.0 (2011/09/01)

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.2.0 (2010/01/01) unMatches メソッドで反転属性の追加
	 * @og.rev 5.3.9.0 (2011/09/01) useDir属性を追加
	 * @og.rev 5.6.6.0 (2013/07/05) unMatchesの引数間違いの訂正
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 5.3.9.0 (2011/09/01) useDir属性を追加
//		HybsFileFilter filter = new HybsFileFilter();
		HybsFileFilter filter = new HybsFileFilter( useDir );
		filter.startsWith(	 startsWith	  );
		filter.endsWith(	 endsWith	  );
		filter.instr(		 instr		  );
		filter.fileEquals(	 fileEquals	  );
		filter.matches(		 matches	  );
//		filter.matches(		 matches   , true );		// 5.1.2.0 (2010/01/01) 反転属性の追加
		filter.matches(		 unMatches , true );		// 5.6.6.0 (2013/07/05) 引数間違いの訂正
//		filter.unMatches(	 unMatches	  );
		filter.lastModified( lastModified );
		filter.isLarger(	 isLarger	  );
		filter.isSmaller(	 isSmaller	  );

		// ファイルフィルタを追加
		FileQueryTag fileQuery = (FileQueryTag)findAncestorWithClass( this, FileQueryTag.class );
		if( fileQuery != null ) {
			fileQuery.setFileFilter( filter );
			return(EVAL_PAGE);
		}

		FileOptionTag fileOption = (FileOptionTag)findAncestorWithClass( this, FileOptionTag.class );
		if( fileOption != null ) {
			fileOption.setFileFilter( filter );
			return(EVAL_PAGE);
		}

		// どちらにも属していなかった場合は、エラーになる。
		String errMsg = "この、FileWhereTag タグは、FileQueryTag か、FileOption の"
					+ "BODY 部にのみ、配置できます。(それ以外の箇所での使用はできません。)" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジックの全面見直し
	 * @og.rev 5.3.9.0 (2011/09/01) useDir属性追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		startsWith		= null;
		endsWith		= null;
		instr			= null;
		fileEquals		= null;
		matches			= null;
		unMatches		= null;
		lastModified	= null;
		isLarger		= -1;
		isSmaller		= -1;
		useDir			= false; // 5.3.9.0 (2011/09/01) useDir属性追加
	}

	/**
	 * 【TAG】指定された接頭辞で始まる場合、スルー(選択)されます。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    prefix 接頭辞
	 * @see java.lang.String#startsWith(String)
	 */
	public void setStartsWith( final String prefix ) {
		startsWith = nval( getRequestParameter( prefix ),startsWith );
	}

	/**
	 * 【TAG】指定された接頭辞で終わる場合、スルー(選択)されます。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    suffix 接尾辞
	 * @see java.lang.String#endsWith(String)
	 */
	public void setEndsWith( final String suffix ) {
		endsWith = nval( getRequestParameter( suffix ),endsWith );
	}

	/**
	 * 【TAG】指定された文字列がファイル名に含まれる場合、スルー(選択)されます。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str 指定の部分文字列
	 */
	public void setInstr( final String str ) {
		instr = nval( getRequestParameter( str ),instr );
	}

	/**
	 * 【TAG】ファイル名が一致する場合、スルー(選択)されます。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列
	 * @see java.lang.String#equals(Object)
	 */
	public void setEquals( final String str ) {
		fileEquals = nval( getRequestParameter( str ),fileEquals );
	}

	/**
	 * 【TAG】ファイル名が、指定された正規表現と一致する場合、スルー(選択)されます。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == true と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列(正規表現)
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setMatches( final String str ) {
		matches = nval( getRequestParameter( str ),matches );
	}

	/**
	 * 【TAG】ファイル名が、指定された正規表現と一致しない場合、スルー(選択)されます。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == false と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列(正規表現) とマッチしない
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setUnMatches( final String str ) {
		unMatches = nval( getRequestParameter( str ),unMatches );
	}

	/**
	 * 【TAG】指定のタイムスタンプ以後に変更されている場合、スルー(選択)されます。
	 * @og.tag
	 * ディレクトリは、ここの判定では無視します。(必ず true を返します)
	 * 日付けの指定に、YYYYMMDD 形式の ８文字数字文字列以外に、
	 * TODAY や YESTERDAY なども使用できます。
	 * TODAY は、実行日の 00:00:00 を基準時刻とし、YESTERDAY は、その前日になります。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    modify 時刻を表す long 値(ミリ秒単位)
	 */
	public void setLastModified( final String modify ) {
		lastModified = nval( getRequestParameter( modify ),lastModified );
	}

	/**
	 * 【TAG】指定の大きさより大きいファイルの場合、スルー(選択)されます。
	 * @og.tag ファイルの大きさ(バイト単位)は同値を含みます。
	 * 引数が 0以下(マイナス) の場合は、追加しません。
	 *
	 * @param	len 	ファイルの大きさ(バイト単位)。同値を含む。
	 */
	public void setIsLarger( final String len ) {
		isLarger = nval( getRequestParameter( len ),isLarger );
	}

	/**
	 * 【TAG】指定の大きさより小さいファイルの場合、スルー(選択)されます。
	 * @og.tag ファイルの大きさ(バイト単位)は同値を含みません。
	 * 引数が 0以下(マイナス) の場合は、追加しません。
	 *
	 * @param	len	ファイルの大きさ(バイト単位)。同値を含まない。
	 */
	public void setIsSmaller( final String len ) {
		isSmaller = nval( getRequestParameter( len ),isSmaller );
	}

	/**
	 * 【TAG】フィルタによる対象の絞込みをディレクトリにも適用するかどうかを指定します。
	 * @og.tag 初期値は、false(ディレクトリはフィルタによる絞込みをしない)です。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 *
	 * @param	flg	フィルタによる対象の絞込みをディレクトリにも適用するかどうか(初期値:false)
	 */
	public void setUseDir( final String flg ) {
		useDir = nval( getRequestParameter( flg ),useDir );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return String型。このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "startsWith"		,startsWith		)
				.println( "endsWith"		,endsWith		)
				.println( "instr"			,instr			)
				.println( "fileEquals"		,fileEquals		)
				.println( "matches"			,matches		)
				.println( "unMatches"		,unMatches		)
				.println( "lastModified"	,lastModified	)
				.println( "isLarger"		,isLarger		)
				.println( "isSmaller"		,isSmaller		)
				.println( "useDir"			,useDir		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
