/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.html;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.io.Writer;
import java.io.IOException;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * DBTableModelの表示を行うインターフェースです。
 * 各種の表示形式をサポートするには、ViewForm インターフェースを implements した
 * クラスを作成して、DBTableModel をセットするだけです。
 * ViewForm 自体は,Controller クラスよりアクセスされます。
 *
 * ViewForm の実装クラス( implements されたクラス)に対する 各種設定(たとえば、HTMLタグなど)
 * は、個々の実装クラス毎に設定します。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface ViewForm {

	/**
	 * カラムの書込み許可属性(clmWritable)の初期値を指定します。
	 * これが true の場合は,デフォルトですべてのカラムが書込み許可されます。
	 * false の場合は,すべてが書込み不許可になります。
	 *
	 * このデフォルト値は、false に設定されています。
	 *
	 */
	static final boolean DEFAULT_CLM_WRITABLE = false;

	/**
	 * 書き込み可能な行(rowWritable == true)のみを表示対象とするかどうかを
	 * 指定します。
	 *
	 */
	static final boolean DEFAULT_SKIP = false;

	/**
	 * 初期化します。
	 * ここでは、内部で使用されているキャッシュをクリアし、
	 * 新しいモデル(DBTableModel)と言語(lang) を元に内部データを再構築します。
	 *
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 *
	 * @param	table DBTableModelオブジェクト
	 */
	void init( DBTableModel table ) ;

	/**
	 * DBTableModel から View文字列を作成して返します。
	 *
	 * @return	DBTableModelから作成された View文字列
	 */
	String create() ;

	/**
	 * DBTableModel から View文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return	DBTableModelから作成された View文字列
	 */
	String create( int startNo, int pageSize ) ;

	/**
	 * DBTableModel から View文字列を作成して、Writer にセットします。
	 * 処理内容は、create( int , int ) と同じですが、中間の文字列(StringBuilder)
	 * を作成せずに、直接、Writer に書き出します。
	 * よって、データ作成途中でエラーが発生しても、すでにいくつかのデータは
	 * クライアントに返されています。
	 *
	 * @og.rev 5.0.0.1 (2009/08/15) 直接出力用の Writer 引数追加
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 * @param  wrt		  直接登録用の Writer
	 * @throws IOException 入出力エラーが発生したとき
	 */
	void create( int startNo, int pageSize, Writer wrt ) throws IOException ;

	/**
	 * 表示開始位置をセットします。
	 *
	 * @param  startNo	  表示開始位置
	 */
	void setStartNo( int startNo ) ;

	/**
	 * 表示開始位置を返します。
	 *
	 * @return	表示開始位置
	 */
	int getStartNo() ;

	/**
	 * 表示件数をセットします。
	 *
	 * @param  pageSize   表示件数
	 */
	void setPageSize( int pageSize ) ;

	/**
	 * 表示件数を返します。
	 *
	 * @return	pageSize	表示件数
	 */
	int getPageSize() ;

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return	DBTableModelオブジェクト
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * ViewForm の識別IDをセットします。
	 * これは、ViewFormFactory でプールする場合の識別キーになります。
	 * プールに戻すときに自分自身に この識別IDを使用します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) メソッド名がまぎらわしい為、変更します。
	 *
	 * @param	id 識別ID
	 */
	void setId( String id ) ;

	/**
	 * ViewForm の識別IDを返します。
	 * これは、ViewFormFactory でプールする場合の識別キーになります。
	 * プールに戻すときに自分自身に この識別IDを使用します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) メソッド名がまぎらわしい為、変更します。
	 *
	 * @return	識別ID
	 */
	String getId() ;

	/**
	 * 内容をクリア(初期化)します。
	 *
	 */
	void clear() ;

	/**
	 * カラムが書き込み可能かどうかをセットします。
	 *
	 * @param	column カラム番号
	 * @param	rw 書込み可能(true)／不可能(false)
	 */
	void setColumnWritable( int column,boolean rw ) ;

	/**
	 * 書き込み可能カラム名を、カンマ区切りで与えます。
	 * これは、書き込み不可カラム名の指定(noWritable)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * setColumnWritable( int column,boolean rw ) の簡易版です。
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @param	columnName	カラム名
	 */
	void setColumnWritable( String columnName ) ;

	/**
	 * 書き込み不可カラム名を、カンマ区切りで与えます。
	 * これは、書き込み可能カラム名の指定(columnWritable)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @param	columnName	カラム名
	 */
	void setNoWritable( String columnName ) ;

	/**
	 * その行が、処理対象かどうかをチェックします。
	 * 処理対象かどうかは、書き込み可能な行(rowWritable == true)
	 * で且つ チェックされた行(rowChecked == true) とします。
	 * 例えば,NEXT/PREVでスキップ等の処理を行う場合は､ 処理対象以外を
	 * スキップすることで実現できます。
	 *
	 * @param	row	行番号
	 *
	 * @return	 処理対象(true)／処理対象でない(false)
	 */
	boolean isMarked( int row ) ;

	/**
	 * カラムが表示可能かどうかをセットします。
	 * これは、表示不可カラム名の指定(noDisplay)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @param	column カラム番号
	 * @param	rw 表示可能(true)／不可能(false)
	 */
	void setColumnDisplay( int column,boolean rw ) ;

	/**
	 * 表示可能カラム名を、カンマ区切りで与えます。
	 * これは、表示不可カラム名の指定(noDisplay)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * setColumnDisplay( int column,boolean rw ) の簡易版です。
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @param	columnName	カラム名
	 */
	void setColumnDisplay( String columnName ) ;

	/**
	 * 表示不可カラム名を、カンマ区切りで与えます。
	 * これは、表示可能カラム名の指定(columnDisplay)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * null を与えた場合は,なにもしません。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 *
	 * @param	columnName	カラム名
	 */
	void setNoDisplay( String columnName ) ;

	/**
	 * 同一表示データをまとめる(表示しない)カラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムのグループ化指定を行います。
	 * 同じカラム番号で一つ上の行番号と同じ内容の場合に、特殊な表示方法で表示します。
	 * 対応は、表示(Renderer)時のみとします。
	 * 特殊な表示方法の指定は、groupClass 属性で指定します。
	 * (例：groupClass にアルファベット以外の文字・・そのまま表示、アルファベットはクラス属性を付与します。)
	 * カラム単位なので、新しいゼブラによる色分けは行いません。(任意のカラムに適用できる為)
	 * また、ファイル出力などのデータ自身は変更されていませんので、そのままデータが落ちます。
	 *
	 * これは、同一表示データをまとめないカラム名の指定(noGroup)と同時にセットする
	 * ことは出来ません。
	 * また、全カラムについて、有効にする場合は、group="*" を設定します。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 *
	 * @param	group まとめるカラム名(CSV形式)
	 */
	void setColumnGroup( String group ) ;

	/**
	 * 同一表示データをまとめないカラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムのグループ化指定で、同一でもまとめないカラムを指定するほうが簡単な場合に
	 * 使用します。例えば、キー情報以外は、グループ化指定したい場合などに便利です。
	 * 動作は、columnGroup の動きと同じです。(まとめないカラムを指定するだけです。)
	 *
	 * これは、同一表示データをまとめるカラム名の指定(columnGroup)と同時にセットする
	 * ことは出来ません。
	 * また、全カラムについて、有効にする場合は、group="*" を設定します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param	group	例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 */
	void setNoGroup( final String group );

	/**
	 * 同一表示データをまとめる場合の、表示方法を指定します。
	 *
	 * これは、カラムのグループ化指定(columnGroup)を行う場合の、まとめ表示の方法を指定します。
	 * なにも指定しない場合は、まとめカラムは、表示しない(ゼロ文字列)になります。
	 * その場合、先の行と同じ場合も、変更されたカラムが、NULL(ゼロ文字列)であった場合も、
	 * 同じ用に表示されるため、区別が付きません。
	 * そこで、前の行と同じデータの場合に、特殊な処理を行うことで、区別できるようにします。
	 * 指定方法が特殊なので、注意が必要です。
	 *   記号の場合：
	 *       記号(-- , → , ↓ , * など)が指定された場合は、そのままの文字に置き換えられます。
	 *   アルファベットの場合：
	 *       アルファベット(a-z,A-Z)の場合は、&lt;span class="アルファベット"&gt;元の文字&lt;/span&gt;
	 * 例えば、バックカラーに色を付ける、文字を薄い灰色にする、などできます。
	 * ただし、データ量が圧倒的に増えるため、大量のデータ等で使用するのは避けたほうが良いと思います。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 *
	 * @param	grpCls まとめ表示の方法
	 */
	void setGroupClass( final String grpCls );

	/**
	 * カラム名リンクソートを表示するカラム名を、カンマ区切りで与えます。
	 *
	 * ヘッダーにソート用リンクを作成する useTableSorter 属性 に対して、
	 * カラム個別に作成する場合のカラム名をカンマ区切り文字で指定します。
	 * この tableSorterKeys 属性は、useTableSorter 属性 と無関係に、指定した
	 * カラムのみ、リンクを表示します。
	 * また、全カラムについて、有効にする場合は、columnName="*" を設定します。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 *
	 * @param	columnName	カラム名
	 */
	void setTableSorterKeys( String columnName ) ;

	/**
	 * 表示時の選択用オブジェクトのタイプを指定します。
	 * ・複数選択可能時は "checkbox" を指定します。
	 * ・一つだけ選ばせる場合は, "radio" を指定します。
	 * ・隠しフィールドで全件を選ぶ場合は、"hidden" を指定します。
	 * 初期値は、"checkbox" です。
	 *
	 * @param	type 選択用オブジェクトのタイプ( "checkbox"/"radio"/"hidden"/"null" )
	 */
	void setSelectedType( String type ) ;

	/**
	 * 表示時の行番号の表示方法を指定します。
	 * ・sequenceは、１から始まる連番です。
	 * ・none を指定すると、番号は表示されません。
	 * ・delete を指定すると、行番号表示そのもののタグを取り除きます。(3.5.5.0 追加)
	 * ・skip=カラム名を指定すると、そのカラムの値が NULL の場合は、番号をスキップします。
	 * ・view=カラム名を指定すると、そのカラムの値が、使用されます。(ラベルは付きません)
	 * skip=XX と、view=XX は、=の前後にスペースを入れないで下さい。
	 * 初期値は、"sequence" です。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 *
	 * @param	type 行番号の表示方法のタイプ( /sequence/none/delete/skip=カラム名/view=カラム名/ )
	 */
	void setNumberType( String type ) ;

	/**
	 * テーブル等のチェックボックスに属性を付加します
	 * JavaScript などの HTML基本タグ以外の属性を、そのまま
	 * チェックボックス/ラジオボタン等に使用します。
	 *
	 * @param	option オプション属性
	 */
	void setOptionTypeAttributes( String option ) ;

	/**
	 * リソースマネージャを設定します。
	 * クロス集計時に、useColumnHeader="true"とした場合のみ設定されます。
	 *
	 * @og.rev 4.0.0.0 新規作成
	 *
	 * @param	res リソースマネージャー
	 */
	void setResourceManager( ResourceManager res ) ;

	//////////////////////////////////////////////////////////////////////////
	//
	//	 フォーマット関連クラスのメソッド
	//
	//////////////////////////////////////////////////////////////////////////

	/**
	 * フォーマットを設定します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 *
	 * @param list TableFormatterのリスト
	 */
	void setFormatterList( List<TableFormatter> list ) ;		// 4.3.3.6 (2008/11/15) Generics警告対応

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return	フォーマットメソッドを使用できるか
	 */
	boolean canUseFormat() ;

	/**
	 * ビューフォームのタイプを返します。
	 * これは、ViewFormFactory で、org.opengion.hayabusa.html. + 『type』 + ViewForm
	 * で作成されるサブクラスの 『type』 部分を返します。
	 *
	 * @return	ビューフォームのタイプ
	 */
	String getViewFormType() ;

	/**
	 * ビューリンクオブジェクトを設定します。
	 * これは、ViewLink タグで作成された キー(カラム名)とリンク文字列を
	 * 持っている Attributes オブジェクトを受け取り、内部でリンク表示に
	 * 使用します。
	 *
	 * @param	link	ビューリンクオブジェクト
	 */
	void setViewLink( ViewMarker link );

	/**
	 * ビューマーカーオブジェクトを設定します。
	 * これは、ViewMarker タグで作成された キー(カラム名)とマーカー文字列を
	 * 持っている Attributes オブジェクトを受け取り、内部でマーカー表示に
	 * 使用します。
	 *
	 * @param	link	ビューマーカーオブジェクト
	 */
	void setViewMarker( ViewMarker link );

	/**
	 * エディットマーカーオブジェクトを設定します。
	 * これは、ViewMarker タグで、isRendere="false" で作成された エディットキー(カラム名)と
	 * マーカー文字列を持っている Attributes オブジェクトを受け取り、内部でマーカー表示に
	 * 使用します。
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param	marker	ビューマーカーオブジェクト
	 */
	void setEditMarker( ViewMarker marker ) ;

	/**
	 * 検索結果メッセージを表示する/しないを設定します
	 * 初期値は、表示する(false)です。
	 *
	 * @param	noMessage [true:表示しない/false:表示する]
	 */
	void setNoMessage( boolean noMessage ) ;

	/**
	 * 先頭へ戻るリンク間隔を返します。
	 *
	 * @return	backLinkCount	 リンク間隔
	 */
	int getBackLinkCount() ;

	/**
	 * 先頭へ戻るリンク間隔をセットします。
	 *
	 * @param  no	リンク間隔
	 */
	void setBackLinkCount( int no ) ;

	/**
	 * ヘッダーを出力する間隔を取得します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @return	ヘッダーの表示間隔 ( 0:通常ヘッダ、n:n回ごとに現れる )
	 */
	int getHeaderSkipCount() ;

	/**
	 * ヘッダーを出力する間隔をセットします。
	 *
	 * 0を指定すると、繰り返しません(つまり、ヘッダーを最初に表示するだけです。)
	 * 数字を指定すると、その回数毎に、ヘッダーをデータの行に挿入します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @param	hsc ヘッダーの表示間隔 ( 0:通常ヘッダ、n:n回ごとに現れる )
	 */
	void setHeaderSkipCount( int hsc ) ;

	/**
	 * チェックの入った行のみを表示させるかどうか指定します。
	 *
	 * "true" で、チェックの入った行のみを表示させます。
	 * 従来は、TextField系のViewに対して、NEXT,PREVでチェックの
	 * 入った行のみを表示させる機能でしたが、Table系のViewに対しても、
	 * 同様に機能するように、しました。
	 * 初期値は、ViewForm.DEFAULT_SKIP です。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 新規追加
	 *
	 * @param	flag チェックの入った行のみを表示させるかどうか(true:のみ表示/false:前件表示)
	 */
	void setSkip( boolean flag ) ;

	/**
	 * チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを指定します。
	 *
	 * 1 で、コントロール用のチェックボックスが現れて、この機能を使用することができるようになります。
	 * 0 は、従来どおりです。
	 * 使用するにあたり、jsp/commpn/default.js にJavaScriptを設定しておきます。(設定済み)
	 * 初期値は、システムパラメータ の VIEW_USE_CHECK_CONTROL です。(0:使用しない)
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール変更( boolean ⇒ String )
	 *
	 * @param	flag チェックボックスの全チェックを選択する機能を使用するかどうか
	 *			(1:使用する/0:使用しない/2:初期値チェック済み)
	 */
	void setUseCheckControl( int flag ) ;

	/**
	 * ヘッダーにソート用リンクを作成するかどうかを指定します。
	 *
	 * "true" で、ヘッダーにソート用リンクを作成します。
	 * false は、作成しません。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規追加
	 *
	 * @param	flag ヘッダーにソート用リンクを作成するかどうか(true:作成する/false:作成しない)
	 */
	void setUseTableSorter( boolean flag ) ;

	/**
	 * ヘッダーにソート用リンクを作成する時の、リクエスト引数のMapを設定します。
	 *
	 * ソート用URLに、リクエスト時の引数を設定する必要があります。
	 * そのため、リクエスト時のキーと値のセットをMapで指定します。
	 * このMap は、リクエストキャッシュより取り出します。
	 * 実装では、このMapを元に、URLのQuery部文字列を作成します。処理過程において、
	 * このMap を書き換えることは行いません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 *
	 * @param	map ヘッダーソート時のリンクに追加するリクエスト変数のキャッシュMap
	 */
	void makeTableSorterQuery( Map<?,?> map ) ;		// 4.3.3.6 (2008/11/15) Generics警告対応

	/**
	 * ViewForm のサブクラスに渡すパラメータマップ
	 *
 	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 *
	 * @param	map パラメータマップ
	 */
	void setParam( Map<String,String> map ) ;	// 4.3.3.6 (2008/11/15) Generics警告対応
	
	/**
	 * ViewForm のサブクラスに渡すパラメータ(文字列配列のArrayList)を設定します。
	 *
 	 * @og.rev 5.5.8.3 (2011/11/17) Object受け渡しのために追加
	 *
	 * @param	list 文字列配列のArrayList
	 */
//	void setViewObject( Map<String,Object> map ) ;
	void setViewArrayList( ArrayList<String[]> list) ;

	/**
	 * マーカーオブジェクト(リンク、マーカー)を設定します。
	 * ここでは、旧 ViewForm 属性を 新ViewForm に直接セットします。
	 *
	 * @og.rev 3.5.6.1 (2004/06/25) 新規追加
	 *
	 * @param	view ViewFormオブジェクト
	 */
	void markerSet( ViewForm view ) ;

	/**
	 * テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします。
	 * 0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 * 初期値は、1(ゼブラ)です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規追加
	 *
	 * @param  sycle  0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 */
	void setBgColorCycle( int sycle ) ;

	/**
	 * テーブルのバックグラウンドカラーのクラスに指定するカラム名を指定します。
	 *
	 * 通常のゼブラ模様は、tr 属性のクラス指定として、row_0、row_1 が指定されます。
	 * その代わりに、ここで指定されたカラムの値がクラス属性として設定されます。
	 * (指定するのはカラム名です。行単位にクラス属性の値を変えることが可能です。)
	 * 選択行(row_sel)は、優先して使用されます。
	 * 出力されるクラス名は、"row_" + 属性値 になります。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param  clsClm ゼブラ模様の替わりに指定するクラスを格納したカラム名
	 */
	void setBgColorClsClm( final String clsClm ) ;

	/**
	 * スクロールバー用のDIV要素を出力するかどうか(初期値は、false)
	 *
	 * スクロールバー対応する為、テーブルの先頭に、DIV要素を出力します。
	 * 初期値は、システムパラメータ の VIEW_USE_SCROLLBAR です。
	 * ※ 互換性の関係より、false になっています。
	 * ※ 互換性の関係より、新しいタイプのヘッダー固定を、"TYPE2" とします。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) 新規追加
	 * @og.rev 3.8.0.3 (2005/07/15) barType 変数の追加
	 * @og.rev 4.0.0.0 (2007/04/10) ヘッダー固定のスクロールタイプは、TYPE2 のみにする。
	 *
	 * @param	useBar	スクロールバー用のDIV要素を出力 [true:出力する/false:出力しない]
	 */
//	void setUseScrollBar( boolean useBar , String barType );
	void setUseScrollBar( boolean useBar );

	/**
	 * 指定の行番号まで画面をスクロールさせる場合の行番号を設定します。
	 *
	 * 画面をスクロール後、リンク等で他画面を表示後、戻ってきた場合に、
	 * 先のスクロール位置まで戻します。
	 * ただし、厳密に戻すことはできないため、大体のあたりに戻します。
	 * useSelRowColor は、選択行に色づけするかどうかを指定します。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規追加
	 * @og.rev 3.7.1.1 (2005/05/31) 選択行マーカーの使用有無
	 *
	 * @param  rowNo  指定の行番号まで画面をスクロールさせる場合の行番号
	 * @param	useSelRowColor	指定の行番号(選択行)の色を変更するかどうか
	 */
	void setScrollRowNo( int rowNo, boolean useSelRowColor ) ;

	/**
	 * 設定値に "_" が含まれている場合にレンデラーを使用するカラムをCSV形式で指定します。
	 *
	 * これは、従来の カラム定義の WRITABLE エディターと同等の働きを行うように
	 * カラム属性を指定します。
	 * WRITABLE エディターは、設定値にアンダーバー "_" が含まれている場合に、
	 * その値を書込み禁止にする機能です。これは、エディター自身が値を判断して
	 * 書き込み許可か禁止かを判断しています。
	 * この動きを汎用的にするため、指定のカラムをカンマ区切り文字(CSV)で指定
	 * することにより、レンデラーとエディターを設定値によって動的に切り替える
	 * 機能を実現します。
	 * その場合、表示/編集ともに、先頭のアンダーバーは削除されます。
	 * また、全カラムについて、有効にする場合は、writableControl="*" を設定します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 *
	 * @param  wrtCtrl 書き込み制御を行いたいカラムをCSV形式で指定
	 */
	void setWritableControl( String wrtCtrl ) ;

	/**
	 * ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  rtnKeys ogPopupで値を返すカラム文字列(CSV形式)
	 */
	void setPopupReturnKeys( String rtnKeys ) ;

	/**
	 * ビューの高さを指定します。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param  height 単位付き高さ
	 */
	void setHeight( String height ) ;

	/**
	 * ビューの幅を指定します。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param  width 単位付き高さ
	 */
	void setWidth( String width ) ;

	/**
	 * table要素に対して class 属性を設定します。
	 *
	 * 従来の システムリソースでのテーブルレイアウトの設定を廃止し、
	 * CSSファイルで、指定するように変更しています。
	 * これに伴い、CSSファイルのキーとして、クラス属性を出力します。
	 * view(または、出力されるtableタグ)のレイアウトは、このクラス属性で
	 * 指定することが可能になります。
	 * 初期値は、viewTable です。
	 *
	 * @og.rev 4.0.0.0 (2007/04/16) 新規追加
	 *
	 * @param	cls	class属性を表す文字列
	 */
	void setTableClass( String cls ) ;

	/**
	 * 改廃Cのついていない行をスキップするかどうか指定します。
	 *
	 * "true" で、改廃Cのついた行をスキップします。
	 * 初期値はfalseです。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) 新規追加
	 *
	 * @param	flag 改廃Cのついていない行をスキップするかどうか(true:スキップする/false:スキップしない)
	 */
	void setSkipNoEdit( boolean flag ) ;

	/**
	 * 画面遷移なしモードに対応した形で処理を行うかを指定します。
	 *
	 * "true" で、画面遷移なしモードに対応します。
	 * 初期値はfalseです。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 *
	 * @param	flag 画面遷移なしモードに対応するかどうか(true:対応する/false:対応しない)
	 */
	void setNoTransition( boolean flag ) ;

	/**
	 * ビューで表示したカラムの一覧をカンマ区切りで返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	ビューで表示したカラムの一覧
	 */
	String getViewClms() ;

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか
	 */
	boolean isEditable() ;

	/**
	 * このViewFormが作成された画面IDをセットします。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @param gamenId 画面ID
	 */
	void setGamenId( final String gamenId );

	/**
	 * このViewFormが作成された画面IDを返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @return	画面ID
	 */
	String getGamenId();

	/**
	 * 一括入力カラムのアシスト機能を利用するカラム名を、カンマ区切りで与えます。
	 *
	 * これは、カラムの書き込み時に、一括入力可能なカラムをヘッダーに表示し、
	 * 一括登録できる機能を提供します。
	 * この機能は、jsp/common/bulkUpdate.js JavaScript とともに提供されます。
	 * IE については、クリップボードも利用できます。Ctrl-C と Ctrl-V でそれぞれ
	 * 一連のカラムの値の取り出しと書き込みが可能になります。
	 * "*" を指定すると、すべてのカラムを(columnBulkSet)指定したことになります。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	columnName	例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 */
	void setColumnBulkSet( final String columnName );
}
