/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対してのデータをファイルに書込みします。
 * 但し、書き込まれるデータについては、旧伝送システムの形式と互換性を持たせるため、
 * データの前30Byteに空白で埋め、さらに全体で標準では500Byteになるように行末にも空白埋めをします。
 * 500byte以外にしたい場合は、書き込みパラメータの第２引数に整数で指定してください。
 *
 * 書込みするファイル名は、実行対象で指定します。ファイル名は絶対パスで指定して下さい。
 * また、書込するテキストファイルのエンコードは書込パラメーターで指定することができます。
 * 指定しない場合、UTF-8が適用されます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_SAMCB implements TransferExec {

	// 書込ファイルオブジェクト
//	private File fileWrite = null;			// 5.5.2.4 (2012/05/16) ローカル変数化

	// 書込ファイルのエンコード
//	private String fileEncode = null;		// 5.5.2.4 (2012/05/16) ローカル変数化

	/**
	 * ファイルに書込みします。
	 *
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @og.rev 5.5.3.3 (2012/06/15) close処理
	 * @og.rev 5.8.1.1 (2014/11/14) パラメータで桁数指定可能にする
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		File fileWrite = new File( config.getExecObj() );
		
		String fileEncode = "UTF-8";
		// 5.8.1.1 (2014/11/14) fillByte追加
		int fillByte = 500;
		String execPrm = config.getExecPrm();
		if( execPrm != null && execPrm.length() > 0 ){
			String[] obj = StringUtil.csv2Array( execPrm, ' ' );
			fileEncode =  obj[0];
			if( obj.length > 1 ) {
				fillByte = Integer.parseInt( obj[1] );
			}
		}

//		String fileEncode = config.getExecPrm();
//		if( fileEncode == null || fileEncode.length() == 0 ) {
//			fileEncode = "UTF-8";
//		}

		PrintWriter writer = FileUtil.getPrintWriter( fileWrite,fileEncode );
		String line = null;
		for( String s : vals ) {
			// 前30Byteを空白埋め
			String preSpace = StringUtil.stringFill( "", 30, fileEncode );
			// 全体で500Byteになるように後ろに空白埋め
			// 500Byte以外も指定可能とする
//			line = StringUtil.stringFill( preSpace + s, 500, fileEncode );
			line = StringUtil.stringFill( preSpace + s, fillByte, fileEncode );
			writer.println( line );
		}
		writer.close(); // 5.5.3.3 (2012/06/15)
	}
}
