/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

// import org.opengion.hayabusa.common.HybsSystem;			// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.util.HybsConst.CR ;		// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FileInfo;					// 6.2.0.0 (2015/02/27)
import org.opengion.fukurou.util.StringUtil;				// 6.0.2.4 (2014/10/17)

import java.io.File;

/**
 * ファイルをサーバーにアップロードする場合に使用されるファイル管理クラスです。
 * HTML5 ファイルアップロードの複数選択（multiple）対応 に伴い、一つのクラスとして public化します。
 *
 * @og.group その他機能
 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class UploadedFile implements Comparable<UploadedFile> {

	/** バッファの初期容量を通常より多い目に設定します。  {@value}  */
	public static final int BUFFER_MIDDLE = 200;

//	/** システム依存の改行記号をセットします。	*/
//	public static final String CR = System.getProperty("line.separator");

	private File filename	;				// 現時点での置き換え後ファイル名

	private final String uniqKey	;		// アップロードされたファイル名(ユニークにしておきます)
	private final String dir		;
	private final String name		;
	private final String original	;
	private final String type		;

	/**
	 * アップロードファイルの管理オブジェクトを作成します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 *
	 * @param	uniqKey		ユニークキー(初期アップロードファイル名)
	 * @param	dir			ファイルを保管するフォルダ
	 * @param	name		ファイルアップロードされた時のname属性
	 * @param	original	ファイル名(オリジナル)
	 * @param	type	コンテントタイプ
	 */
	UploadedFile( final String uniqKey, final String dir, final String name, final String original, final String type ) {
		this.uniqKey	= uniqKey;		// 5.7.1.1 (2013/12/13) uniqKey を確定させる。
		this.dir		= dir;
		this.name		= name;
		this.original	= original;
		this.type		= type;
	}

	/**
	 * ファイルアップロードされた時のname属性を取得します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 *
	 * @return	ファイルアップロードされた時のname属性
	 */
	public String getName() {
		return name;
	}

	/**
	 * コンテントタイプを取得します。
	 *
	 * @return	コンテントタイプ
	 */
	public String getContentType() {
		return type;
	}

	/**
	 * ファイル名(置き換え後)を取得します。
	 *
	 * @og.rev 5.7.1.2 (2013/12/20) zip 対応で、Fileオブジェクトを返すようにします。
	 *
	 * @return	ファイル名(置き換え後)
	 */
	public File getUploadFile() {
		return filename;
	}

	/**
	 * ファイル名(置き換え後)の置き換えを実行します。
	 * useBackup = true にすると、dir の直下に、"_backup" フォルダを作成します。
	 * バックアップファイル名は、元のファイル名(拡張子含む) ＋ "_" + 現在時刻のlong値 + "." + 元のファイルの拡張子
	 *
	 * newName が null の場合は、original のファイル名に、変換します。
	 *
	 * 6.0.2.4 (2014/10/17)
	 * useBackup="rename" で、すでに同名のファイルが存在した場合に、"_001" のような文字列を追加したファイルにリネームします。
	 * Windowsの " - コピー (2)" に近いですが、桁数を抑えるのと、useBackup="true" と異なり、過去の同一ファイル名は
	 * そのまま、有効になります。同一ファイルが同一フォルダに存在する場合のみ連番が付与されます。
	 *
	 * newName の指定に、フォルダ名を含めることを可能にしました。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) 新規追加
	 * @og.rev 5.7.2.1 (2014/01/17) FileUtil.renameTo の引数の順番を間違えていた。
	 * @og.rev 6.0.2.4 (2014/10/17) useBackup 修正、newName に、フォルダ名を含めることが可能
	 * @og.rev 6.2.0.0 (2015/02/27) FileInfoクラスを使用。 (FileUtil#getExtension(String) の廃止)
	 *
	 * @param	newName		ファイル名(nullの場合は、オリジナル)
	 * @param	prefix		接頭辞(nullの場合は、何もつけない)
	 * @param	sufix		接尾辞(nullの場合は、何もつけない)
	 * @param	useBackup	置き換え後ファイルの処理方法(true:backupフォルダ/false:しない/rename:重複しない連番)
	 * @return	最終的に作成されたファイルオブジェクト
	 */
//	public File renameTo( final String newName , final boolean useBackup ) {
	public File renameTo( final String newName , final String prefix , final String sufix , final String useBackup ) {

		// 6.0.2.4 (2014/10/17) prfix が null でなければ、連結。newName が null なら、original を使う。
		String newNm = StringUtil.nvalAdd( prefix, StringUtil.coalesce( newName,original ) );

		if( newNm == null || newNm.isEmpty() ) {
			final String errMsg = "新ファイル名が存在しません。[" + newNm + "]" ;
			throw new RuntimeException( errMsg );
		}

//		// 新ファイル名から拡張子取得
//		final String newExt = FileUtil.getExtension( newNm );
//		if( newExt == null || newExt.isEmpty() ) {							// 拡張子なし
//			final String oldExt = FileUtil.getExtension( original );
//			newNm = StringUtil.nvalAdd( newNm , sufix , "." , oldExt ) ;	// sufix を入れ込む。
//		}
		// 6.2.0.0 (2015/02/27) FileInfo を使用します。
		final FileInfo newInfo = new FileInfo( new File( newNm ) );				// 6.2.0.0 (2015/02/27)
		// 新ファイル名から拡張子取得
		final String newExt = newInfo.SUFIX;
		if( newExt == null || newExt.isEmpty() ) {							// 拡張子なし
			final FileInfo oldInfo = new FileInfo( new File( original ) );		// 6.2.0.0 (2015/02/27)
			newNm = StringUtil.nvalAdd( newNm , sufix , "." , oldInfo.SUFIX ) ;	// sufix を入れ込む。
		}
		else if( sufix != null ) {											// 拡張子あり、sufixあり
			final StringBuilder buf = new StringBuilder( newNm );
			buf.insert( newNm.length()-newExt.length()-1 , sufix );
			newNm = buf.toString() ;
		}

//		String newNm = newName ;
//		// 新規ファイル名を作成します。(拡張子等)
//		if( newNm != null && newNm.length() > 0 ) {
//			// 新ファイル名から拡張子取得
//			String newExt = FileUtil.getExtension( newNm );
//			if( newExt == null || newExt.isEmpty() ) {
//				String oldExt = FileUtil.getExtension( original );
//				newNm = newNm + "." + oldExt ;
//			}
//		}
//		else {
//			newNm = original;
//		}

		File newFile = new File( dir,newNm );
//		File newFile = null ;
//		if( newNm != null && newNm.length() > 0 ) {
//			newFile = new File( dir,newNm );

			final File uniqFile = new File( dir , uniqKey );		// 5.7.1.1 (2013/12/13) アップロードされたファイル

			// 5.7.2.1 (2014/01/17) FileUtil.renameTo の引数の順番を間違えていた。
//			FileUtil.renameTo( uniqFile, newFile, useBackup );		// from ⇒ to の順番に指定。
			// 6.0.2.4 (2014/10/17) 戻り値は、変更後の新しい File オブジェクト
			newFile = FileUtil.renameTo( uniqFile, newFile, useBackup );		// from ⇒ to の順番に指定。
//		}
//		// 5.7.1.1 (2013/12/13) ここの処理が走ることは無いはず。
//		else {
//		final String errMsg = "新ファイル名が存在しません。[" + newNm + "]" ;
//			throw new RuntimeException( errMsg );
//		}
		// 新ファイル名のセットは、すべての処理が完了してから、設定する。
		filename = newFile ;
		return filename;
	}

	/**
	 * ファイル名(オリジナル)を取得します。
	 *
	 * @return	ファイル名(オリジナル)
	 */
	public String getOriginalFileName() {
		return original;
	}

	/**
	 * 自然比較メソッド
	 * インタフェース Comparable の 実装に関連して、再定義しています。
	 * 登録されたシーケンス(画面の表示順)で比較します。
	 * equals メソッドでは、キーの同一性のみに着目して判定しています。
	 * この比較では、(運用上同一キーは発生しませんが)たとえ同一キーが存在した
	 * としても、その比較値が同じになることを保証していません。
	 *
	 * @param   other 比較対象のObject
	 *
	 * @return  このオブジェクトが指定されたオブジェクトより小さい場合は負の整数、等しい場合はゼロ、大きい場合は正の整数
	 * @throws  ClassCastException 引数が UploadedFile ではない場合
	 * @throws  IllegalArgumentException 引数が null の場合
	 */
	@Override
	public int compareTo( final UploadedFile other ) {
		if( other == null ) {
			final String errMsg = "引数が、null です。" ;
			throw new IllegalArgumentException( errMsg );
		}

		return uniqKey.compareTo( other.uniqKey );
	}

	/**
	 * このオブジェクトと他のオブジェクトが等しいかどうかを示します。
	 * 画面は、画面IDが等しければ、言語や表示順に関係なく同一とみなされます。
	 * GUIInfo は、ユーザー個別に扱われ、そのグループには、key は唯一で、かつ
	 * 同一言語内で扱われるオブジェクトの為、同一とみなします。
	 *
	 * @param   object 比較対象の参照オブジェクト
	 *
	 * @return	引数に指定されたオブジェクトとこのオブジェクトが等しい場合は true、そうでない場合は false
	 */
	@Override
	public boolean equals( final Object object ) {
		if( object instanceof UploadedFile ) {
			return uniqKey.equals( ((UploadedFile)object).uniqKey );
		}

		return false ;
	}

	/**
	 * オブジェクトのハッシュコード値を返します。
	 * このメソッドは、java.util.Hashtable によって提供されるような
	 * ハッシュテーブルで使用するために用意されています。
	 * equals( Object ) メソッドをオーバーライトした場合は、hashCode() メソッドも
	 * 必ず 記述する必要があります。
	 * この実装では、getKey().hashCode() と同値を返します。
	 *
	 * @return  このオブジェクトのハッシュコード値
	 */
	@Override
	public int hashCode() {
		return uniqKey.hashCode() ;
	}

	/**
	 * オブジェクトの識別子として，詳細な画面情報を返します。
	 *
	 * @return  詳細な画面情報
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( this.getClass().getName()			).append( CR )
			.append( "  uniqKey  :").append( uniqKey	).append( CR )
			.append( "  filename :").append( filename	).append( CR )
			.append( "  name     :").append( name		).append( CR )
			.append( "  dir      :").append( dir		).append( CR )
			.append( "  original :").append( original	).append( CR )
			.append( "  type     :").append( type		).append( CR );
		return rtn.toString();
	}
}
