/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.mail;

import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Argument ;
import org.opengion.fukurou.util.LogWriter;
import static org.opengion.fukurou.util.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring

import java.util.Map;
import java.util.LinkedHashMap;

import java.util.logging.Logger;
import java.util.logging.Level;
import javax.mail.MessagingException ;

/**
 * MailReceiver は、POP3プロトコルによるメール受信プログラムです。
 *
 * 引数には、host,user,passwd,headerList,mailSaveDir,fileSaveDir,filter,help が指定できます。
 * メールは、メッセージID名をファイル名として、セーブします。
 * 添付ファイルは、元のファイル名で、指定のディレクトリに出力される為、複数のメールを
 * 同時に抜く場合は、uniq 属性を付与してください。その場合は、メッセージID＋連番＋元のファイル名
 * で、ファイルをセーブします。添付ファイルには、同名のファイルを複数添付することが
 * できるため、保存時には、添付ファイルの番号を連番としてファイル名に、付与します。
 *
 * 引数のプロパテイのキー部は、大文字・小文字が厳格に適用されますので、正確に記述願います。
 *
 * Usage: java org.opengion.fukurou.fukurou.mail.MailReceiver
 *        -host=メールサーバー(必須)
 *        -user=メールを取得するログインユーザー(必須)
 *        -passwd=メールを取得するログインパスワード(必須)
 *        -protocol=受信サーバーのプロトコル[imap/pop3]を指定(初期値:pop3)
 *        -port=受信サーバーのポートを指定(初期値:-1)
 *        -mailSaveDir=受信メールをセーブするディレクトリ。指定がない場合は、標準出力へ出力する。
 *        -fileSaveDir=添付ファイルをセーブするディレクトリ。指定がない場合は抜き出さない。
 *        -useMsgId=添付ファイルをセーブするディレクトリに、MesssageIdフォルダを個別に割り当てるかどうか(初期値:false)
 *        -maxRowCount=受信メールの最大取り込み件数(初期値:100)(0:[無制限])
 *        -match_Subject=受信メールのSubjectを選択する条件
 *        -match_Body=受信メールのBodyを選択する条件
 *        -match_From=受信メールのFromを選択する条件
 *        -match_XXXX=受信メールのヘッダー部のキーXXXXを選択する条件
 *        -delete=検索後、メールをサーバーから削除するかどうかを、true/falseで指定(初期値:false)
 *        -help=使用方法を出力して、終了します。
 *
 * @version  0.9.0  2000/11/13
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class MailReceiver {
	private static Logger logger = Logger.getLogger( "org.opengion.fukurou.fukurou.mail.MailReceiver" );		// 4.3.3.5 (2008/11/08)

//	private static final String CR = System.getProperty("line.separator");

	/** 受信メールの最大取り込み件数を指定します。 {@value}  */
	public static final int MAX_ROW_COUNT = 100 ;

	/** 検索後、メールをサーバーから削除するかどうかを、true/falseで指定します。 {@value}  */
	public static final boolean DELETE_MESSAGE = false ;

	/** メールサーバーのデフォルトプロトコル {@value}  */
	public static final String PROTOCOL = "pop3" ;

	/** メールサーバーのデフォルトポート番号 {@value}  */
	public static final int PORT = -1 ;

	private Argument argment ;

	private static final Map<String,String> mustProparty   ;
	private static final Map<String,String> usableProparty ;

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "host",	"メールサーバー(必須)" );
		mustProparty.put( "user",	"メールを取得するログインユーザー(必須)" );
		mustProparty.put( "passwd",	"メールを取得するログインパスワード(必須)" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "protocol",	"受信サーバーのプロトコル(imap,pop3)を指定(初期値:pop3)" );
		usableProparty.put( "port",		"受信サーバーのポートを指定(初期値:-1)" );
		usableProparty.put( "mailSaveDir",	"受信メールをセーブするディレクトリ。" +
									CR + "指定がない場合は、標準出力へ出力する。" );
		usableProparty.put( "fileSaveDir",	"添付ファイルをセーブするディレクトリ。" +
									CR + "指定がない場合は抜き出さない。" );
		usableProparty.put( "useMsgId",	"添付ファイルをセーブするディレクトリに、" +
									CR + "MesssageIdフォルダを個別に割り当てるかどうか。" );
		usableProparty.put( "maxRowCount",	"受信メールの最大取り込み件数(初期値:100)(0:[無制限])" );
		usableProparty.put( "match_Subject", "受信メールのSubjectを選択する条件" );
		usableProparty.put( "match_Body",	"受信メールのBodyを選択する条件" );
		usableProparty.put( "match_From",	"受信メールのFromを選択する条件" );
		usableProparty.put( "match_",		"受信メールのヘッダー部のキーXXXXを選択する条件" );
		usableProparty.put( "delete",		"検索後、メールをサーバーから削除するかどうかを、" +
									CR + "true/falseで指定(初期値:false)" );
		usableProparty.put( "help",			"使用方法を出力して、終了します。" );
	}

	/**
	 * レシーバーを開始します。
	 *
	 * @og.rev 4.3.3.5 (2008/11/08) Argument オブジェクトへの引数を util → mail に訂正します。
	 *
	 * @param   args 引数配列(可変長引数)
	 * @throws MessagingException なんらかのエラーが発生した場合。
	 */
//	public void start( final String[] args ) throws MessagingException {
	public void start( final String... args ) throws MessagingException {

		// パラメータの解析、取得
		logger.fine( "パラメータの解析、取得" );
		argment = new Argument( "org.opengion.fukurou.fukurou.mail.MailReceiver" );		// 4.3.3.5 (2008/11/08)
		argment.setMustProparty( mustProparty );
		argment.setUsableProparty( usableProparty );

		argment.setArgument( args );

		// help パラメータが true に指定された場合の処理。
		if( argment.getProparty( "help",false ) ) {
			System.out.println( argment.toString() );
			return;
		}

		// 処理に必要な各種パラメータを取得しておきます。
		logger.fine( "処理に必要な各種パラメータを取得します。" );
		final MailRX recive = new MailRX();

		recive.setHost(			argment.getProparty( "host" ) ) ;
		recive.setUser(			argment.getProparty( "user" ) ) ;
		recive.setPasswd(		argment.getProparty( "passwd" ) ) ;
		recive.setProtocol(		argment.getProparty( "protocol",PROTOCOL ) ) ;
		recive.setPort(			argment.getProparty( "port",PORT ) ) ;
		recive.setDelete(		argment.getProparty( "delete",DELETE_MESSAGE ) ) ;
		recive.setMaxRowCount(	argment.getProparty( "maxRowCount",MAX_ROW_COUNT ) ) ;

		// 指定の条件にマッチしたメッセージのみ抜き出す為の、SearchTerm オブジェクトの作成
		logger.fine( "指定の条件にマッチしたメッセージのみ抜き出す条件を設定します。" );
		final HybsEntry[] matchs = argment.getEntrys( "match_" );
		for( int i=0; i<matchs.length; i++ ) {
			recive.addMatchTerm( matchs[i] ) ;
		}

		// リスナーを設定して、受信メールを一件ずつ処理します。
		logger.fine( "リスナーを設定して、受信メールを一件ずつ処理します。" );
		final String mailSaveDir = argment.getProparty( "mailSaveDir" );
		final String fileSaveDir = argment.getProparty( "fileSaveDir" );
		final boolean useMsgId   = argment.getProparty( "useMsgId",false );

		final MailReceiveListener listener = new ReceiveListener( mailSaveDir,fileSaveDir,useMsgId ) ;
		recive.setMailReceiveListener( listener );

		recive.start();
	}

	/**
	 * メール受信プログラムで使用する MailReceiveListener の実装内部クラスです。
	 *
	 * @version  0.9.0  2000/11/13
	 * @author   Kazuhiko Hasegawa
	 * @since    JDK5.0,
	 */
	private static class ReceiveListener implements MailReceiveListener {
		private final String  mailSaveDir ;
		private final String  fileSaveDir ;
		private final boolean useMsgId ;
		private int   counter ;

		/**
		 * コンストラクター
		 *
		 * @param	mailSaveDir	メールをセーブする場合の保存フォルダ名
		 * @param	fileSaveDir	メールの添付ファイルをセーブする場合の保存フォルダ名
		 * @param	useMsgId	添付ファイルをセーブする場合に、メッセージIDを使用するかどうか
		 */
		public ReceiveListener( final String mailSaveDir,final String fileSaveDir,final boolean useMsgId ) {
			this.mailSaveDir = mailSaveDir;
			this.fileSaveDir = fileSaveDir;
			this.useMsgId    = useMsgId;
		}

		/**
		 * 受信処理を行います。
		 *
		 * @param   message MailMessage
		 * @return	結果(true:正常/false:異常)
		 */
		public boolean receive( final MailMessage message ) {
			final String msg = "[" + counter++ + "]" + message.getMessageID() + " 受信中" ;
			System.out.println( msg );

			if( mailSaveDir != null ) {
				message.saveMessage( mailSaveDir );
			}
			else {
				System.out.println( message.getSubject() );
				System.out.println( message.getContent() );
			}

			if( fileSaveDir != null ) {
				message.saveAttachFiles( fileSaveDir,useMsgId );
			}
			return true ;
		}
	}

	/**
	 * main メソッドです。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main ( final String[] args ) {
		final MailReceiver receiver = new MailReceiver();
		try {
			logger.info( "メール受信処理を開始します  ---------------------------------------------" );
			receiver.start( args );
			logger.info( "正常に終了しました。" );
		}
		catch( Throwable th ) {
			final String errMsg = "メール受信中に例外が発生しました。 "
							+ CR + receiver.argment
							+ CR + th.getMessage() ;
			LogWriter.log( errMsg );
			logger.log( Level.SEVERE,errMsg, th );
			throw new RuntimeException( errMsg,th );	// 4.0.0 (2005/01/31)
		}
	}
}
