/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.util.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

// import org.opengion.hayabusa.common.HybsSystem;

/**
 * データのコード情報を取り扱うクラスです。
 *
 * 文字列の 「キー:ラベル キー:ラベル」の情報から、HTMLのメニューやリストを作成するための 
 * オプションタグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを
 * 作成したりします。
 * ラベル にスペースを含ませる場合は、ダブルクォーテーションで囲ってください。
 *
 * @og.rev 5.6.6.0 (2013/07/05) 新規追加
 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 基本実装とします。
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Selection_KEYVAL implements Selection {
public class Selection_KEYVAL extends Selection_NULL {
	private final String	ORG_KEYVAL ;

	private final String	CACHE ;

	/**
	 * コンストラクター
	 *
	 * @og.rev 5.6.7.1 (2013/08/09) 「キー:ラベル キー:ラベル」分解に、クオート処理を加味
	 *
	 * @param	strCode	コードデータパラメータ文字列
	 */
	public Selection_KEYVAL( final String strCode ) {
		ORG_KEYVAL = strCode ;

		if( strCode != null && strCode.indexOf( ':' ) > 0 ) {
			final String[] keyvals = StringUtil.csv2Array( strCode, ' ' );	// 5.6.7.1 (2013/08/09) クオート処理を加味
			final int size = keyvals.length;

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			for( int i=0; i<size; i++ ) {
				final String keyval = keyvals[i] ;
				if( keyval == null || keyval.length() <= 0 ) { continue; }
				final int idx = keyval.indexOf( ':' );
				if( idx < 0 ) { continue; }						// スペースで分解した結果に、コロンを含まない場合

				final String key = keyval.substring( 0,idx ).trim();
				String val = keyval.substring( idx+1 ).trim();

				// 5.6.7.1 (2013/08/09) クオート処理を加味。csv2Array では、クオートは残ったままである。
				if( val.length() >= 2 && val.charAt(0) == '"' && val.charAt(val.length()-1) == '"' ) {
					// 前後の クオート を取り除く。さらに、その結果を trim する。
					val = val.substring( 1,val.length()-1 ).trim();
				}

				// 6.0.2.5 (2014/10/31) char を append する。
				buf.append( "<option value=\"" ).append( key )
					.append( "\">" ).append( val ).append( "</option>" );
			}

			CACHE = buf.toString();
		}
		else {
			CACHE = "";
		}
	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * このメソッドでは、ラベル(短)が設定されている場合でも、これを使用せずに必ずラベル(長)を使用します。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue  選択されている値
//	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
//	 *
//	 * @return  オプションタグ
//	 * @see     #getOption( String, boolean, boolean )
//	 */
//	public String getOption( final String selectValue,final boolean seqFlag ) {
//		return getOption( selectValue, seqFlag, false );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このクラスでは、useShortLabel は、無視されます。(常に、false です)
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか(常にfalse)。
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。キーの前後のダブルクオートを加味して検索
		final String selVal = "\"" + selectValue + "\"" ;

		final int indx = CACHE.indexOf( selVal );

		if( indx < 0 ) {
			// 4.0.0 (2005/01/31)
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "コードに存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR + ORG_KEYVAL ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			final int addIndx = indx + selVal.length() ;	// selected の挿入位置

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( CACHE.substring( 0,addIndx ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( CACHE.substring( addIndx ) );
			return buf.toString() ;
		}
	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * ※ このクラスでは実装されていません。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   name         ラジオの name
//	 * @param   selectValue  選択されている値
//	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
//	 *
//	 * @return  オプションタグ
//	 */
//	public String getRadio( final String name,final String selectValue,final boolean useLabel ) {
//	final String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * ※ このクラスでは実装されていません。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue  選択されている値
//	 *
//	 * @return  オプションタグ
//	 */
//	public String getRadioLabel( final String selectValue ) {
//	final String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

//	/**
//	 * 選択肢(value)に対するラベルを返します。
//	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
//	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue 選択肢の値
//	 *
//	 * @return  選択肢のラベル
//	 * @see     #getValueLabel( String,boolean )
//	 */
//	public String getValueLabel( final String selectValue ) {
//		return getValueLabel( selectValue,false );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * ※ このクラスでは、短縮ラベルは使用されません。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを [true:使用する/false:しない](常に false)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// マッチするアドレスを探す。キーの前後のダブルクオートを加味して検索
		final String selVal = "\"" + selectValue + "\"" ;

		final int indx = CACHE.indexOf( selVal );

		if( indx < 0 ) {
			// マッチしなければ、選択肢そのものを返す。
			return selectValue;
		}
		else {
			// マッチすれば、キー以下のBODY部の文字列を切り出して返す。
			final int stIdx = indx + selVal.length() + 1 ;			// ＋１ は、">" の位置
			final int edIdx = CACHE.indexOf( '<',stIdx );				// 終了アドレス

			return CACHE.substring( stIdx,edIdx );
		}
	}

//	/**
//	 * マルチ・キーセレクトを使用するかどうかを返します。
//	 * true：使用する。false:使用しない です。
//	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
//	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
//	 * true を返します。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * ※ ここでは、常に false を返します。
//	 *
//	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)
//	 */
//	public boolean useMultiSelect() {
//		return false;
//	}

//	/**
//	 * オブジェクトのキャッシュが時間切れかどうかを返します。
//	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
//	 * false を返します。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * ※ ここでは、常に false を返します。
//	 *
//	 * @return  キャッシュが時間切れなら true
//	 */
//	public boolean isTimeOver() {
//		return false;
//	}
}
