/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import java.io.File;												// 5.7.3.2 (2014/02/28) Tomcat8 対応
import java.io.BufferedReader;
// import java.io.FileInputStream;
import java.io.IOException;
// import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import org.opengion.fukurou.security.HybsCryptography;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;							// 6.2.0.0 (2015/02/27)
import org.opengion.hayabusa.common.HybsSystem;
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * URLCheckFilter は、Filter インターフェースを継承した URLチェッククラスです。
 * web.xml で filter 設定することにより、該当のリソースに対して、og:linkタグで、
 * useURLCheck="true"が指定されたリンクURL以外を拒否することができます。
 * また、og:linkタグを経由した場合でも、リンクの有効期限を設定することで、
 * リンクURLの漏洩に対しても、一定時間の経過を持って、アクセスを拒否することができます。
 * また、リンク時にユーザー情報も埋め込んでいますので(初期値は、ログインユーザー)、
 * リンクアドレスが他のユーザーに知られた場合でも、アクセスを拒否することができます。
 *
 * フィルターに対してweb.xml でパラメータを設定します。
 *   ・filename :停止時メッセージ表示ファイル名
 *   ・ignoreURL:暗号化されたURLのうち空白に置き換える接頭文字列を指定します。
 *              外部からアクセスしたURLがロードバランサで内部向けURLに変換されてチェックが動作しないような場合に
 *              利用します。https://wwwX.のように指定します。通常は設定しません。
 *
 * 【WEB-INF/web.xml】
 *     &lt;filter&gt;
 *         &lt;filter-name&gt;URLCheckFilter&lt;/filter-name&gt;
 *         &lt;filter-class&gt;org.opengion.hayabusa.filter.URLCheckFilter&lt;/filter-class&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;filename&lt;/param-name&gt;
 *             &lt;param-value&gt;jsp/custom/refuseAccess.html&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *     &lt;/filter&gt;
 *
 *     &lt;filter-mapping&gt;
 *         &lt;filter-name&gt;URLCheckFilter&lt;/filter-name&gt;
 *         &lt;url-pattern&gt;/jsp/*&lt;/url-pattern&gt;
 *     &lt;/filter-mapping&gt;
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public final class URLCheckFilter implements Filter {

	private static final HybsCryptography HYBS_CRYPTOGRAPHY = new HybsCryptography(); // 4.3.7.0 (2009/06/01)

	private String	filename	;			// アクセス拒否時メッセージ表示ファイル名
	private boolean	isDebug		;
	private boolean	isDecode	= true;		// 5.4.5.0(2012/02/28) URIDecodeするかどうか

	private String	ignoreURL	;			// 5.8.6.1 (2015/04/17) 飛んできたcheckURLから取り除くURL文字列

	/**
	 * フィルター処理本体のメソッドです。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) new BufferedReader … を、FileUtil.getBufferedReader … に変更。
	 *
	 * @param	request		ServletRequestオブジェクト
	 * @param	response	ServletResponseオブジェクト
	 * @param	chain		FilterChainオブジェクト
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 */
	public void doFilter( final ServletRequest request, final ServletResponse response, final FilterChain chain ) throws IOException, ServletException {

		if( !isValidAccess( request ) ) {
			BufferedReader in = null ;
			try {
				response.setContentType( "text/html; charset=UTF-8" );
				final PrintWriter out = response.getWriter();
//				in = new BufferedReader( new InputStreamReader( new FileInputStream( filename ) ,"UTF-8" ) );
				in = FileUtil.getBufferedReader( new File( filename ), "UTF-8" );		// 6.2.0.0 (2015/02/27)
				String str ;
				while( (str = in.readLine()) != null ) {
					out.println( str );
				}
				out.flush();
			}
			catch( UnsupportedEncodingException ex ) {
				final String errMsg = "指定されたエンコーディングがサポートされていません。[UTF-8]" ;
				throw new RuntimeException( errMsg,ex );
			}
			catch( IOException ex ) {
				final String errMsg = "ストリームがオープン出来ませんでした。[" + filename + "]" ;
				throw new RuntimeException( errMsg,ex );
			}
			finally {
				Closer.ioClose( in );
			}
			return;
		}

		chain.doFilter(request, response);
	}

	/**
	 * フィルターの初期処理メソッドです。
	 *
	 * フィルターに対してweb.xml で初期パラメータを設定します。
	 *   ・maxInterval:リンクの有効期限
	 *   ・filename   :停止時メッセージ表示ファイル名
	 *   ・decode     :URLデコードを行ってチェックするか(初期true)
	 *
	 * @og.rev 5.4.5.0 (2102/02/28)
	 * @og.rev 5.7.3.2 (2014/02/28) Tomcat8 対応。getRealPath( "/" ) の互換性のための修正。
	 * @og.rev 5.8.6.1 (2015/04/17) DMZのURL変換対応
	 *
	 * @param filterConfig FilterConfigオブジェクト
	 */
	public void init(final FilterConfig filterConfig) {
		final ServletContext context = filterConfig.getServletContext();
		final String realPath = context.getRealPath( "" ) + File.separator;		// 5.7.3.2 (2014/02/28) Tomcat8 対応

		filename  = realPath + filterConfig.getInitParameter("filename");
		isDebug   = StringUtil.nval( filterConfig.getInitParameter("debug"), false );
		isDecode  = StringUtil.nval( filterConfig.getInitParameter("decode"), true );	// 5.4.5.0(2012/02/28)
		ignoreURL = filterConfig.getInitParameter("ignoreURL");							// 5.8.6.1 (2015/04/17)
	}

	/**
	 * フィルターの終了処理メソッドです。
	 *
	 */
	public void destroy() {
		// ここでは処理を行いません。
	}

	/**
	 * フィルターの内部状態をチェックするメソッドです。
	 *
	 * @og.rev 5.4.5.0 (2012/02/28) Decode
	 * @og.rev 5.8.6.1 (2015/04/17) DMZのURL変換対応
	 *
	 * @param request ServletRequestオブジェクト
	 *
	 * @return	(true:許可  false:拒否)
	 */
	private boolean isValidAccess( final ServletRequest request ) {
		String checkKey = request.getParameter( HybsSystem.URL_CHECK_KEY );
		if( checkKey == null || checkKey.isEmpty() ) {
			if( isDebug ) {
				System.out.println( "  check NG [ No Check Key ]" );
			}
			return false;
		}

		boolean rtn = false;
		try {
			checkKey = HYBS_CRYPTOGRAPHY.decrypt( checkKey ).replace( "&amp;", "&" );

			if( isDebug ) {
				System.out.println( "checkKey=" + checkKey );
			}

			// 5.8.6.1 (2015/04/17) DMZのURL変換対応 (ちょっと整理しておきます)
//			String url = checkKey.substring( 0 , checkKey.lastIndexOf( ",time=") );
//			final long time = Long.parseLong( checkKey.substring( checkKey.lastIndexOf( ",time=") + 6, checkKey.lastIndexOf( ",userid=" ) ) );
//			final String userid = checkKey.substring( checkKey.lastIndexOf( ",userid=") + 8 );

			final int tmAd = checkKey.lastIndexOf( ",time=" );
			final int usAd = checkKey.lastIndexOf( ",userid=" );

			String url = checkKey.substring( 0 , tmAd );
			final long  time    = Long.parseLong( checkKey.substring( tmAd + 6, usAd ) );
			final String userid = checkKey.substring( usAd + 8 );

			// 4.3.8.0 (2009/08/01)
			final String[] userArr = StringUtil.csv2Array( userid );

			// 5.8.6.1 (2015/04/17)ignoreURL対応
			if( ignoreURL != null && ignoreURL.length()>0 && url.indexOf( ignoreURL ) == 0 ){
				url = url.substring( ignoreURL.length() );
			}

			if( isDebug ) {
				System.out.println( " [ignoreURL]=" + ignoreURL );	// 2015/04/17 (2015/04/17)
				System.out.println( " [url]    =" + url );
				System.out.println( " [vtime]  =" + time );
				System.out.println( " [userid] =" + userid );
			}

			String reqStr =  ((HttpServletRequest)request).getRequestURL().toString() + "?" + ((HttpServletRequest)request).getQueryString();
			// 5.4.5.0 (2012/02/28) URLDecodeを行う
			if(isDecode){
				if( isDebug ) {
					System.out.println( "[BeforeURIDecode]="+reqStr );
				}
				reqStr = StringUtil.urlDecode( reqStr );
			}
			reqStr = reqStr.substring( 0, reqStr.lastIndexOf( HybsSystem.URL_CHECK_KEY ) -1 );
			//	String reqStr =  ((HttpServletRequest)request).getRequestURL().toString();
			final String reqUser = ((HttpServletRequest)request).getRemoteUser();

			if( isDebug ) {
				System.out.println( " [reqURL] =" + reqStr );
				System.out.println( " [ctime]  =" + System.currentTimeMillis() );
				System.out.println( " [reqUser]=" + reqUser );
			}

			if( reqStr.endsWith( url )
					&& System.currentTimeMillis() - time < 0
					&& userArr != null && userArr.length > 0 ) {
				// 4.3.8.0 (2009/08/01)
				for( int i=0; i<userArr.length; i++ ) {
					if( "*".equals( userArr[i] ) || reqUser.equals( userArr[i] ) ) {
						rtn = true;
						if( isDebug ) {
							System.out.println( "  check OK" );
						}
						break;
					}
				}
			}
		}
		catch( RuntimeException ex ) {
			if( isDebug ) {
				final String errMsg = "チェックエラー。 "
							+ " checkKey=" + checkKey
							+ " " + ex.getMessage();			// 5.1.8.0 (2010/07/01) errMsg 修正
				System.out.println( errMsg );
				ex.printStackTrace();
			}
			rtn = false;
		}
		return rtn;
	}

	/**
	 * 内部状態を文字列で返します。
	 *
	 * @return	このクラスの文字列表示
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "UrlCheckFilter" )
			.append( "filename=[" ).append( filename  ).append( "],")
			.append( "isDecode=[" ).append( isDecode  ).append( ']');		// 6.0.2.5 (2014/10/31) char を append する。
		return buf.toString();
	}
}
